/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   history.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: This is a small utility class to keep a track of what
 *              happened to each player since its last game session started.
 *              This is usefull in a network game for the server to be able
 *              to send to the new clients a complete history of what
 *              happened to each player.
 */

/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include <zlib.h>

#include "history.h"
#include "log.h"
#include "global.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

/*
 * Delay in ticks between 2 map serialization.
 */
#define U61_HISTORY_DELAY_BETWEEN_MAP_SERIALIZATION 300

/*
 * If there's less than this number of events in the queue, then
 * we do not "serialize".
 */
#define U61_HISTORY_MAP_SERIALIZATION_MIN_QUEUE_SIZE 20

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a history item
 */ 
U61_History::U61_History()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * resets all the history values
 */ 
void U61_History::reset()
{
  clear();
  state=false;
  ready_all=false;
  put_ready_all=false;
  last_map_serialization_time=0;
  player_id=0;
}

/*--------------------------------------------------------------------------*/
/* 
 * empties the history
 */ 
void U61_History::clear()
{
  events_playing.clear();
  events_init.clear();
  /*
   * The next line will automatically add a "READY_ALL" event
   * at the beginning of history if needed.
   */
  put_ready_all=ready_all;
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the history queue
 */
void U61_History::put(U61_Event evt)
{
  U61_Event evt_ready;

  switch(evt.code)
    {
    case U61_EVENT_LOOSE:
      loose(&evt);
      break;
    case U61_EVENT_START_STOP:
      start_stop(evt);
      break;
    case U61_EVENT_KILL:
      kill();
      break;
    case U61_EVENT_READY_ALL:
      ready_all=true;
      break;
    }

  if (put_ready_all)
    {
      put_ready_all=false;

      evt_ready=evt;
      evt_ready.code=U61_EVENT_READY;
      put(evt_ready);

      evt_ready=evt;
      evt_ready.code=U61_EVENT_READY_ALL;
      put(evt_ready);
    }

  switch(evt.code)
    {
    case U61_EVENT_CHAT_LETTER:
      /*
       * We volontarily skip chat events, for it's not very
       * usefull. No one really cares nor needs to know what's
       * been said _before_ he gets connected
       */
      break;
    default:
      events_playing.push_back(evt);
      break;
    }

  switch(evt.code)
    {
    case U61_EVENT_NAME_LETTER:
    case U61_EVENT_READY:
    case U61_EVENT_READY_ALL:
      /*
       * We push name letter events here, this
       * way we'll have a quick way to tell the
       * players' name when someone connects
       */
      events_init.push_back(evt);
      break;
    }

  if (evt.time>last_map_serialization_time+U61_HISTORY_DELAY_BETWEEN_MAP_SERIALIZATION)
    {
      last_map_serialization_time=evt.time;
      serialize_map();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the history size
 */
int U61_History::size()
{
  int size;

  if (playing())
    {
      size=events_playing.size();
    }
  else
    {
      size=events_init.size();
    }

  return size;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns a given event in the history
 * 0 should be the oldest event (ie the first one) and size-1 the last one
 */
U61_Event U61_History::get(int i)
{
  U61_Event event;

  if (playing())
    {
      event=events_playing[i];
    }
  else
    {
      event=events_init[i];
    }

  U61_LOG_DEBUG("reading history, event "<<i<<" = "<<event);

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * To be called each time a loose event is detected
 */
void U61_History::loose(U61_Event *evt)
{
  clear();
  state=true;
  evt->code=U61_EVENT_START_STOP;
}

/*--------------------------------------------------------------------------*/
/*
 * To be called each time a start_stop event is detected
 */
void U61_History::start_stop(U61_Event evt)
{
  clear();
  if (state)
    {
      /*
       * Game ends
       */
      state=false;
    }
  else
    {
      /*
       * Game begins
       */
      state=true;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * To be called each time a kill event is detected
 */
void U61_History::kill()
{
  clear();
  state=false;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the player associated to this id is/should be playing
 */
bool U61_History::playing()
{
  return state;
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the player ID for this history object.
 */
void U61_History::set_player_id(int pl_id)
{
  player_id=pl_id;
}

/*--------------------------------------------------------------------------*/
/*
 * Serializes the map and prepares events to be sent so that new client
 * players can reconstruct the original map without interpreting the
 * whole history, for this process can be quite long when people have been
 * playing for hours...
 */
void U61_History::serialize_map()
{
  int i;
  U61_Player *pl;
  int size;
  U61_Event evt;
  U61_Map *map=NULL;
  unsigned int map_time=0;
  bool map_serialized=false;
  std::vector<U61_Event> events_tmp;
  unsigned char *buf1;
  unsigned char *buf2;
  unsigned char *buf1_tmp;
  unsigned char *buf2_tmp;
  unsigned long buf2_size;
  short par;
  int zerr;

  U61_LOG_DEBUG("serialize_map");

  size=events_playing.size();
  
  /*
   * We do this serialization stuff only if the player's playing
   * with a non-null history
   */
  if (playing() &&
      size>0 &&
      size>=U61_HISTORY_MAP_SERIALIZATION_MIN_QUEUE_SIZE)
    {
      pl=U61_Global::game.find_player_by_id(player_id);
      if (pl!=NULL)
	{
	  map=pl->get_history();
	  map_time=pl->get_history_time();

	  events_tmp=events_playing;
	  events_playing.clear();

	  for (i=0; i<size;++i)
	    {
	      evt=events_tmp[i];
	      if (map_serialized)
		{
		  /*
		   * the map has already been serialized, so this is
		   * a recent event, we keep it as is.
		   */
		  events_playing.push_back(evt);
		}
	      else
		{
		  if (evt.time<=map_time)
		    {
		      switch (evt.code)
			{
			case U61_EVENT_READY:
			case U61_EVENT_READY_ALL:
			case U61_EVENT_START_STOP:
			case U61_EVENT_KILL:
			case U61_EVENT_NAME_LETTER:
			  /*
			   * These events still go through the filter,
			   * however we change their date to for some
			   * performance issues: we don't want the
			   * system to waste time on "U61_Map::compute_up_to"
			   */
			  evt.time=map_time;
			  events_playing.push_back(evt);
			}
		    } 
		  else
		    {
		      /*
		       * Now the event is older so there are no more
		       * "previous outdated" events, we can serialize
		       * the map.
		       */
		      buf1=new unsigned char[U61_MAP_SERIALIZED_SIZE];
		      buf2_size=U61_MAP_COMPRESSED_SIZE;
		      buf2=new unsigned char[buf2_size];
		      if (buf1 && buf2)
			{
			  buf1_tmp=buf1;
			  if (map->serialize(&buf1_tmp,buf1_tmp+U61_MAP_SERIALIZED_SIZE))
			    {
			      zerr=compress(buf2,&buf2_size,buf1,U61_MAP_SERIALIZED_SIZE);
			      if (zerr==Z_OK)
				{
				  if (buf2_size<65536)
				    {
				      U61_LOG_DEBUG("Map history serialize buffer sizes: "<<((int) buf2_size)<<" / "<<U61_MAP_SERIALIZED_SIZE);
				      /*
				       * Now we write the events in the queue.
				       */

				      evt.author=evt.target=player_id;
				      evt.time=map_time;

				      /*
				       * First we send a "begin" message, which also indicates
				       * the size of data to be sent.
				       */
				      evt.code=U61_EVENT_SERIAL_BEGIN;
				      evt.par=buf2_size;

				      events_playing.push_back(evt);

				      /*
				       * Now we send the actual data.
				       */
				      evt.code=U61_EVENT_SERIAL_DATA;
				      for (i=0;i<(int) buf2_size;i+=sizeof(short))
					{
					  buf2_tmp=buf2+i;
					  /*
					   * We use the unserial function here since the conversion
					   * between the zlib buffer and the short in the event
					   * struct needs to be endianess proof
					   */
					  U61_Serial::unserialize_short(&par,&buf2_tmp,buf2_tmp+sizeof(short));
					  evt.par=par;

					  events_playing.push_back(evt);				      
					}
				  
				      /*
				       * And now we send an "end" message.
				       */
				      evt.code=U61_EVENT_SERIAL_END;
				      evt.par=0;
				      events_playing.push_back(evt);				      
				    }
				  else
				    {
				      U61_LOG_ERROR("Serialized map is too big!");
				    }
				}
			      else
				{
				  U61_LOG_ERROR("Unable to compress serialized map (zlib error "<<zerr<<":\""<<zError(zerr)<<"\")!");
				}
			    }
			  else
			    {
			      U61_LOG_ERROR("Unable to serialize map!");
			    }
			}
		      else
			{
			  U61_LOG_ERROR("Not enough memory for map bufferization!");
			}

		      if (buf1)
			{
			  delete buf1;
			}
		      if (buf2)
			{
			  delete buf2;
			}


		      map_serialized=true;
		    }
		}
	    }
	}
    }
}
