/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   http.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: utilities to handle basic http requests. Used for
 *              interaction with the meta-server.
 */



/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>

#include "http.h"
#include "macro.h"
#include "const.h"
#include "socket.h"
#include "log.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_HTTP_GET_STRING \
    "GET http://%s HTTP/1.1\x0d\x0a" \
    "Host: %s\x0d\x0a" \
    "Connection: close\x0d\x0a"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a http request handler
 */ 
U61_Http::U61_Http(char *h, int p, char *r)
{
  reset();
  set_host(h);
  set_port(p);
  set_request(r);
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a http request handler
 */ 
U61_Http::~U61_Http()
{

}

/*--------------------------------------------------------------------------*/
/* 
 * resets a http request handler
 */ 
void U61_Http::reset()
{
  U61_MACRO_MEMSET0(request);
  U61_MACRO_MEMSET0(answer);

  set_host("");
  set_port(0);
  set_request("");
}

/*--------------------------------------------------------------------------*/
/*
 * sets the name of the server
 */
void U61_Http::set_host(char *h)
{
  if (h!=NULL)
    {
      U61_MACRO_STRCPY(host,h);
    }
  else
    {
      host[0]='\0';
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the port of the server
 */
void U61_Http::set_port(int p)
{
  if (p>=1 && p<65536)
    {
      port=p;
    }
  else
    {
      port=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the HTTP request to execute
 */
void U61_Http::set_request(char *r)
{
  if (r!=NULL)
    {
      U61_MACRO_SPRINTF2(request,U61_HTTP_GET_STRING,r,host);
    }
  else
    {
      request[0]='\0';
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * executes an http request
 */ 
bool U61_Http::execute()
{
  bool result=false;
  U61_Socket socket;

  if (socket.connect(host,port))
    {
      if (socket.send_str(request,sizeof(request)-1))
	{
	  if (socket.recv_str(answer,sizeof(answer)-1))
	    {
	      result = true;
	    }
	  else
	    {
	      U61_LOG_WARNING("Unable to receive HTTP answer from "<<host<<":"<<port);
	    }
	}
      else
	{
	  U61_LOG_WARNING("Unable to send HTTP request to "<<host<<":"<<port);
	}
    }
  else
    {
      U61_LOG_WARNING("Unable to connect on "<<host<<":"<<port);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the HTTP answer (the page returned by the server)
 */
char *U61_Http::get_answer()
{
  return answer;
}

/*--------------------------------------------------------------------------*/
/*
 * Replaces special chars by their hexa description, to be used as URL
 * parameters. For instance " " will become "%20"
 */
void U61_Http::format_url_param(char dst[U61_CONST_STRING_SIZE], char *src)
{
  int j,i;
  unsigned char c;
  char buffer[4];

  j=0;
  for (i=0;src[i] && j<U61_CONST_STRING_SIZE-1;++i)
    {
      c=src[i];

      if (isalnum(c))
	{
	  dst[j]=c;
	  j++;
	}
      else
	{
	  if (j<U61_CONST_STRING_SIZE-3)
	    {
	      U61_MACRO_SPRINTF1(buffer,"%X",c);
	      dst[j]='%';
	      dst[j+1]=buffer[0];
	      dst[j+2]=buffer[1];
	    }
	  j+=3;
	}
    }
  dst[j]='\0';
}

/*--------------------------------------------------------------------------*/
/*
 * Skip spaces, tabs and the likes.
 */
void U61_Http::parser_skip_space(char **str)
{
  while ((isspace(**str) || (**str)=='\x0d') && 
	 (**str)!='\0' && (**str)!='\x0a')
    {
      ++(*str);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Reads a string between ''. Double '' are interpreted as a single '
 * which is the standard SQL convention
 */
bool U61_Http::parser_read_string(char *result,char **str,int size)
{
  int i=0;
  bool done=false;
  char c;

  parser_skip_space(str);

  c=**str;
  if (c=='\'')
    {
      ++(*str);
      while ((c=**str)!='\0' && c!='\n' && !done)
	{
	  ++(*str);
	  if (c=='\'')
	    {
	      if ((**str)=='\'')
		{
		  ++(*str);
		  if (i<size-1)
		    {
		      result[i++]='\'';
		    }
		}
	      else
		{
		  done=true;
		}
	    }
	  else
	    {
	      if (i<size-1)
		{
		  result[i++]=c;
		}
	    }
	}
    }

  result[i++]='\0';

  parser_skip_space(str);

  return done;
}

/*--------------------------------------------------------------------------*/
/*
 * Reads an integer
 */
bool U61_Http::parser_read_int(int *result,char **str)
{
  bool ok=false;
  char c;

  (*result)=0;

  parser_skip_space(str);

  while (isdigit(c=(**str)))
    {
      ++(*str);

      (*result)*=10;
      (*result)+=c-'0';

      ok=true;
    }

  parser_skip_space(str);

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Reads a field separator
 */
bool U61_Http::parser_read_separator(char **str)
{
  bool ok=false;
  char c;

  parser_skip_space(str);

  c=**str;

  if (c==',')
    {
      ++(*str);
      ok=true;
    }

  parser_skip_space(str);

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Reads a new-line character
 */
bool U61_Http::parser_read_eol(char **str)
{
  bool ok=false;
  char c;

  parser_skip_space(str);

  c=**str;

  if (c=='\x0a')
    {
      ++(*str);
      ok=true;
    }

  parser_skip_space(str);

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Reads an end of line
 */
void U61_Http::parser_next_line(char **str)
{
  char *pos;
  int len;

  parser_skip_space(str);

  pos=strchr(*str,'\x0a');
  if (pos)
    {
      (*str)=pos+1;
    }
  else
    {
      /*
       * No '\n' found, we decided to go to the
       * very end of the string
       */
      len=strlen(*str);
      (*str)+=len;
    }

  parser_skip_space(str);
}
