/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   inputmonitor.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: This class is responsible for listening to keyboard events.
 *              It gets key-down and key-up signals, and updates the
 *              array of pressed-down keys. Then repetition is handled
 *              separately by other classes. 
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "inputmonitor.h"
#include "log.h"
#include "macro.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_INPUTMONITOR_FIRST_ASCII 32 
#define U61_INPUTMONITOR_LAST_ASCII 127

/*---------------------------------------------------------------------------
 methods
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * Creates a default input monitor
 */ 
U61_InputMonitor::U61_InputMonitor()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * Binds the keyboard to the input monitor
 * Could not be done in the constructor since this object is global/static
 */ 
void U61_InputMonitor::init()
{
  device = CL_Input::keyboards[0];

  slot_button_press = CL_Input::sig_button_press().connect
		 (this, &U61_InputMonitor::on_button_press);
  slot_button_release = CL_Input::sig_button_release().connect
		 (this, &U61_InputMonitor::on_button_release);
}

/*--------------------------------------------------------------------------*/
/* 
 * Destroys the input monitor
 */ 
U61_InputMonitor::~U61_InputMonitor()
{
}

/*--------------------------------------------------------------------------*/
/*
 * reads information from the config
 */
void U61_InputMonitor::read_config(U61_Config *conf)
{
  int i;

  for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_ASCII;++i)
    {
      ascii_labels[2*i]=U61_INPUTMONITOR_FIRST_ASCII+i;
      ascii_labels[2*i+1]=0;
      ascii[i].define(U61_INPUTMONITOR_FIRST_ASCII+i,
		      ascii_labels+2*i,
		      CL_KEY_NONE_OF_THE_ABOVE);
    }
  ascii[0].define(' ',"SPACE",CL_KEY_NONE_OF_THE_ABOVE);

  for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_SCANCODE;++i)
    {
      U61_LOG_DEBUG("reading from config, key scancode="<<i<<", ascii="<<((int) conf->key[i].ascii)<<", label=\""<<conf->key[i].label<<"\"");

      scancode[i].define(conf->key[i].ascii,conf->key[i].label,i);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * reads information from the config
 */
void U61_InputMonitor::write_config(U61_Config *conf)
{
  int i;

  for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_SCANCODE;++i)
    {    
      conf->key[i].ascii=scancode[i].get_ascii();
      U61_MACRO_STRCPY(conf->key[i].label,scancode[i].get_label());

      U61_LOG_DEBUG("writing to config, key scancode="<<i<<", ascii="<<((int) conf->key[i].ascii)<<", label=\""<<conf->key[i].label<<"\"");
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Resets all the keys.
 */ 
void U61_InputMonitor::reset()
{
  int i;

  for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_ASCII;++i)
    {
      ascii[i].reset();
    }
  for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_SCANCODE;++i)
    {
      scancode[i].reset();
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the state of a given key (pressed or not)
 */ 
bool U61_InputMonitor::is_pressed(int key)
{
  bool result=false;

  if (key<U61_INPUTMONITOR_NUM_KEYS_SCANCODE)
    {
      result=scancode[key].get_state();
    }
  else
    {
      result=ascii[key-U61_INPUTMONITOR_NUM_KEYS_SCANCODE].get_state();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Converts an internal keycode to a printable ascii code
 */ 
char U61_InputMonitor::to_ascii(int key)
{
  char result=U61_KEYDEF_ASCII_UNKNOWN;

  if (key>=U61_INPUTMONITOR_NUM_KEYS_SCANCODE)
    {
      result=ascii[key-U61_INPUTMONITOR_NUM_KEYS_SCANCODE].get_ascii();
    }
  else
    {
      result=scancode[key].get_ascii();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Converts an internal scancode to a ClanLib scancode
 */ 
int U61_InputMonitor::to_scancode(int key)
{
  int result=CL_KEY_NONE_OF_THE_ABOVE;

  if (key<U61_INPUTMONITOR_NUM_KEYS_SCANCODE)
    {
      return key;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Converts an internal keycode to a readable label
 */ 
char *U61_InputMonitor::to_label(int key)
{
  char *result=U61_KEYDEF_LABEL_UNKNOWN;

  if (key>=U61_INPUTMONITOR_NUM_KEYS_SCANCODE)
    {
      result=ascii[key-U61_INPUTMONITOR_NUM_KEYS_SCANCODE].get_label();
    }
  else
    {
      result=scancode[key].get_label();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a key is pressed down
 */ 
void U61_InputMonitor::on_button_press(CL_InputDevice *device,
				       const CL_Key &key)
{
  int id,asc;

  if (this->device != device) return; // wrong device.

  id=key.id;
  asc=key.ascii;

  /*
   * Now the following test is very important. For all the keys that
   * return a correct scancode, we consider this is the right way
   * to poll the key, and it's usually better than using the ascii code,
   * since keyboards can have different layouts (I personnally have
   * both AZERTY and QWERTY keyboards at home, and I like it when
   * games know how to make the difference).
   */
  if (id!=CL_KEY_NONE_OF_THE_ABOVE)
    {
      scancode[id].press(id,asc);
    }
  /* 
   * we exclude the "space" char and all the numbers,
   * since they are usually allways at the same place on
   * keyboards, and on my AZERTY keyboard, the number keys return
   * accents, &, # and many other things which _are_ valid ascii
   * codes, but everyone thinks of the key as being 1 2 or 6.
   * So numbers are polled by their scancode, that's the point.
   */ 
  else if (asc != ' '
	   && asc>=U61_INPUTMONITOR_FIRST_ASCII 
	   && asc<=U61_INPUTMONITOR_LAST_ASCII && 
	   !(id>=CL_KEY_0 && id<=CL_KEY_9))
    {
      ascii[asc-U61_INPUTMONITOR_FIRST_ASCII].press(id,asc);
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a key is released
 */ 
void U61_InputMonitor::on_button_release(CL_InputDevice *device,
					 const CL_Key &key)
{
  int i;
  int id,asc;

  if (this->device != device) return; // wrong device.

  id=key.id;
  asc=key.ascii;

  /*
   * Now the following test is very important. For all the keys that
   * return a correct scancode, we consider this is the right way
   * to poll the key, and it's usually better than using the ascii code,
   * since keyboards can have different layouts (I personnally have
   * both AZERTY and QWERTY keyboards at home, and I like it when
   * games know how to make the difference).
   */ 
  if (id!=CL_KEY_NONE_OF_THE_ABOVE)
    { 
      scancode[id].release(id,asc);
    }
  /* 
   * we exclude the "space" char and all the numbers,
   * since they are usually allways at the same place on
   * keyboards, and on my AZERTY keyboard, the number keys return
   * accents, &, # and many other things which _are_ valid ascii
   * codes, but everyone thinks of the key as being 1 2 or 6.
   * So numbers are polled by their scancode, that's the point.
   */ 
  else if (asc != ' '
	   && asc>=U61_INPUTMONITOR_FIRST_ASCII 
	   && asc<=U61_INPUTMONITOR_LAST_ASCII && 
	   !(id>=CL_KEY_0 && id<=CL_KEY_9))
    {
      ascii[asc-U61_INPUTMONITOR_FIRST_ASCII].release(id,asc);
    }

  /*
   * Now for every key, we check up that any key that is mapped
   * to the same scancode is released. This is to avoid the
   * dreadfull: a player presses 'a', another one presses 'shift'
   * and the released key is 'A'. Of course this technique can
   * lead to inconsistency if a given ascii code can be otbained
   * with several scancodes. In this case a key will be released
   * when it should not have been released, but this is much less
   * annoying than having a key permanently pressed...
   */
  if (id!=CL_KEY_NONE_OF_THE_ABOVE)
    {
      for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_SCANCODE;++i)
	{
	  if (scancode[i].get_scancode()==id)
	    {
	      scancode[i].release(id,asc);
	    }
	}
      for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_ASCII;++i)
	{
	  if (ascii[i].get_scancode()==id)
	    {
	      ascii[i].release(id,asc);
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a key is released
 */ 
void U61_InputMonitor::check()
{
  int i;

  for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_ASCII;++i)
    {
      ascii[i].check();
    }

  for (i=0;i<U61_INPUTMONITOR_NUM_KEYS_SCANCODE;++i)
    {
      scancode[i].check();
    }
}







