/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   key.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: used to poll a key. this class implements key repetition
 *              on a per-key basis. since repetition settings cna be
 *              changed for any key, u61 is able to have your "move" keys
 *              to repeat faster then "rotate" keys. and the drop key
 *              may not repeat at all ;)
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "key.h"
#include "log.h"
#include "global.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a key object
 */ 
U61_Key::U61_Key(int key_code, 
                 int evt_code_out, 
                 int first_repeat_delay,
                 int standard_repeat_delay)
{
  init(key_code,evt_code_out,first_repeat_delay,standard_repeat_delay);
  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a key object
 */ 
U61_Key::~U61_Key()
{
 
}

/*--------------------------------------------------------------------------*/
/*
 * inits the current key with a key to poll and a code to return
 */
void U61_Key::init(int key_code_in,
                   int evt_code_out,
                   int first_repeat_delay,
                   int standard_repeat_delay)
{
  code_in=key_code_in; 
  code_out=evt_code_out;
  first_delay=first_repeat_delay;
  standard_delay=standard_repeat_delay;
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * fires the event when the key has been received
 */
void U61_Key::fire(int time)
{
  U61_Event event;

  event.code=code_out;
  event.time=time;

  queue.push_back(event);

  last_fired=time;
  U61_LOG_DEBUG("fire "<<event<<" at time "<<time);
}

/*--------------------------------------------------------------------------*/
/* 
 * polls the current key, marking events with a precise time
 */ 
void U61_Key::poll(int time)
{
  bool pressed;

  pressed=U61_Global::input_monitor.is_pressed(code_in);    

  if (pressed)
    {
      switch (state)
	{
        case U61_KEY_RELEASED:
	  state=U61_KEY_PRESSED;
	  fire(time);
	  break;
        case U61_KEY_PRESSED:
	  if (time-last_fired>first_delay)
	    {
	      state=U61_KEY_REPEATING;
	      fire(last_fired+first_delay);
	    }
	  break;
        case U61_KEY_REPEATING:
	  while (time-last_fired>standard_delay)
	    {
	      fire(last_fired+standard_delay);
	    }
	  break;
	}  
    }
  else
    {
      state=U61_KEY_RELEASED;
      last_fired=0;
    } 
}

/*--------------------------------------------------------------------------*/
/*
 * resets the informations about the presses on this key
 */
void U61_Key::reset()
{
  last_fired=0;
  state=U61_KEY_RELEASED;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there are no key events pending
 */
bool U61_Key::empty()
{
  return queue.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next event
 * usually there should very rarely be several key events
 * generated in one call to poll, but who knows...
 */
U61_Event U61_Key::get()
{
  U61_Event event;

  event=queue.front();
  queue.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the input code, ie the scan code used to poll the key
 */
int U61_Key::get_code_in()
{
  return code_in;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the output code, ie the code returned when the key is pressed
 */
int U61_Key::get_code_out()
{
  return code_out;
}


