/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   map.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: I called this class map for I can't imagine a game without
 *              a "map" class 8) . seriously, a map is where the information
 *              about where the blocks are is stored. it's basically an array
 *              of squares.
 *              it also provides an API to move the player's falling block
 *              and handle any event.
 *              it's important to note that most of the behaviors of a map
 *              are determined by embedded LUA scripts
 */


/*---------------------------------------------------------------------------
  include
  ---------------------------------------------------------------------------*/

#include "global.h"
#include "map.h"
#include "script.h"
#include "time.h"
#include "sound.h"
#include "utils.h"
#include "log.h"
#include "music.h"
#include "serial.h"
#include "checksum.h"
#include "macro.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

#define U61_MAP_MIN_WIDTH 2
#define U61_MAP_MIN_HEIGHT 5
#define U61_MAP_SPEED_SCALE 1000
#define U61_MAP_ACCEL_SCALE 100000
#define U61_MAP_MAX_SYSTEM_SPEED 50
#define U61_MAP_MAX_SYSTEM_ACCEL 15
#define U61_MAP_MAX_SCORE 1000000
#define U61_MAP_TIME_CALLBACK_1_DELAY 100
#define U61_MAP_TIME_CALLBACK_10_DELAY 10
#define U61_MAP_TIME_CALLBACK_100_DELAY 1
#define U61_MAP_CHECK_TARGET_DELAY 100
#define U61_MAP_CHECK_CURSE_DELAY 100
#define U61_MAP_ANTICIPATE_MAX 1000
#define U61_MAP_SERIALIZED_FOR_CHECKSUM_SIZE (U61_MAP_MAX_WIDTH*U61_MAP_MAX_HEIGHT*U61_SQUARE_SERIALIZED_SIZE + 2*U61_BLOCK_SERIALIZED_SIZE + U61_MAP_NB_GLOBAL * U61_SERIAL_INT_SERIALIZED_SIZE + U61_MAP_MAX_CURSE_ID*U61_REGISTEREDCURSE_SERIALIZED_SIZE + 20*U61_SERIAL_INT_SERIALIZED_SIZE + 2*U61_SERIAL_BOOL_SERIALIZED_SIZE)

/*--------------------------------------------------------------------------*/
/* 
 * these arrays contain constants used for speeds, this way one can
 * have something more subtle than just a linear behavior
 */ 
static int speeds[U61_MAP_MAX_SYSTEM_SPEED+1]=
  {
    1,  2,  5,  10, 20, 25, 30, 35, 40, 45, 50,
    55, 60, 65, 70, 75, 80, 85, 90, 95, 100,
    105,110,115,120,125,130,135,140,145,150,
    155,160,165,170,175,180,185,190,195,200,
    220,240,260,280,300,350,400,450,500,1000
  };

static int accels[U61_MAP_MAX_SYSTEM_ACCEL+1]=
  {
    0,  2,  5,  10, 15, 20,
    30, 40, 50, 75, 100,
    150,200,250,500,1000
  };

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default map
 */ 
U61_Map::U61_Map()
{
  active=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a map
 */ 
U61_Map::~U61_Map()
{

}

/*--------------------------------------------------------------------------*/
/*
 * clears a map
 */
void U61_Map::clear()
{
  int i,j;

  for (j=0;j<U61_MAP_MAX_HEIGHT;++j)
    {
      for (i=0;i<U61_MAP_MAX_WIDTH;++i)
        {
	  squares[i][j].disable();
	  //squares[i][j].enable();
	  //squares[i][j].set_color((i+j)%8);
        }
    }
  curse_chosen=false;
  curse_asked=false;
  curse_x=0;
  curse_y=0;
}


/*--------------------------------------------------------------------------*/
/*
 * begin function, to call each time the player has lost 
 */
void U61_Map::begin()
{
  int i;
  U61_Event evt;

  width=U61_MAP_MAX_WIDTH;
  height=U61_MAP_MAX_HEIGHT;

  auto_events.clear();
  request_events.clear();
  
  clear();
 
  block.reset();
  next_block.reset();
  block_requested=false;
  matching=false;
  speed_counter=0;
  accel_counter=0;
  speed=base_speed;
  accel=base_accel; 
  background=0;
  anticipation_state=true;
  preview_state=true;
  match_count=0;
  curse_counter=0;
  curse_chosen=false;
  curse_asked=false;
  curse_state=false;
  nb_antidote=0;
  score=0;
  curse_sent=0;
  curse_received=0;
  start_time=0;
  for (i=0;i<U61_MAP_NB_GLOBAL;++i)
    {
      global_val[i]=0;
    }        
  for (i=0;i<U61_MAP_MAX_CURSE_ID;++i)
    {
      registered_curse[i].reset();
    }        
  change_background();

  evt.code=U61_EVENT_SET_VICTIM;
  evt.par=target_id;
  put_auto_event(evt);
}

/*--------------------------------------------------------------------------*/
/*
 * initialization of a map, must be called before the map is used
 */
void U61_Map::init(int id,int s,int a,int d)
{
  player_id=id;
  base_speed=s;
  base_accel=a;
  curse_delay=d*100;

  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * resets completely a map
 */
void U61_Map::reset()
{
  target_id=player_id;
  map_time=0;
  active=false;
  score=0;
  silent=false;
  name[0]='\0'; 
  last_checksum=0;
  checksum_requested=false;

  begin();
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the map is activated
 * by activated map, we mean a map where blocks are falling and someone
 * is actually playing. a player can have a disactivated map,in this
 * case u61 is waiting for him to press the start key and activate the map
 */
bool U61_Map::is_active()
{
  return active;
}

/*--------------------------------------------------------------------------*/
/*
 * activates a map
 */
void U61_Map::set_active(unsigned int time)
{
  map_time=time;

  begin();
  /*
   * We call the start user func, which will perform initializations.
   */
  start();

  active=true;
}

/*--------------------------------------------------------------------------*/
/*
 * disactivates a map
 */
void U61_Map::set_inactive()
{
  active=false;
}
 
/*--------------------------------------------------------------------------*/
/*
 * Returns the width of the map.
 */
int U61_Map::get_width()
{
  return width;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the height of the map.
 */
int U61_Map::get_height()
{
  return height;
}

/*--------------------------------------------------------------------------*/
/*
 * changes the background of the map randomly
 */
void U61_Map::set_width(int w)
{
  if (w<U61_MAP_MIN_WIDTH) 
    {
      w=U61_MAP_MIN_WIDTH;
    }
  if (w>U61_MAP_MAX_WIDTH)
    {
      w=U61_MAP_MAX_WIDTH;
    }

  /*
   * We re-center the map
   */
  shift((w-width)/2,0);

  width=w;

  check_map();
}

/*--------------------------------------------------------------------------*/
/*
 * changes the background of the map randomly
 */
void U61_Map::set_height(int h)
{
  if (h<U61_MAP_MIN_HEIGHT) 
    {
      h=U61_MAP_MIN_HEIGHT;
    }
  if (h>U61_MAP_MAX_HEIGHT)
    {
      h=U61_MAP_MAX_HEIGHT;
    }

  /*
   * We re-center the map
   */
  shift(0,h-height);

  height=h;

  check_map();
}

/*--------------------------------------------------------------------------*/
/*
 * changes the background of the map randomly
 */
void U61_Map::change_background()
{
  U61_LOG_DEBUG("Number of backgrounds"<<U61_Global::data.nb_map);
  background=U61_Utils::random(U61_Global::data.nb_map);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the background of the map
 */
int U61_Map::get_background()
{
  return background;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
void U61_Map::set_square_color(int x,int y,int color)
{
  if (x>=0 && y>=0 && x<width && y<height)
    {
      if (color>=0 && color<U61_SQUARE_NB_COLORS)
        {
	  squares[x][y].enable();
	  squares[x][y].set_color(color);
        }
      else
        {
	  squares[x][y].disable();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
int U61_Map::get_square_color(int x,int y)
{
  int color=-1;

  if (x>=0 && y>=0 && x<width && y<height)
    {
      if (squares[x][y].is_enabled())
        {
	  color=squares[x][y].get_color();
        }
    }
  return color;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the score of the map, negative scores are forbidden
 */
void U61_Map::set_score(int s)
{
  if (s>=0)
    {
      if (s<U61_MAP_MAX_SCORE)
	{
          score=s;
	}
      else
        {
	  score=U61_MAP_MAX_SCORE-1;
	}
    }
  else
    {
      score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the score associated to the map
 */
int U61_Map::get_score()
{
  return score;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the time of the last game
 */
int U61_Map::get_elapsed_time()
{
  int elapsed;

  if (map_time<=0 || start_time<=0 || map_time<start_time)
    {
      elapsed=0;
    }
  else
    {
      elapsed=map_time-start_time;
    }

  return elapsed;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of curses sent during the last game
 */
int U61_Map::get_curse_sent()
{
  return curse_sent;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of curses received during the last game
 */
int U61_Map::get_curse_received()
{
  return curse_received;
}

/*--------------------------------------------------------------------------*/
/*
 * adds the given value to the current score
 */
void U61_Map::add_score(int s)
{
  score+=s;
  if (score<0)
    {
      score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the time associated to the map
 */
int U61_Map::get_time()
{
  return map_time;
}

/*--------------------------------------------------------------------------*/
/*
 * returns a number which is used by the lua match function
 */
int U61_Map::get_match_count()
{
  return match_count;
}

/*--------------------------------------------------------------------------*/
/*
 * sets a global integer into the map
 * used for lua scripts to store global permanent values
 */
void U61_Map::set_global(int i, int glob)
{
  if (i>=0 && i<U61_MAP_NB_GLOBAL)
    {
      global_val[i]=glob;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets a global integer from the map
 * used for lua scripts to store global permanent values
 */
int U61_Map::get_global(int i)
{
  int glob=0;

  if (i>=0 && i<U61_MAP_NB_GLOBAL)
    {
      glob=global_val[i];
    }

  return glob;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the age of a persistent curse
 * If negative, it means the curse is not registered or is over
 */
int U61_Map::get_curse_age(int i)
{
  int age=false;

  if (i>=0 && i<U61_MAP_MAX_CURSE_ID)
    {
      if ((!registered_curse[i].active) 
	  || map_time>registered_curse[i].end)
	{
	  age=-1;
	}
      else
	{
          age=((map_time-registered_curse[i].begin));
	}
    }

  return age;
}

/*--------------------------------------------------------------------------*/
/*
 * Registers a persistent curse. After this, calls to get_curse_age
 * will return how long ago the curse has been registered
 */
void U61_Map::register_curse(int i, int length, bool good)
{
  if (i>=0 && i<U61_MAP_MAX_CURSE_ID)
    {
      /*
       * if length is zero, the default length is 10 hours
       * that's too say infinite...
       */
      if (length<=0)
	{
	  length=3600000;
	}
      registered_curse[i].begin=map_time;
      registered_curse[i].end=map_time+length;
      registered_curse[i].active=true;
      registered_curse[i].good=good;
      U61_LOG_DEBUG("Registering "<<i<<" begin="<<registered_curse[i].begin<<" end="<<registered_curse[i].end);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Cancels a registered curse
 */
void U61_Map::cancel_curse(int i)
{
  if (i>=0 && i<U61_MAP_MAX_CURSE_ID)
    {
      registered_curse[i].active=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the oldest curse available
 * Returns -1 if none existing
 */
int U61_Map::get_oldest_curse(bool good)
{
  int oldest;
  int i;
  int result=-1;

  oldest=map_time;
  
  for (i=0;i<U61_MAP_MAX_CURSE_ID;++i)
    {
      if (get_curse_age(i)>=0 &&
	  registered_curse[i].good==good)
	{ 
	  oldest=map_time;
          result=i;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of persisten curses which are affecting the map
 */
int U61_Map::get_nb_curse(bool good)
{
  int nb;
  int i;

  nb=0;
  for (i=0;i<U61_MAP_MAX_CURSE_ID;++i)
    {
      if (get_curse_age(i)>=0 &&
	  registered_curse[i].good==good)
	{
	  nb++;
	}
    }
  return nb;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the number of antidotes
 */
int U61_Map::get_nb_antidote()
{
  return nb_antidote;
}

/*--------------------------------------------------------------------------*/
/*
 * Adds an antidote
 */
void U61_Map::add_antidote()
{
  nb_antidote++;
}

/*--------------------------------------------------------------------------*/
/*
 * Deletes an antidote
 */
void U61_Map::delete_antidote()
{
  if (nb_antidote>0)
    {
      nb_antidote--;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the x coordinate of the curse
 */
void U61_Map::set_curse_x(int x)
{
  if (x>=0 && x<width)
    {
      curse_x=x;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the y coordinate of the curse
 */
void U61_Map::set_curse_y(int y)
{
  if (y>=0 && y<height)
    {
      curse_y=y;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the state of the curse
 */
void U61_Map::set_curse_state(bool state)
{
  if (state)
    {
      curse_state=true;
    }
  else
    {
      curse_counter=0;
      curse_state=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the x coordinate of the curse
 */
int U61_Map::get_curse_x()
{
  int x=-1;

  if (curse_state && curse_chosen)
    {
      x=curse_x;
    }

  return x;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the y coordinate of the curse
 */
int U61_Map::get_curse_y()
{
  int y=-1;

  if (curse_state && curse_chosen)
    {
      y=curse_y;
    }

  return y;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the state of the curse
 */
bool U61_Map::get_curse_state()
{
  return curse_state!=false;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the curse state is true and the curse is located in
 * a square which is not empty/disabled
 */
bool U61_Map::is_curse_available()
{
  bool result=false;

  if (curse_x>=0 && curse_y>=0 && curse_x<width && curse_y<height
      && curse_state && curse_chosen
      && squares[curse_x][curse_y].is_enabled())
    {
      result=true;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends a curse to a remote player
 * In fact it justs puts a REQUEST_CURSE event in the event queue, and
 * the rest is done by dispatchers
 */
void U61_Map::send_curse(int curse_id)
{
  U61_Event evt;

  if (player_id!=target_id && curse_id>=0 && curse_id<U61_MAP_MAX_CURSE_ID)
    {
      U61_LOG_DEBUG("Sending curse "<<curse_id);

      curse_sent++;
      evt.code=U61_EVENT_REQUEST_CURSE;
      evt.par=curse_id;
      /*
       * We use put_request_event instead of calling put_auto_event
       * since the target is not the player here...
       */
      put_request_event(evt);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the next curse that will be fired
 */
char *U61_Map::get_curse_name()
{
  char *name;

  if (is_curse_available())
    {
      name=U61_Script::get_curse_name(this,curse_id);
    }
  else
    {
      name="";
    }

  return name;
}

/*--------------------------------------------------------------------------*/
/*
 * compute the map to the next instant
 */
void U61_Map::compute_next()
{
  map_time++;
  curse_counter++;
  update_explosions();
  if (!(map_time%U61_MAP_TIME_CALLBACK_1_DELAY))
    {
      time_callback_1();
    }
  if (!(map_time%U61_MAP_TIME_CALLBACK_10_DELAY))
    {
      time_callback_10();
    }
  if (!(map_time%U61_MAP_TIME_CALLBACK_100_DELAY))
    {
      time_callback_100();
    }
  if (!(map_time%U61_MAP_CHECK_TARGET_DELAY))
    {
      check_target();
    }
  if (!(map_time%U61_MAP_CHECK_CURSE_DELAY))
    {
      check_curse();
    }
  if (!(map_time%U61_MAP_DELAY_BETWEEN_CHECKSUM))
    {
      start_checksum();
    }
  if (!exists_explosion())
    {
      match_pattern();
    }
    
  update_speed();
}

/*--------------------------------------------------------------------------*/
/* 
 * updates the current speed
 */
void U61_Map::update_speed()
{
  check_speed();
  check_accel();

  speed_counter+=speeds[speed];
  if (speed_counter>U61_MAP_SPEED_SCALE)
    {
      speed_counter-=U61_MAP_SPEED_SCALE;
      block_move_down(); 
    }    
  accel_counter+=accels[accel];
  if (accel_counter>U61_MAP_ACCEL_SCALE)
    {
      accel_counter-=U61_MAP_ACCEL_SCALE;
      speed++;
      U61_LOG_DEBUG("Next speed ("<<speed<<")");
    }    

  check_speed();
  check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * computes the level up to the givent time, no events are handled
 */
void U61_Map::compute_up_to(unsigned int time)
{
  if (active)
    {
      while (map_time<time)
        {
	  compute_next();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * handles an event, ie does anything that has to be done for this event
 */
void U61_Map::handle_event(U61_Event *event)
{
  if (start_time==0)
    {
      start_time=map_time;
    }
  if (event->time<map_time)
    {
      U61_LOG_DEBUG("Inconsitency in event dates detected with event "<<*event<<" and map_time="<<map_time);
    }  
  else
    {
      compute_up_to(event->time);
      
      if (active)
	{
	  switch (event->code)
            {
            case U61_EVENT_ROTATE_LEFT:
	      block_rotate_left();
	      break;
            case U61_EVENT_ROTATE_RIGHT:
	      block_rotate_right();
	      break;
            case U61_EVENT_MOVE_RIGHT:
	      block_move_right();
	      break;
            case U61_EVENT_MOVE_LEFT:
	      block_move_left();
	      break;
            case U61_EVENT_MOVE_DOWN:
	      block_move_down();
	      break;
            case U61_EVENT_DROP:
	      block_drop();
	      break;
            case U61_EVENT_NEW_BLOCK:
	      new_block(event->par);
	      break;
            case U61_EVENT_LOOSE:
	      loose();
	      break;
            case U61_EVENT_PREV_VICTIM:
              prev_victim(event);
              break;
            case U61_EVENT_NEXT_VICTIM:
              next_victim(event);
              break;
            case U61_EVENT_SET_VICTIM:
              set_victim(event->par);
              break;
            case U61_EVENT_NEW_CURSE:
              new_curse(event->par);
	      break;
            case U61_EVENT_REQUEST_CURSE:
              request_curse(*event);
	      break;
            case U61_EVENT_DO_CURSE:
              do_curse(*event);
	      break;
            case U61_EVENT_USE_ANTIDOTE:
              use_antidote();
              break;
            case U61_EVENT_NAME_LETTER:
              name_add_letter(event->par);
	      break;
            case U61_EVENT_REQUEST_CHECKSUM:
              request_checksum();
	      break;
            case U61_EVENT_VERIFY_CHECKSUM:
              verify_checksum(event->par);
	      break;
            }
	}
      else
	{
	  switch (event->code)
            {
            case U61_EVENT_NAME_LETTER:
              name_add_letter(event->par);
	      break;
            case U61_EVENT_REQUEST_CHECKSUM:
              request_checksum();
	      break;
            case U61_EVENT_VERIFY_CHECKSUM:
              verify_checksum(event->par);
	      break;
	    }
	}
    }

  U61_LOG_DEBUG("Event computed "<<*event);
}

/*--------------------------------------------------------------------------*/
/*
 * anticipates the level to the given time, it's usefull to display the
 * "last" map. the computer just supposes no events have been generated
 * and anticipates the behavior of the map
 * but careful, since no events are computed, themap may become incoherent
 * it's not a problem for in U61 we keep 2 copies of the map, a sure and
 * an anticipated one, called "last"
 * BTW, this function deletes all events from the auto_events list
 * since they are of no use when the map has been anticipated and/or
 * we don't want the list to grow and explose system memory...
 */
void U61_Map::anticipate(unsigned int time)
{
  /*
   * we do not anticipate too much for this can lead to
   * deadlocks when the CPU is overloaded.
   * We limit the anticipation to 10 real seconds, knowing
   * that fake events are sent 10 times/sec on local players
   * and 1 time/sec for remote players
   */
  if (time-map_time>U61_MAP_ANTICIPATE_MAX)
    {
      time=map_time+U61_MAP_ANTICIPATE_MAX;
    }
  compute_up_to(time);
  auto_events.clear();
  request_events.clear();
}

/*--------------------------------------------------------------------------*/
/*
 * polls some auto generated events, such as requests for new blocks
 */
void U61_Map::poll_auto_events()
{
  U61_Event event;

  if (active)
    {
      if ((!is_block_active()) 
	  && (!block_requested) 
	  && (!matching))
        {
	  block_requested=true;
	  event.code=U61_EVENT_NEW_BLOCK;
	  event.par=U61_Script::new_shape(this,&block);
	  put_auto_event(event);
	  U61_LOG_DEBUG("new block event sent");
        }
      if (curse_state
          && (((!curse_asked) && (!curse_chosen)) 
	      || curse_counter>curse_delay)
	  && (!exists_explosion()))
	{
	  curse_asked=true;
	  curse_counter=0;
          event.code=U61_EVENT_NEW_CURSE;
          event.par=U61_Script::new_curse(this,&block);
          put_auto_event(event);
          U61_LOG_DEBUG("Auto curse event "<<event.par);
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next auto generated event
 */
U61_Event U61_Map::get_auto_event()
{
  U61_Event event;

  event=auto_events.front();
  auto_events.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there is an auto event
 */
bool U61_Map::exists_auto_event()
{
  return !auto_events.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the generated event queue
 */
void U61_Map::put_auto_event(U61_Event event)
{
  if (auto_events.size()<U61_MAP_MAX_AUTO_EVENTS)
    {
      auto_events.push_back(event);
    }
  else
    {
      U61_LOG_WARNING("Too many auto events!");
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next generated event request
 */
U61_Event U61_Map::get_request_event()
{
  U61_Event event;

  event=request_events.front();
  request_events.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there is an auto event request
 */
bool U61_Map::exists_request_event()
{
  return !request_events.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the request event queue
 */
void U61_Map::put_request_event(U61_Event event)
{
  if (auto_events.size()<U61_MAP_MAX_REQUEST_EVENTS)
    {
      event.target=target_id;
      event.author=player_id;
      event.target=target_id;
      event.time=0;
      request_events.push_back(event);
    }
  else
    {
      U61_LOG_WARNING("Too many request events!");
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the player playing on this map
 */
char *U61_Map::get_name()
{
  return name;
}

/*--------------------------------------------------------------------------*/
/*
 * draws the map onto the screen
 */
void U61_Map::draw(int x,int y, int size, bool anti)
{
  int i,j;
  int draw_x,draw_y;
  int di,dj;
  U61_Square filler;

  if (active)
    {
      di=(U61_MAP_MAX_WIDTH-width)/2;
      dj=(U61_MAP_MAX_HEIGHT-height)/2;

      /*
       * First we draw the "filler" squares in case map is smaller
       * than 10*25.
       */    
      filler.enable();
      for (j=0;j<U61_MAP_MAX_HEIGHT;++j)
	{
	  for (i=0;i<U61_MAP_MAX_WIDTH;++i)
	    {
	      if (i<di || i>=di+width || j<dj || j>=dj+height)
		{
		  filler.draw_filler(x+i*U61_Global::data.square_w[size],
				     y+j*U61_Global::data.square_h[size],
				     size);
		}
	    }
	}
      
      x+=di*U61_Global::data.square_w[size];
      y+=dj*U61_Global::data.square_h[size];

      /*
       * Now we draw map contents
       */    
      for (j=0;j<height;++j)
        {
	  for (i=0;i<width;++i)
            {
	      draw_x=x+i*U61_Global::data.square_w[size];

	      draw_y=y+j*U61_Global::data.square_h[size];

              if (j==curse_y && i==curse_x && 
		  curse_state && curse_chosen &&
                  squares[i][j].is_enabled())
		{
                  if (U61_Square::should_blink())
		    {
		      squares[i][j].draw_curse(draw_x,draw_y,size);
		    }
		  else
		    {
		      squares[i][j].draw(draw_x,draw_y,size);
		    }
		}
	      else
		{
		  squares[i][j].draw(draw_x,draw_y,size);
		}
            }
        }
      if (anticipation_state && anti)
        {
	  draw_anticipation(x,y,size);
        }
      block.draw(x,y,size,width,height);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the width of the map on the screen
 */
int U61_Map::get_draw_width(int size)
{
  return U61_MAP_MAX_WIDTH*U61_Global::data.square_w[size];
}

/*--------------------------------------------------------------------------*/
/*
 * returns the height of the map on the screen
 */
int U61_Map::get_draw_height(int size)
{
  return U61_MAP_MAX_HEIGHT*U61_Global::data.square_h[size];
}

/*--------------------------------------------------------------------------*/
/*
 * sets the map in a mute state (does not generate sounds any more)
 */
void U61_Map::mute()
{
  silent=true;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the anticipation state
 */
void U61_Map::set_anticipation_state(bool state)
{
  anticipation_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the anticipation state
 */
bool U61_Map::get_anticipation_state()
{
  return anticipation_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the preview state
 */
void U61_Map::set_preview_state(bool state)
{
  preview_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the preview state
 */
bool U61_Map::get_preview_state()
{
  return preview_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the speed 
 */
void U61_Map::set_speed(int s)
{
  speed=s; 
  check_speed();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the speed 
 */
int U61_Map::get_speed()
{
  return speed; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the speed is within the correct range
 */
void U61_Map::check_speed()
{
  if (speed>U61_MAP_MAX_SYSTEM_SPEED)
    {
      speed=U61_MAP_MAX_SYSTEM_SPEED;
    }
  if (speed<0)
    {
      speed=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the accel
 */
void U61_Map::set_accel(int a)
{
  accel=a; 
  check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the accel
 */
int U61_Map::get_accel()
{
  return accel; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the accel is within the correct range
 */
void U61_Map::check_accel()
{
  if (accel>U61_MAP_MAX_SYSTEM_ACCEL)
    {
      accel=U61_MAP_MAX_SYSTEM_ACCEL;
    }
  if (accel<0)
    {
      accel=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the nest block 
 */
U61_Block U61_Map::get_next_block()
{
  return next_block; 
}

/*--------------------------------------------------------------------------*/
/*
 * puts the square in an exploding state
 */
void U61_Map::blow_up_square(int x,int y)
{
  if (x>=0 && y>=0 && x<width && y<height)
    {
      squares[x][y].begin_explosion();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the square is exploding
 */
bool U61_Map::is_square_exploding(int x,int y)
{
  bool result=false;

  if (x>=0 && y>=0 && x<width && y<height)
    {
      result= squares[x][y].is_exploding();
    }
    
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if there's at least one square exploding
 */
bool U61_Map::exists_explosion()
{
  bool result=false;
  int x,y;

  for (y=0;y<height;++y)
    for (x=0;x<width;++x)
      {
	if (squares[x][y].is_exploding())
	  {
	    result=true;
	  } 
      }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * display the anticipation, the anticipation is "where the block will land if
 * nothing happens" ie where it will land if I press drop  
 */
void U61_Map::draw_anticipation(int x,int y,int size)
{
  U61_Block test;
  U61_Block prev;

  if (block.get_nb_items()>0)
    {
      test=block;
    
      while (is_block_ok(&test))
        {
	  prev=test;
	  U61_Script::move_down(this,&test);
          test.center();
        }

      U61_Script::land(this,&prev);
      prev.draw_anticipation(x,y,size,width,height); 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user "start" callback (at the beginning of each game)
 * it is not possible
 */
void U61_Map::start()
{
  U61_LOG_DEBUG("start");

  U61_Script::start(this,&block);

  check_block();
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_left()
{
  U61_Block test;

  U61_LOG_DEBUG("rotate left");

  test=block;
  U61_Script::rotate_left(this,&test);
  test.center();

  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      test=block;
      U61_Script::move_left(this,&test);
      U61_Script::rotate_left(this,&test);
      test.center();

      if (is_block_ok(&test))
        {
	  block=test;
        }
      else
        {
	  test=block;
	  U61_Script::move_right(this,&test);
	  U61_Script::rotate_left(this,&test);
          test.center();
	  if (is_block_ok(&test))
            {
	      block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_right()
{
  U61_Block test;

  U61_LOG_DEBUG("rotate right");

  test=block;
  U61_Script::rotate_right(this,&test);
  test.center();
  
  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      test=block;
      U61_Script::move_right(this,&test);
      U61_Script::rotate_right(this,&test);
      test.center();

      if (is_block_ok(&test))
        {
	  block=test;
        }
      else
        {
	  test=block;
	  U61_Script::move_left(this,&test);
	  U61_Script::rotate_right(this,&test);
          test.center();
	  if (is_block_ok(&test))
            {
	      block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_left()
{
  U61_Block test;

  U61_LOG_DEBUG("move left");

  test=block;
  U61_Script::move_left(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_right()
{
  U61_Block test;

  U61_LOG_DEBUG("move right");

  test=block;
  U61_Script::move_right(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block down, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_down()
{
  U61_Block test;

  test=block;
  U61_Script::move_down(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      stabilize_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * performs calls to move_down as much as possible so that the block lands
 */
void U61_Map::block_drop()
{
  U61_Block test;

  U61_LOG_DEBUG("drop");

  test=block;
  if (block.get_nb_items()>0)
    {
      while (is_block_ok(&test))
        {
	  block=test;
	  U61_Script::move_down(this,&test);
          test.center();
        }
    
      stabilize_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Adds a letter to the map name. If the value of c is 0 then the name
 * is "reseted" to "".
 */
void U61_Map::name_add_letter(char c)
{
  int len;

  if (c)
    {
      len=strlen(name);
      if (len<U61_DATA_PLAYER_NAME_SIZE)
	{
	  name[len]=c;
	  name[len+1]=0;
	}
    }
  else
    {
      name[0]='\0';
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Uses an antidote
 */
void U61_Map::use_antidote()
{
  U61_LOG_DEBUG("Using antidote");

  if (nb_antidote>0)
    {
      nb_antidote--;
      U61_Script::use_antidote(this,&block);
      /*
       * The call to check_block is very important since use_antidote
       * has full rights on the map and block.
       */
      check_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * checks if there's an active block (size greater than 0)
 */
bool U61_Map::is_block_active()
{
  return block.get_nb_items()>0;
}

/*--------------------------------------------------------------------------*/
/*
 * checks if a position is possible, if yes, returns true
 * this function is used to check if the result of functions such
 * block_rotate_right or block_move_down are incoherent, and therefore
 * should be cancelled
 */
bool U61_Map::is_block_ok(U61_Block *test)
{
  int i,n,x,y;
  bool ok=true;

  n=test->get_nb_items();
  for (i=0;i<n && ok;++i)
    {
      x=test->get_x()+test->get_item_x(i);
      y=test->get_y()+test->get_item_y(i);
      /*
       * the square must be in the map, or over the map,
       * if the block is brand new and didn't show himself yet
       */
      if (x>=0 && x<width && y<height)
        {
	  /*
	   * if y>=0, there could potentially be a stable square
	   * at this location, so we check that
	   */
	  if (y>=0)
            {
	      if (squares[x][y].is_enabled()) 
                {
		  ok=false;
                }
            }
        }
      else
        {
	  ok=false;
        }
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * gets a new block, the shape of the new block is calculated with
 * Lua user defined functions
 */
void U61_Map::new_block(int shape)
{
  block=next_block;
  block.set_x(width/2);
  block_requested=false;

  next_block.reset();
  U61_Script::do_shape(this,&next_block,shape);
  next_block.center();
  /*
   * I used to corner the block before drawing it. Now they are drawn centered
   * right away, so this line became useless.
   */
  //next_block.corner();

  block.center();

  U61_LOG_DEBUG("new block given");
}

/*--------------------------------------------------------------------------*/
/*
 * this function transfoem the moving block into a stable block,
 * ie it adds it to the current map
 */
void U61_Map::stabilize_block()
{
  U61_Event evt;
  int x,y,i,n;
  bool outside=false;

  /*
   * Before we merge the block with the map, we move the block up
   * until there's no conflict with the map. At first sight, it seems
   * useless since usually stabilize comes after a move_down, but it's
   * just safest to do this check, since many things can happen in
   * a curse for instance.
   */
  while (!is_block_ok(&block))
    {
      block.set_y(block.get_y()-1);
    }

  U61_Script::land(this,&block);
  n=block.get_nb_items();
  for (i=0;i<n;++i)
    {
      x=block.get_x()+block.get_item_x(i);
      y=block.get_y()+block.get_item_y(i);
      if (x>=0 && y>=0 && x<width && y<height)
        {
	  squares[x][y]=block.get_item(i)->square;
        }
      else
        {
	  outside=true;
        }
    }
  if (outside)
    {
      evt.code=U61_EVENT_LOOSE;
      put_auto_event(evt);
    }
  else
    {
      match_count=0;
      matching=true;
    }

  block.reset();
}

/*--------------------------------------------------------------------------*/
/*
 * what has to be done when the player looses
 */
void U61_Map::loose()
{
  if (!silent)
    {
      U61_Sound::play_player_loose();
      U61_Music::change();
    }

  U61_LOG_DEBUG("The game is lost");

  begin();
  /*
   * We call the start user func, which will perform initializations.
   */
  start();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function to match patterns
 * pattern match can be find a line and delete it, or find some colors
 * together, well, it can be pretty much anything...
 * it is usually quite a complex function
 * it also updates the score
 */
void U61_Map::match_pattern()
{
  if (matching)
    {
      if ((U61_Script::match_pattern(this)))
        {
	  if (!silent)
            {
	      U61_Sound::play_block_pattern();
            }
	  match_count++;
        }
      else
        {
	  if (match_count==0 && !silent)
            {
	      if (!silent)
		{
		  U61_Sound::play_block_touch();
		}
            }
	  matching=false;
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback_1
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback_1()
{
  U61_Script::time_callback_1(this,&block);
  /*
   * The call to check_block is very important since time_callback
   * has full rights on the map and block.
   */
  check_block();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback_10
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback_10()
{
  U61_Script::time_callback_10(this,&block);
  /*
   * The call to check_block is very important since time_callback
   * has full rights on the map and block.
   */
  check_block();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback_100
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback_100()
{
  U61_Script::time_callback_100(this,&block);
  /*
   * The call to check_block is very important since time_callback
   * has full rights on the map and block.
   */
  check_block();
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the update_explosion function for each square,
 * ie each explosion counter will be increased by 1
 */
void U61_Map::update_explosions()
{
  int x,y;
  bool curse_exploded=false;
  U61_Event event;

  for (y=0;y<height;++y)
    for (x=0;x<width;++x)
      { 
	squares[x][y].update_explosion();
	if (squares[x][y].is_explosion_ended())
	  {
	    U61_Script::square_blown_up(this,x,y);
	  }
	if (squares[x][y].is_explosion_new())
	  {
            /*
	     * If we find out that the square that just blew up
             * is the special curse square, then we need to launch the
             * script associated to this curse
             */
            if (curse_state && curse_chosen && curse_x==x && curse_y==y)
	      {
                curse_exploded=true;
	      }
	  }
      }

  if (curse_exploded)
    {
      /*
       * Now we could execute the Lua script right away, but we prefer
       * to send an event for 2 reasons:
       * 1 - the curse function can be quite long so it's useless to launch
       *     it too often
       * 2 - this way there's no difference between a curse you got because
       *     someone sent it to you and one you got by yourself matching
       *     a pattern, and I just like this idea
       */
      event.code=U61_EVENT_DO_CURSE;
      event.par=curse_id;
      put_auto_event(event);

      /*
       * The curse has been used so we disable it and ask for another one
       * by forcing curse_counter to curse_delay
       */
      curse_chosen=false;
      curse_asked=false;
      curse_counter=curse_delay;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Changes the curse weapon on the map. The weapon is changed on a regular
 * basis. This is done quite randomly, but still we can garantee it will
 * be the same "random" on every machine since the algorithm is based
 * on the map time.
 */
void U61_Map::new_curse(int id)
{
  int nb_squares;
  int x,y;
  int pos,chosen_pos;
 
  U61_LOG_DEBUG("New curse");

  nb_squares=0;
  for (y=0;y<height;++y)
    for (x=0;x<width;++x)
      {
        if (squares[x][y].is_enabled() && 
	    !squares[x][y].is_exploding())
	  {
	    nb_squares++;
	  }
      }
  U61_LOG_DEBUG("nb_squares="<<nb_squares);
  if (nb_squares>0)
    {
      /*
       * OK, now that we know there's at least one square, we choose to
       * put the curse square on one of these squares, the random touch
       * being added by dividing the map_time by the number of squares
       * enabled, and taking the rest. It's very important not to use
       * a random-like function since we need to garantee the map
       * will behave the same on every machine.
       */
      chosen_pos=map_time%nb_squares;
      pos=0;
      for (y=0;y<height;++y)
        for (x=0;x<width;++x)
	  {
	    if (squares[x][y].is_enabled() &&
		!squares[x][y].is_exploding())
	      {
		if (pos==chosen_pos)
		  {
		    curse_x=x;
		    curse_y=y;
		    U61_LOG_DEBUG("Curse chosen x="<<x<<" y="<<y);
		  }
		pos++;
	      }	
	  }
      
      curse_id=id;
      curse_chosen=true;
      curse_asked=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * handles REQUEST_CURSE events
 */
void U61_Map::request_curse(U61_Event event)
{
  /*
   * Now we have received a request for a curse so
   * we decide to issue a real curse. It's important to call
   * put_auto_event because of time stamp issues.
   */
  event.code=U61_EVENT_DO_CURSE;
  event.par+=U61_MAP_MAX_CURSE_ID;

  put_auto_event(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the lua function corresponding to the curse
 */
void U61_Map::do_curse(U61_Event evt)
{
  if (evt.par<(unsigned int) U61_MAP_MAX_CURSE_ID)
    {
      U61_LOG_DEBUG("Do own curse "<<evt);
      U61_Script::do_curse(this,&block,evt.par,false);
    }
  else
    {
      curse_received++;
      evt.par-=U61_MAP_MAX_CURSE_ID;
      U61_LOG_DEBUG("Do remote curse "<<evt);
      U61_Script::do_curse(this,&block,evt.par,true);
      if (!silent)
	{
	  U61_Sound::play_curse_receive();
	}
    }
  /*
   * The call to check_block is very important since do_curse
   * has full rights on the map and block.
   */
  check_block();
}

/*-------------------------------------------------------------------------*/
/*
 * Re-enables the curse_state if it has been inactive for too long
 * It also asks sets the curse_chosen parameter to false if there's no
 * curse available (this will cause a new curse to be chosen).
 * We use this mecanism to avoid calling is_curse_available all the time.
 */
void U61_Map::check_curse()
{
  if (curse_counter>=curse_delay)
    {
      curse_state=true;
    }
  if (curse_state==true && !is_curse_available())
    {
      curse_chosen=false;
      curse_asked=false;
    }
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the next victim
 */
void U61_Map::next_victim(U61_Event *event)
{
  target_id=U61_Global::layout.get_next_player_id(target_id,player_id);
  event->code=U61_EVENT_SET_VICTIM;
  event->par=target_id;
  U61_LOG_DEBUG("Next victim "<<target_id);
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the prev victim
 */
void U61_Map::prev_victim(U61_Event *event)
{
  target_id=U61_Global::layout.get_prev_player_id(target_id,player_id);
  event->code=U61_EVENT_SET_VICTIM;
  event->par=target_id;
  U61_LOG_DEBUG("Prev victim "<<target_id);
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the specified victim
 */
void U61_Map::set_victim(int id)
{
  target_id=id;
  U61_LOG_DEBUG("Set victim "<<target_id);
}

/*-------------------------------------------------------------------------*/
/*
 * returns the id of the current target
 */
int U61_Map::get_target_id()
{
  return target_id;
}

/*-------------------------------------------------------------------------*/
/*
 * we check that the target we have is not ourself or nobody, and send
 * an auto event if not
 */
void U61_Map::check_target()
{
  int test_id;
  U61_Event event;

  test_id=U61_Global::layout.check_player_id(target_id,player_id);
  if (test_id!=target_id)
    {
      event.code=U61_EVENT_SET_VICTIM;
      event.par=test_id;
      put_auto_event(event);
      U61_LOG_DEBUG("Correcting victim "<<test_id);
    }  
}

/*-------------------------------------------------------------------------*/
/*
 * This function is very important. It checks that the block has a correct
 * position. It is called after most of the Lua scripts which can
 * modify the block and/or the map. Indeed, it is important not to leave
 * the map in an incoherent position.
 * So this function moves the block sideways if it's too much on the left
 * or too much on the right, and then it moves it up until there's no
 * inconsistency.
 */
void U61_Map::check_block()
{
  int i,n,x;
  int min_x,max_x;
  U61_Block test;
  bool ok;
  bool moved_sideways=false;

  block.center();

  n=block.get_nb_items();

  /*
   * First step, we check that the block is not on the right or on
   * the left. Basically we want to put it "back in the middle"
   */
  min_x=width-1;
  max_x=0;
  for (i=0;i<n;++i)
    {
      x=block.get_x()+block.get_item_x(i);
      if (x<min_x)
	{
	  min_x=x;
	}
      if (x>max_x)
	{
	  max_x=x;
	}
    }
  /*
   * we move the block to the right if needed
   */
  if (min_x<0)
    {
      block.set_x(block.get_x()-min_x);
    }
  /*
   * we move the block to the left if needed
   */
  if (max_x>width-1)
    {
      block.set_x(block.get_x()-max_x+width-1);
    }

  /*
   * Now we are going to move the block up until it fits correctly.
   * This should solve a very common problem, which is that when
   * you fill the map with garbage, the block may be stuck in the
   * new added blocks. Therefore we move the block up. Note that in
   * this case the user is very likely to be in trouble to place the
   * block correctly, but there's no way to solve this except make
   * the block disappear, which is not a very good solution.
   */
  ok=is_block_ok(&block);
  while (!ok)
    {
      test=block;
      /*
       * we move the block up of 1 square
       */
      block.set_y(test.get_y()-1);
      if (!is_block_ok(&block))
	{
          /*
           * Ok, now if we get here it means one step up is not enough.
           * So what we'll do is try to move the block sideways a little
           * bit. This avoid the ridiculous situation where a block "climbs"
           * a very high wall to get stuck on it while there's plenty of
           * room elsewhere in the map.
           */
	  test.set_x(test.get_x()-1);
          if (is_block_ok(&test))
	    {
	      /*
	       * The block is OK after it has been moved on the left
	       */
	      moved_sideways=true;
	      block=test;
	      ok=true;
	    }
          else
	    {
	      test.set_x(test.get_x()+2);
	      if (is_block_ok(&test))
		{
		  /*
		   * The block is OK after it has been moved on the right
		   */
		  moved_sideways=true;
		  block=test;
		  ok=true;
		}
	    }
	}
      else
	{
	  /*
	   * The block is OK after it has been moved up.
	   * We move it up of one more square to give the player a chance to
	   * place it, otherwise it would be really unfair. However, it's
	   * still hard to place it correctly, but it's the game...
	   * Note that this is only done if we had needed to move the block
	   * sideways, otherwise one gets some strange buggy boucing effects
	   * when the time callback moves the block down for instance.
	   */
	  if (moved_sideways)
	    {
	      block.set_y(block.get_y()-1);
	    }
	  ok=true;
	}
    }
}

/*-------------------------------------------------------------------------*/
/*
 * This function is not as important as check_block but it can be usefull,
 * especially when resizing the map. Basically it disactivates any square
 * outside the map borders.
 */
void U61_Map::check_map()
{
  int i,j;

  for (j=height;j<U61_MAP_MAX_HEIGHT;++j)
    {
      for (i=width;i<U61_MAP_MAX_WIDTH;++i)
        {
	  squares[i][j].disable();
        }
    }
}

/*-------------------------------------------------------------------------*/
/*
 * This function shifts the content of a map vertically or horizontally.
 * It's mostly used when resizing a map, to recenter its contents. It's
 * also available within scripts.
 */
void U61_Map::shift(int x, int y)
{
  int i,j;

  if (x>0 && x<width)
    {
      for (j=0;j<height;++j)
	{
	  for (i=0;i<x;++i)
	    {
	      squares[i][j].disable();
	    }
	  for (i=width-1;i>=x;--i)
	    {
	      squares[i][j]=squares[i-x][j];  
	    }
	}
    }

  if (x<0 && x>-width)
    {
      for (j=0;j<height;++j)
	{
	  for (i=width+x;i<width;++i)
	    {
	      squares[i][j].disable();
	    }
	  for (i=0;i<width+x;++i)
	    {
	      squares[i][j]=squares[i-x][j];  
	    }
	}
    }

  if (y>0 && y<height)
    {
      for (i=0;i<width;++i)
	{
	  for (j=0;j<y;++j)
	    {
	      squares[i][j].disable();
	    }
	  for (j=height-1;j>=y;--j)
	    {
	      squares[i][j]=squares[i][j-y];  
	    }
	}
    }

  if (y<0 && y>-height)
    {
      for (i=0;i<width;++i)
	{
	  for (j=height+y;j<height;++j)
	    {
	      squares[i][j].disable();
	    }
	  for (j=0;j<height+y;++j)
	    {
	      squares[i][j]=squares[i][j-y];  
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Serializes a map into a byte strem
 */
bool U61_Map::serialize(unsigned char **buffer, unsigned char *buffer_limit)
{
  bool result=true;
  unsigned char *buffer_start=*buffer;
  int i,j,size;
  U61_Event evt;
  list<U61_Event> list_evt;

  for (i=0;i<U61_MAP_MAX_WIDTH && result;++i)
    for (j=0;j<U61_MAP_MAX_HEIGHT && result;++j)
      {
	result=result && squares[i][j].serialize(buffer,buffer_limit);
      }

  result=result && block.serialize(buffer,buffer_limit);
  result=result && next_block.serialize(buffer,buffer_limit);

  list_evt=auto_events;
  size=list_evt.size();
  result=result && U61_Serial::serialize_int(size,buffer,buffer_limit);
  for (i=0;i<U61_MAP_MAX_AUTO_EVENTS && result;++i)
    {
      if (i<size)
	{
	  result=result && list_evt.front().serialize(buffer,buffer_limit);	  
	  list_evt.pop_front();
	}
      else
	{
	  result=result && evt.serialize(buffer,buffer_limit);
	}
    }

  list_evt=request_events;
  size=list_evt.size();
  result=result && U61_Serial::serialize_int(size,buffer,buffer_limit);
  for (i=0;i<U61_MAP_MAX_REQUEST_EVENTS && result;++i)
    {
      if (i<size)
	{
	  result=result && list_evt.front().serialize(buffer,buffer_limit);	  
	  list_evt.pop_front();
	}
      else
	{
	  result=result && evt.serialize(buffer,buffer_limit);
	}
    }

  for (i=0;i<U61_MAP_NB_GLOBAL && result;++i)
    {
      result=result && U61_Serial::serialize_int(global_val[i],buffer,buffer_limit);
    }

  for (i=0;i<U61_MAP_MAX_CURSE_ID && result;++i)
    {
      result=result && registered_curse[i].serialize(buffer,buffer_limit);
    }

  result=result && U61_Serial::serialize_int(player_id,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(target_id,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(background,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(active,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(silent,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(map_time,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(width,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(height,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(block_requested,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(matching,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_delay,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(speed_counter,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(accel_counter,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(speed,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(accel,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(base_speed,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(base_accel,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(score,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_sent,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_received,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(start_time,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(match_count,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(anticipation_state,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(preview_state,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_x,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_y,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(curse_state,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(curse_chosen,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(curse_asked,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_id,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_counter,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(nb_antidote,buffer,buffer_limit);
  result=result && U61_Serial::serialize_short(last_checksum,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(checksum_requested,buffer,buffer_limit);

  if ((*buffer) - buffer_start != U61_MAP_SERIALIZED_SIZE)
    {
      U61_LOG_WARNING("Wrong buffer size for map serialization (diff="<<(buffer_start - (*buffer) + U61_MAP_SERIALIZED_SIZE)<<")!");
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Un-serializes a map from a byte strem
 */
bool U61_Map::unserialize(unsigned char **buffer, unsigned char *buffer_limit)
{
  bool result=true;
  unsigned char *buffer_start=*buffer;
  int i,j,size;
  U61_Event evt;

  for (i=0;i<U61_MAP_MAX_WIDTH && result;++i)
    for (j=0;j<U61_MAP_MAX_HEIGHT && result;++j)
      {
	result=result && squares[i][j].unserialize(buffer,buffer_limit);
      }

  result=result && block.unserialize(buffer,buffer_limit);
  result=result && next_block.unserialize(buffer,buffer_limit);

  auto_events.clear();
  result=result && U61_Serial::unserialize_int(&size,buffer,buffer_limit);
  for (i=0;i<U61_MAP_MAX_AUTO_EVENTS && result;++i)
    {
      result=result && evt.unserialize(buffer,buffer_limit);
      if (i<size)
	{
	  auto_events.push_back(evt);
	}
    }

  request_events.clear();
  result=result && U61_Serial::unserialize_int(&size,buffer,buffer_limit);
  for (i=0;i<U61_MAP_MAX_REQUEST_EVENTS && result;++i)
    {
      result=result && evt.unserialize(buffer,buffer_limit);
      if (i<size)
	{
	  request_events.push_back(evt);
	}
    }

  for (i=0;i<U61_MAP_NB_GLOBAL && result;++i)
    {
      result=result && U61_Serial::unserialize_int(&(global_val[i]),buffer,buffer_limit);
    }

  for (i=0;i<U61_MAP_MAX_CURSE_ID && result;++i)
    {
      result=result && registered_curse[i].unserialize(buffer,buffer_limit);
    }

  result=result && U61_Serial::unserialize_int(&player_id,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&target_id,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&background,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&active,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&silent,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int((int *) &map_time,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&width,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&height,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&block_requested,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&matching,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&curse_delay,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&speed_counter,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&accel_counter,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&speed,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&accel,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&base_speed,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&base_accel,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&score,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&curse_sent,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&curse_received,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int((int *) &start_time,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&match_count,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&anticipation_state,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&preview_state,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&curse_x,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&curse_y,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&curse_state,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&curse_chosen,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&curse_asked,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&curse_id,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&curse_counter,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&nb_antidote,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_short(&last_checksum,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&checksum_requested,buffer,buffer_limit);

  /*
   * We call change background() for background depends on local settings,
   * since it's defined in the theme.
   */
  change_background();

  if ((*buffer) - buffer_start != U61_MAP_SERIALIZED_SIZE)
    {
      U61_LOG_WARNING("Wrong buffer size for map unserialization (diff="<<(buffer_start - (*buffer) + U61_MAP_SERIALIZED_SIZE)<<")!");
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Serializes a map into a byte stream, omitting informations which might
 * interfere with checksum calculus, but still preserving a rather precise
 * image of map state.
 */
bool U61_Map::serialize_for_checksum(unsigned char **buffer, unsigned char *buffer_limit)
{
  bool result=true;
  unsigned char *buffer_start=*buffer;
  int i,j;
  U61_Event evt;
  list<U61_Event> list_evt;

  for (i=0;i<U61_MAP_MAX_WIDTH && result;++i)
    for (j=0;j<U61_MAP_MAX_HEIGHT && result;++j)
      {
	result=result && squares[i][j].serialize(buffer,buffer_limit);
      }

  result=result && block.serialize(buffer,buffer_limit);
  result=result && next_block.serialize(buffer,buffer_limit);

  for (i=0;i<U61_MAP_NB_GLOBAL && result;++i)
    {
      result=result && U61_Serial::serialize_int(global_val[i],buffer,buffer_limit);
    }

  for (i=0;i<U61_MAP_MAX_CURSE_ID && result;++i)
    {
      result=result && registered_curse[i].serialize(buffer,buffer_limit);
    }

  result=result && U61_Serial::serialize_int(player_id,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(target_id,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(active,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(map_time,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(width,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(height,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_delay,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(speed_counter,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(accel_counter,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(speed,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(accel,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(base_speed,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(base_accel,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(score,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_sent,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_received,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(start_time,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_x,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_y,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(curse_state,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(curse_id,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(nb_antidote,buffer,buffer_limit);

  if ((*buffer) - buffer_start != U61_MAP_SERIALIZED_FOR_CHECKSUM_SIZE)
    {
      U61_LOG_WARNING("Wrong buffer size for map serialization in checksum mode (diff="<<(buffer_start - (*buffer) + U61_MAP_SERIALIZED_FOR_CHECKSUM_SIZE)<<")!");
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Starts the checksum process.
 */
void U61_Map::start_checksum()
{
  U61_Event event;

  event.code=U61_EVENT_REQUEST_CHECKSUM;
  event.par=0;

  put_auto_event(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Generates a REQUEST_CHECKSUM event.
 */
void U61_Map::request_checksum()
{
  U61_Event event;
  unsigned char *buf;
  unsigned char *buf_tmp;
  short sig=0;
  U61_Checksum checksum;

  /*
   * We need to test if we're already waiting for a verify checksum event
   * to come, otherwise everything would be messed up for the "last_checksum"
   * value would be overwritten...
   */
  if (!checksum_requested)
    {
      checksum_requested=true;

      event.code=U61_EVENT_VERIFY_CHECKSUM;

      buf=new unsigned char[U61_MAP_SERIALIZED_FOR_CHECKSUM_SIZE];
      if (buf)
	{
	  buf_tmp=buf;
	  this->serialize_for_checksum(&buf_tmp,buf_tmp+U61_MAP_SERIALIZED_FOR_CHECKSUM_SIZE);
	  checksum.calc((char *) buf,U61_MAP_SERIALIZED_FOR_CHECKSUM_SIZE);
	  sig=checksum.get_short_sig();

	  U61_LOG_DEBUG("Generated checksum: "<<sig<<" map: "<<*this);
#ifdef U61_DEF_DEBUG
	  dump_on_log();
#endif

	  delete buf;
	}

      event.par=last_checksum=sig;

      /*
       * Now with this "put_auto_event" trick, only the map on the computer
       * where the player is actually playing will send the event on the
       * network, but everybody will receive it. Since last_checksum is
       * checked _locally_ on every computer, matching last_checksum with
       * the value in the event should reveal inconsistencies.
       */
      put_auto_event(event);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Verifies a checksum.
 */
void U61_Map::verify_checksum(short checksum)
{
  checksum_requested=false;

  U61_LOG_DEBUG("Comparing checksums : "<<checksum<<" / "<<last_checksum);

  if (checksum != last_checksum)
    {
      U61_LOG_WARNING("Wrong checksum for player \""<<name<<"\"!");
    }
}

/*------------------------------------------------------------------------- */
/*
 * Dumps a map to the log, for debugging purpose.
 * We don't use the ostream operator here since log messages are limited
 * in size and we need to split the dump since it's too big.
 */
void U61_Map::dump_on_log()
{
  unsigned char buffer_bin[U61_MAP_SERIALIZED_SIZE];
  char buffer_ascii[U61_MAP_SERIALIZED_SIZE*2+1];
  unsigned char *buffer_tmp;
  char buffer_log[73]; 
  char *buffer_tmp2;

  buffer_tmp=buffer_bin;
  this->serialize(&buffer_tmp,buffer_tmp+U61_MAP_SERIALIZED_SIZE);
  U61_Utils::to_hexascii(buffer_ascii,buffer_bin,U61_MAP_SERIALIZED_SIZE);

  U61_LOG_DEBUG("------8<--------------------------- [begin dump map]")
  for (buffer_tmp2=buffer_ascii;buffer_tmp2<buffer_ascii+sizeof(buffer_ascii);buffer_tmp2+=sizeof(buffer_log)-1)
    {
      U61_MACRO_STRCPY(buffer_log,buffer_tmp2);
      U61_LOG_DEBUG(buffer_log);
    }
  U61_LOG_DEBUG("------8<--------------------------- [end dump map]")
}

/*------------------------------------------------------------------------- */
/*
 * Usefull to print a map using iostreams
 */
ostream &operator <<(ostream &o, U61_Map map)
{
  char square_buf[(U61_MAP_MAX_WIDTH + 1) * U61_MAP_MAX_HEIGHT + 2];
  char square_tmp[2];
  int i,j;
  int c;

  square_buf[0]='\0';
  square_tmp[1]='\0';
  for (j=0;j<map.get_height();++j)    
    {
      for (i=0;i<map.get_width();++i)
	{
	  c=map.get_square_color(i,j);
	  if (c<0)
	    {
	      square_tmp[0]='_';
	    }
	  else
	    {
	      square_tmp[0]='0'+c;
	    }

	  U61_MACRO_STRCAT(square_buf,square_tmp);
	}
      if (j<map.get_height()-1)
	{
	  U61_MACRO_STRCAT(square_buf,"/");
	}
    }

  return o<<"name=\""<<map.get_name()<<"\""
	  <<" / time="<<map.get_time()
	  <<" / score="<<map.get_score()
	  <<" / size="<<map.get_width()<<"x"<<map.get_height()
	  <<" / data="<<square_buf;
}

