/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   menuitem.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: abstract class, parent of all menu items
 *              fondamentaly, a menu item is a line in a menu
 *              this line is made of letters, stored in a array
 *              the menu item will basically handle key presses
 *              and modify
 *              - its display
 *              - some parameter in the config object
 *              special events may open another menu or go back
 *              of course the class provides a "draw" function
 *              and automatically centers itself
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#ifdef U61_DEF_MSVC
#include <minmax.h>
#endif

#include "menuitem.h"
#include "global.h"
#include "letter.h"
#include "log.h"
#include "menu.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * generation of a default menu item
 */ 
U61_MenuItem::U61_MenuItem() 
  : text_displayer(U61_Global::data.font_menu)
{
  /*
   * Using the following line did cause problems,
   * for the constructor "with a string" did not
   * really seem to be called. Weird. I seem to
   * be a C++ idiot anyway 8-)
   */
  //  U61_MenuItem("Default menu item");

  align=U61_TEXTDISPLAYER_ALIGN_CENTER;
  selection=0;
  set_text("Default menu item");
  unselect();
}

/*--------------------------------------------------------------------------*/
/* 
 * generation of a default menu item
 */ 
U61_MenuItem::U61_MenuItem(char *str)
  : text_displayer(U61_Global::data.font_menu)
{
  align=U61_TEXTDISPLAYER_ALIGN_CENTER;
  selection=0;
  set_text(str);
  unselect();
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of the menu-item
 */ 
U61_MenuItem::~U61_MenuItem()
{
  check_value();
}

/*--------------------------------------------------------------------------*/
/*
 * treats a key press on a menu item
 */
bool U61_MenuItem::action(int key)
{
  U61_LOG_DEBUG("menu item action, key="<<key) 
    return false;
}

/*--------------------------------------------------------------------------*/
/*
 * updates the text so that it is up to date
 */
void U61_MenuItem::update()
{

}

/*--------------------------------------------------------------------------*/
/*
 * draws a menu item
 */
void U61_MenuItem::draw(int y) 
{
  /*
   * The update is forced here because blinking cursors require it,
   * and in a general manner, nasry bugs can happen if there are not
   * enough updates. Should not be so slow however...
   */
  update();

  text_displayer.draw(0,y,
		      align,
		      U61_Global::data.screen_w,
		      U61_TEXTDISPLAYER_MAX_H);
}

/*--------------------------------------------------------------------------*/
/*
 * tests if the mouse is within the menuitem drawable area
 */
void U61_MenuItem::test_mouse(int y, 
			      int mouse_x,
			      int mouse_y,
			      bool *mouse_in, 
			      int *mouse_char)
{
  text_displayer.test_mouse(0,y,
			    align,
			    U61_Global::data.screen_w,
			    U61_TEXTDISPLAYER_MAX_H,
			    mouse_x,
			    mouse_y,
			    mouse_in,
			    mouse_char);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the height of the menu item
 */
int U61_MenuItem::get_height()
{
  int height=0;

  height=text_displayer.get_height(U61_Global::data.screen_w);

  return height;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the width of the menu item
 */
int U61_MenuItem::get_width()
{
  int width=0;

  width=text_displayer.get_width(U61_Global::data.screen_w);

  return width;
}

/*--------------------------------------------------------------------------*/
/*
 * selects a letter in the current menu item 
 */
void U61_MenuItem::select_char(int i)
{
  text_displayer.enable_dance(i);

  selected=true;
}

/*--------------------------------------------------------------------------*/
/*
 * make a letter in the current menu item blink 
 */
void U61_MenuItem::blink_char(int i)
{
  text_displayer.enable_blink(i);
}

/*--------------------------------------------------------------------------*/
/*
 * selects the current menu item, starting at a given letter
 */
void U61_MenuItem::select_right(int start)
{
  int i;
  int nb_letter;
    
  nb_letter=text_displayer.get_text_size();
  for (i=start;i<nb_letter;++i)
    {
      select_char(i);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * selects the current menu item
 */
void U61_MenuItem::select()
{
  if (selectable())
    {
      select(selection);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * selects the current menu item
 */
void U61_MenuItem::select(int pos)
{
  if (selectable())
    {
      selection=pos;
      
      select_right(0);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * unselects the current menu item
 */
void U61_MenuItem::unselect()
{
  if (!check_value())
    {
      update();
    }

  unselect_ex();    
}

/*--------------------------------------------------------------------------*/
/*
 * unselects the current menu item
 */
void U61_MenuItem::unselect_ex()
{
  int i;
  int nb_letter;

  nb_letter=text_displayer.get_text_size();
  for (i=0;i<nb_letter;++i)
    {
      text_displayer.disable_dance(i);
      text_displayer.disable_blink(i);
    }

  selected=false;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the text of the menu item
 */
void U61_MenuItem::set_text(char *str)
{
  char old[U61_CONST_STRING_SIZE];

  /*
   * Before we empty the queue of letters and refill it with some new,
   * we check up that this is needed, ie the strings are not the same
   * if they are the same, it's no use to bother, and moreover it can
   * become a bug since the dancing letters don't behave correctly
   */
  put_text(old,sizeof(old)-1);
  if (strcmp(str,old)!=0)
    {
      text_displayer.set_text(str);

      /*
       * This is not useless, it just means if the item is selected, we
       * have to re-apply the effects of the selection since emptying
       * the queue has cancelled everything...
       */
      if (selected)
	{
	  select();
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * puts the text in a string buffer. buf must contain at least max+1
 * characters to store the trailing '\0'.
 */
void U61_MenuItem::put_text(char *buf, int max)
{
  text_displayer.put_text(buf,max);
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the alignement of the text drawn
 */
void U61_MenuItem::set_align(int al)
{
  align=al;
}

/*--------------------------------------------------------------------------*/
/*
 * true if the menuitem must handle all keys, including vertical arrows
 */
bool U61_MenuItem::is_modal()
{ 
  return false;
}

/*--------------------------------------------------------------------------*/
/*
 * Function which can override the main validate function of the menu.
 * This way the menu item can directly return a pointer on a new menu
 * and make it the current menu. We return NULL if we want the standard
 * menu function to be called.
 */
U61_Menu *U61_MenuItem::validate()
{ 
  return NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the item can be selected
 */
bool U61_MenuItem::selectable()
{
  return true;
}

/*--------------------------------------------------------------------------*/
/*
 * Function called to check if the value within an "edit-like" item are
 * correct, and to change them if needed. Basically called when 
 * de-selecting and deleting the object. Returns true if the value was
 * correct so remain unchanged, and false if it was incorrect and had
 * to be changed.
 */
bool U61_MenuItem::check_value()
{
  /*
   * By default we do nothing for we have no idea of which value
   * has to be checked
   */
  return true;
}

