/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   mousemonitor.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: This class is responsible for listening to mouse events.
 *              It gets events like move, button press, and maintains 
 *              informations about the mouse. One of its main interest is
 *              to provide a "has_moved" function.
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "mousemonitor.h"
#include "log.h"
#include "macro.h"
#include "global.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 methods
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * Creates a default input monitor
 */ 
U61_MouseMonitor::U61_MouseMonitor()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * Binds the keyboard to the input monitor
 * Could not be done in the constructor since this object is global/static
 */ 
void U61_MouseMonitor::init()
{
  slot_button_press = CL_Mouse::sig_button_press().connect
    (&U61_MouseMonitor::on_button_press);
  slot_button_release = CL_Mouse::sig_button_release().connect
    (&U61_MouseMonitor::on_button_release);
  slot_move = CL_Mouse::sig_move().connect
    (&U61_MouseMonitor::on_move);
}

/*--------------------------------------------------------------------------*/
/* 
 * Destroys the input monitor
 */ 
U61_MouseMonitor::~U61_MouseMonitor()
{
}

/*--------------------------------------------------------------------------*/
/* 
 * Resets all the keys.
 */ 
void U61_MouseMonitor::reset()
{
  x = -1;
  y = -1;
  moved_state = false;
  left_pressed_state = false;
  right_pressed_state = false;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the x position
 */ 
int U61_MouseMonitor::get_x()
{
  if (x<0)
    {
      x=CL_Mouse::get_x();
    }

  return x;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the y position
 */ 
int U61_MouseMonitor::get_y()
{
  if (y<0)
    {
      y=CL_Mouse::get_y();
    }

  return y;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns true is the mouse has been moved since the last call to
 * "acknowledge_move". Usefull to handle the menus.
 */ 
bool U61_MouseMonitor::moved(bool acknowledge)
{
  bool result;

  result=moved_state;

  if (acknowledge)
    {
      acknowledge_move();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns true if the left button has been pressed since the last call
 * "acknowledge_left_press".
 */ 
bool U61_MouseMonitor::left_pressed(bool acknowledge)
{
  bool result;

  result=left_pressed_state;

  if (acknowledge)
    {
      acknowledge_left_press();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns true if the right button has been pressed since the last call
 * "acknowledge_right_press".
 */ 
bool U61_MouseMonitor::right_pressed(bool acknowledge)
{
  bool result;

  result=right_pressed_state;

  if (acknowledge)
    {
      acknowledge_right_press();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Acknowledges the fact that the mouse has moved, and resets the
 * "has_moved" value.
 */ 
void U61_MouseMonitor::acknowledge_move()
{
  moved_state=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * Acknowledges the fact that the left button has been oressed, and resets the
 * "left_pressed" value.
 */ 
void U61_MouseMonitor::acknowledge_left_press()
{
  left_pressed_state=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * Acknowledges the fact that the right button has been oressed, and resets the
 * "right_pressed" value.
 */ 
void U61_MouseMonitor::acknowledge_right_press()
{
  right_pressed_state=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when the mouse has moved
 */ 
void U61_MouseMonitor::on_move(int x, int y)
{
  U61_MouseMonitor *global_monitor;

  global_monitor=&(U61_Global::mouse_monitor);

  U61_LOG_DEBUG("mouse move message x="<<x<<", y="<<y);
  global_monitor->x = x;
  global_monitor->y = y;
  global_monitor->moved_state = true;
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a button is pressed down
 */ 
void U61_MouseMonitor::on_button_press(const CL_Key &key)
{
  U61_MouseMonitor *global_monitor;

  global_monitor=&(U61_Global::mouse_monitor);

  U61_LOG_DEBUG("mouse press message id="<<key.id);
  switch (key.id)
    {
    case CL_MOUSE_LEFTBUTTON:
      U61_LOG_DEBUG("mouse left button pressed");
      global_monitor->left_pressed_state = true;
      break;
    case CL_MOUSE_RIGHTBUTTON:
      U61_LOG_DEBUG("mouse right button pressed");
      global_monitor->right_pressed_state = true;
      break;
    case CL_MOUSE_WHEELUP:
      U61_LOG_DEBUG("mouse wheel up");
      break;
    case CL_MOUSE_WHEELDOWN:
      U61_LOG_DEBUG("mouse wheel down");
      break;
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called when a button released
 */ 
void U61_MouseMonitor::on_button_release(const CL_Key &key)
{
  U61_MouseMonitor *global_monitor;

  global_monitor=&(U61_Global::mouse_monitor);

  U61_LOG_DEBUG("mouse release message id="<<key.id);
  switch (key.id)
    {
    case CL_MOUSE_LEFTBUTTON:
      break;
    case CL_MOUSE_RIGHTBUTTON:
      break;
    case CL_MOUSE_WHEELUP:
      break;
    case CL_MOUSE_WHEELDOWN:
      break;
    }
}







