/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   player.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: a player is a bundle of
 *              - an event source
 *              - an event copy
 *              - a "sure" map
 *              - a "last" map
 *              - a API to handle all that...
 *              it's fundamental to understand the difference between the
 *              last map and the sure map. 
 *              the sure map is a map freezed at the time of the latest
 *              event received. since events are ordered by time, we can
 *              assume this is the right map at this givent instant
 *              the last map is an anticipated map. from the suremap state,
 *              it simulates game cycles so that the map is "as it would have
 *              been now (which may be later than the latest event) if no
 *              event occured". This way, in a network game, map never freeze
 *              waiting for network events, and this gives an impression of
 *              speed even with poor bandwidth
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <string.h>
#include <zlib.h>

#include "player.h"
#include "time.h"
#include "localsource.h"
#include "localcopy.h"
#include "networksource.h"
#include "networkcopy.h"
#include "global.h"
#include "log.h"
#include "sound.h"
#include "macro.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

/*
 * Delay after which one considers that a checksum is "missing".
 */
#define U61_PLAYER_MAX_DELAY_BETWEEN_CHECKSUM (U61_MAP_DELAY_BETWEEN_CHECKSUM*3+1000)

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default player
 */ 
U61_Player::U61_Player()
{
  /*
   * Note: we don't use reset for it's not possible to
   * call map::reset() until all the map backgrounds have
   * been loaded, and the U61_Player objects are created
   * as global variables, long before the backgrounds are loaded...
   */
  id=0;
  available=false;
  local=true;
  history_done=true;

  source=NULL;
  copy=NULL;

  anticipation=NULL;
  key_start_stop=NULL;

  ready_state=U61_PLAYER_STATE_NOT_READY;
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a player
 */ 
U61_Player::~U61_Player()
{

}

/*--------------------------------------------------------------------------*/
/*
 * resets a player to default
 */
void U61_Player::reset()
{
  id=0;
  available=false;
  local=true;
  history_done=true;

  if (source!=NULL)
    {
      delete source;
      source=NULL;
    }
  if (copy!=NULL)
    {
      delete copy;
      copy=NULL;
    }

  last.reset();
  sure.reset();
  history.reset();
  history_time=U61_TIME_START;
  U61_MACRO_MEMSET0(history_zlib_buffer);
  history_zlib_buffer_size=0;
  history_zlib_buffer_cpt=0;

  last_event_time=U61_TIME_START;
  last_checksum_time=U61_TIME_START;

  anticipation=NULL;
  key_start_stop=NULL;

  ready_state=U61_PLAYER_STATE_NOT_READY;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the id of a player
 */
void U61_Player::set_id(int player_id)
{
  id=player_id;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the id of a player
 */
int U61_Player::get_id()
{
  return id;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the id of the target of a player
 */
int U61_Player::get_target_id()
{
  return last.get_target_id();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of a player
 */
char *U61_Player::get_name()
{
  return last.get_name();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the score of a player
 */
int U61_Player::get_score()
{
  return last.get_score();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next block of a player
 */
U61_Block U61_Player::get_next_block()
{
  return last.get_next_block();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of persistent curses activated
 */
int U61_Player::get_nb_curse(bool good)
{
  return last.get_nb_curse(good);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of antidotes available
 */
int U61_Player::get_nb_antidote()
{
  return last.get_nb_antidote();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the current curse
 */
char *U61_Player::get_curse_name()
{
  return last.get_curse_name();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the index of the background to be displayed
 */
int U61_Player::get_background()
{
  int bg=-1;

  /*
   * When the player is not active we return -1, for the background
   * does not have to be drawn, and this way we can notice the layout
   * manager that the background has changed, ie the background image
   * has to be redrawn.
   */
  if (last.is_active())
    {
      bg=last.get_background();
    }

  return bg;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the player as a local player
 * Note that even if the game is a networked game, local players are
 * initialized with this function
 */
void U61_Player::init_local(U61_Input *input,
			    U61_Input *chat_input,
                            U61_Dispatcher *d,
			    int id,
			    bool prep_hist,
			    int time,
			    U61_PlayerConfig *conf,
			    bool ready)
{
  U61_LocalSource *src; 
  U61_LocalCopy *cpy;

  reset();

  local=true;

  set_id(id);
  prepare_history=prep_hist;

  sure.init(id,
	    U61_Global::game.initial_speed,
	    U61_Global::game.acceleration,
	    U61_Global::game.curse_delay);
  copy_sure_to_last();

  anticipation=&(conf->anticipation);
  key_start_stop=&(conf->key_start_stop);

  src=new U61_LocalSource(input,chat_input,&sure,d,id,conf->name);

  src->send_nothing(time);
  src->send_name(time);
  if (ready)
    {
      src->send_ready(time);
    }

  cpy=new U61_LocalCopy(id,d);

  source=src;
  copy=cpy;

  available=true;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the player as a network player
 */
void U61_Player::init_network(U61_Dispatcher *d,
                              int id,
			      bool prep_hist)
{
  U61_NetworkSource *src; 
  U61_NetworkCopy *cpy;

  reset();

  local=false;;

  set_id(id);
  prepare_history=prep_hist;

  sure.init(id,
	    U61_Global::game.initial_speed,
	    U61_Global::game.acceleration,
	    U61_Global::game.curse_delay);
  sure.mute();
  copy_sure_to_last();

  anticipation=&(U61_Global::config.network_anticipation);
  key_start_stop=NULL;

  src=new U61_NetworkSource(d,&sure,id);
  cpy=new U61_NetworkCopy(id,d);

  source=src;
  copy=cpy;

  U61_LOG_DEBUG("Starting network player "<<id);

  available=true;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the player is available
 */
bool U61_Player::is_available()
{
  return available;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the player is active
 */
bool U61_Player::is_active()
{
  return last.is_active();;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the player is a local player
 */
bool U61_Player::is_local()
{
  return local;
}

/*--------------------------------------------------------------------------*/
/*
 * this function pumps the events for a player, ie it:
 * read the events
 * interprets them
 * makes a copy of them if necessary
 */
void U61_Player::pump_events()
{
  bool changed=false;
  U61_Event evt;
  bool kill_needed=false;
  bool history_copied=false;

  source->poll();

  while((!source->empty()) && 
        ((source->check().time))<=U61_Time::for_logic())
    {
      changed=true;
      evt=source->get();

      /*
       * In case we have an event which is older
       * than the latest processed event, we copy
       * the "sure" map to the "history" map which will
       * be used when sending info to network clients
       */
      if (prepare_history && last_event_time<evt.time && !history_copied)
	{
	  copy_sure_to_history();
	  /*
	   * We flag history_copied as true so that we do not copy
	   * sure to history for each event...
	   */
	  history_copied=true;
	}
      last_event_time=evt.time;

      switch (evt.code)
	{
	case U61_EVENT_KILL:
	  kill_needed=true;
	  break;
        case U61_EVENT_LOOSE:
	  register_score();
	  break;
        case U61_EVENT_HISTORY_BEGIN:
	  history_done=false;
	  break;
        case U61_EVENT_HISTORY_END:
	  history_done=true;
	  break;
	case U61_EVENT_READY:
	  if (ready_state!=U61_PLAYER_STATE_READY_PLAYING)
	    {
	      ready_state=U61_PLAYER_STATE_READY_WAITING;
	    }
	  break;
	case U61_EVENT_READY_ALL:
	  if (ready_state!=U61_PLAYER_STATE_READY_PLAYING)
	    {
	      U61_Global::game.notify_session_start();

	      ready_state=U61_PLAYER_STATE_READY_PLAYING;
	      if (is_local())
		{
		  ((U61_LocalSource *) source)->send_start_stop(evt.time);
		  ((U61_LocalSource *) source)->send_name(evt.time);
		}
	    }
	  break;
	case U61_EVENT_CHAT_LETTER:
	  U61_Global::layout.put_chat_letter(evt);
	  break;
	case U61_EVENT_VERIFY_CHECKSUM:
	  last_checksum_time=evt.time;
	  break;
	}

      /*
       * It's important to put this code after the previous "switch"
       * or else scores are not correctly handled
       */
      switch (evt.code)
	{
	case U61_EVENT_SERIAL_BEGIN:
	  if (!serial_begin(evt))
	    {
	      kill_needed=true;
	    }
	  break;
	case U61_EVENT_SERIAL_DATA:
	  if (!serial_data(evt))
	    {
	      kill_needed=true;
	    }
	  break;
	case U61_EVENT_SERIAL_END:
	  if (!serial_end(evt))
	    {
	      kill_needed=true;
	    }
	  break;
	default:
	  handle_event(&evt);
	  copy->put(evt);
	}
    }

  while (sure.exists_request_event())
    {
      changed=true;
      evt=sure.get_request_event();
      copy->put(evt);
    }

  if (changed)
    {
      copy_sure_to_last();
    }

  copy->flush();

  check_missing_checksum();

  if (kill_needed)
    {
      kill();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * pumps an event
 */
void U61_Player::handle_event(U61_Event *evt)
{
  if (evt->code==U61_EVENT_START_STOP)
    {
      if (sure.is_active())
	{
	  stop();
	}
      else
	{
	  start(evt->time);
	}
    }
  else
    {
      sure.handle_event(evt);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * called to start a player, ie enter him in the game
 */
void U61_Player::start(int time)
{
  sure.set_active(time);
  copy_sure_to_last();

  U61_LOG_DEBUG("Player "<<id<<" started");
}

/*--------------------------------------------------------------------------*/
/*
 * called to stop a player, ie remove him from the game
 */
void U61_Player::stop()
{
  register_score();
  sure.set_inactive();
  copy_sure_to_last();
  U61_LOG_DEBUG("Player "<<id<<" stopped");
}

/*--------------------------------------------------------------------------*/
/*
 * called to kill a player. This is stronger than a stop, it should
 * happen when a remote network players quits the game.
 * In this case we kill the player to free the player slot.
 */
void U61_Player::kill()
{
  register_score();
  reset();
  //available=false;
  U61_LOG_DEBUG("Player "<<id<<" killed");
}

/*--------------------------------------------------------------------------*/
/*
 * draws the map of the player
 */
void U61_Player::draw(int x,int y,int size)
{
  U61_TextDisplayer message(U61_Global::data.font_info[size]);
  bool draw_message = false;
  char buffer[U61_CONST_STRING_SIZE];

  if (history_done)
    {
      last.anticipate(U61_Time::for_event());
      last.draw(x,y,size,(anticipation!=NULL) && *anticipation);
    }

  switch (ready_state)
    {
    case U61_PLAYER_STATE_NOT_READY:
      if (local && key_start_stop)
	{
	  U61_MACRO_SPRINTF1(buffer,
			     "Press %s when ready",
			     U61_Global::input_monitor.to_label(*key_start_stop));
	}
      else
	{
	  U61_MACRO_STRCPY(buffer,
			   "Not ready...");
	}
      message.set_text(buffer);	
      draw_message = true;
      break;
    case U61_PLAYER_STATE_READY_WAITING:
      message.set_text("Ready!");
      draw_message = true;
      break;
    case U61_PLAYER_STATE_READY_PLAYING:
      break;
    default:
      break;
    }

  if (draw_message)
    {
      int message_y,message_w;

      message_y = y+(last.get_draw_height(size)-message.get_height())/2;
      message_w = last.get_draw_width(size);

      message.draw(x,message_y,
		   U61_TEXTDISPLAYER_ALIGN_CENTER,
		   message_w);			  
    }
}

/*--------------------------------------------------------------------------*/
/*
 * copies the sure map to the last map
 */
void U61_Player::copy_sure_to_last()
{
  last=sure;
  last.mute();
}

/*--------------------------------------------------------------------------*/
/*
 * copies the sure map to the history map
 */
void U61_Player::copy_sure_to_history()
{
  history_time=last_event_time;

  history=sure;
}

/*--------------------------------------------------------------------------*/
/*
 * copies the sure map to the last map
 */
void U61_Player::register_score()
{
  if (sure.is_active() && sure.get_elapsed_time()>0)
    {
      U61_Global::score.register_game(sure.get_name(),
				      local,
				      sure.get_score(),
				      sure.get_elapsed_time(),
				      sure.get_curse_sent(),
				      sure.get_curse_received());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the "history" map, which is usefull for serializing the map
 * into events within the U61_History object.
 */
U61_Map *U61_Player::get_history()
{
  return &history;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the time when the "history" map has been built.
 */
unsigned int U61_Player::get_history_time()
{
  return history_time;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the time when the "history" map has been built.
 */
bool U61_Player::serial_begin(U61_Event evt)
{
  bool ok=false;

  if (evt.par>U61_MAP_COMPRESSED_SIZE)
    {
      U61_LOG_ERROR("Network header reports that compressed map is too big ("<<evt.par<<" vs "<<U61_MAP_COMPRESSED_SIZE<<")!");
    }
  else
    {
      U61_MACRO_MEMSET0(history_zlib_buffer);
      history_zlib_buffer_size=evt.par;
      history_zlib_buffer_cpt=0;

      ok=true;
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the time when the "history" map has been built.
 */
bool U61_Player::serial_data(U61_Event evt)
{
  bool ok=false;
  unsigned char *buffer;

  if (history_zlib_buffer_cpt>=((history_zlib_buffer_size-1)/(int) sizeof(short))+1)
    {
      U61_LOG_ERROR("Too many network packets for compressed map ("<<(history_zlib_buffer_cpt+1)<<" vs "<<(((history_zlib_buffer_size-1)/sizeof(short))+1)<<")!");
    }
  else
    {
      buffer=history_zlib_buffer+history_zlib_buffer_cpt*sizeof(short);
      /*
       * We use serialization functions for the conversion char[2]->short
       * depends on machine endianess.
       */
      U61_Serial::serialize_short(evt.par,&buffer,buffer+sizeof(short));     

      history_zlib_buffer_cpt++;

      ok=true;
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the time when the "history" map has been built.
 */
bool U61_Player::serial_end(U61_Event evt)
{
  bool ok=false;
  unsigned char *buf;
  unsigned char *buf_tmp;
  unsigned long buf_size;
  int zerr;

  if (history_zlib_buffer_cpt!=((history_zlib_buffer_size-1)/(int) sizeof(short))+1)
    {
      U61_LOG_ERROR("Wrong network packets number for compressed map ("<<history_zlib_buffer_cpt<<" vs "<<(((history_zlib_buffer_size-1)/sizeof(short))+1)<<")!");

    }
  else
    {
      U61_LOG_DEBUG("received "<<history_zlib_buffer_size<<" bytes serialized map, going to uncompress it...");

      buf_size=U61_MAP_SERIALIZED_SIZE;
      buf=new unsigned char[buf_size];
      if (buf)
	{
	  zerr=uncompress(buf,&buf_size,history_zlib_buffer,history_zlib_buffer_size);

	  if (zerr!=Z_OK)
	    {
	      U61_LOG_ERROR("Unable to uncompress serialized map (zlib error "<<zerr<<":\""<<zError(zerr)<<"\")!");
	    }
	  else
	    {
	      /*
	       * Now we're going to restore the map for good, since we've
	       * collected all the network packets.
	       */ 
	      buf_tmp=buf;	      
	      if (!sure.unserialize(&buf_tmp,buf_tmp+U61_MAP_SERIALIZED_SIZE))
		{
		  U61_LOG_ERROR("Serialized map is not correct!");
		}
	      else
		{
		  /*
		   * Everything's OK, map should have been restored
		   */
		  ok=true;
		}
	    }

	  delete buf;
	}
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if there's been a checksum sent recently.
 */
void U61_Player::check_missing_checksum()
{
  if (last_checksum_time<=U61_TIME_START || !is_active())
    {
      last_checksum_time=last_event_time;
    }

  if (last_checksum_time+U61_PLAYER_MAX_DELAY_BETWEEN_CHECKSUM<last_event_time)
    {
      last_checksum_time=last_event_time;
      U61_LOG_WARNING("Missing checksum for player \""<<get_name()<<"\"!")
    }
}
