/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   protocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: contains the network code which is needed to initialize the
 *              maps on the client. this can be quite a complex process
 *              since we can not interrupt the game when someone logs in.
 *              and the amount of data to transmit might be high 
 *              (especially if the lua scripts are big)
 *              So we use lots of small messages and this forms what
 *              I call the "login protocol" and is maybe the most complex
 *              thing in U61...
 */


/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include "protocol.h"
#include "packet.h"
#include "version.h"
#include "const.h"
#include "log.h"
#include "macro.h"

/*---------------------------------------------------------------------------
  variants
  ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a protocol
 */
U61_Protocol::U61_Protocol(int p)
{
  port=p;

  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * deletes a protocol
 */
U61_Protocol::~U61_Protocol()
{

}

/*--------------------------------------------------------------------------*/
/*
 * Checks if all the connections are OK
 */
void U61_Protocol::check_connection()
{

}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet (to be defined in a sub-class)
 */
int U61_Protocol::send_next_packet()
{
  stage=U61_PROTOCOL_BEGIN;
  return U61_PROTOCOL_LATER;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet (to be defined in a sub-class)
 */
int U61_Protocol::recv_next_packet()
{
  stage=U61_PROTOCOL_BEGIN;
  return U61_PROTOCOL_LATER;
}

/*--------------------------------------------------------------------------*/
/*
 * sends all the pending packets
 */
int U61_Protocol::send_all()
{
  int result=U61_PROTOCOL_LATER;

  if (!failed)
    {  
      while ((result=send_next_packet())==U61_PROTOCOL_DONE);
      if (result==U61_PROTOCOL_FAILED)
	{
	  failed=true;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives all the pending packets
 */
int U61_Protocol::recv_all()
{
  int result=U61_PROTOCOL_LATER;

  if (!failed)
    {
      while ((result=recv_next_packet())==U61_PROTOCOL_DONE);
      if (result==U61_PROTOCOL_FAILED)
	{
	  failed=true;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * resets the protocol 
 */
void U61_Protocol::reset()
{
  stage=U61_PROTOCOL_BEGIN;
  step=0;
  U61_MACRO_STRCPY(diagnostic,"Unknown problem");
  connecting=false;
  failed=false;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true is the protocol is ended and successful
 */
bool U61_Protocol::done()
{
  return (stage==U61_PROTOCOL_END);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the current stage of the protocol (the higher, the more advanded
 */
int U61_Protocol::get_stage()
{
  return stage;
}

/*--------------------------------------------------------------------------*/
/*
 * sends the ID for game authentification
 * this is in theory useless since ClanLib does this checking but well,
 * I just enjoy checking 8-)
 */
int U61_Protocol::send_program(U61_Connection *conn)
{
  int result=U61_PROTOCOL_LATER;

  U61_LOG_DEBUG("Sending program");

  if (conn!=NULL)
    {
      if (conn->send(U61_CONST_PROGRAM,strlen(U61_CONST_PROGRAM)))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the game ID (goes with send_program...)
 */
int U61_Protocol::recv_program(U61_Connection *conn)
{
  int result=U61_PROTOCOL_LATER;
  char buffer[U61_CONST_STRING_SIZE];

  U61_LOG_DEBUG("Receiving program");

  if (conn!=NULL)
    {
      if (conn->peek_str())
	{
	  if (conn->recv(buffer,sizeof(buffer)-1))
	    {
	      if (strcmp(U61_CONST_PROGRAM,buffer)==0)
		{
		  result=U61_PROTOCOL_DONE;
		  U61_LOG_DEBUG("Received program "<<buffer<<"");
		}
	      else
		{
		  result=U61_PROTOCOL_FAILED;
		  U61_MACRO_SPRINTF2(diagnostic,
				     "Wrong program, expecting \"%s\" and received \"%s\"",
				     U61_CONST_PROGRAM,
				     buffer);
		}
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the game version. We need to check if the exes are the same on
 * all the machines, for game behavior might be different. Of course
 * This is easy to hack but we assume that "cracking" u61 is not so
 * thrilling...
 */
int U61_Protocol::send_version(U61_Connection *conn)
{
  int result=U61_PROTOCOL_LATER;
  char version[U61_CONST_STRING_SIZE];

  U61_LOG_DEBUG("Sending version");

  if (conn!=NULL)
    {
      U61_Version::get_version_text(version);

      if (conn->send(version,strlen(version)))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the game verison (goes with send_version...)
 */
int U61_Protocol::recv_version(U61_Connection *conn)
{
  int result=U61_PROTOCOL_LATER;
  char version[U61_CONST_STRING_SIZE];
  char buffer[U61_CONST_STRING_SIZE];

  U61_LOG_DEBUG("Receiving version");

  if (conn!=NULL)
    {
      U61_Version::get_version_text(version);
      if (conn->peek_str())
	{
	  if (conn->recv(buffer,sizeof(buffer)-1))
	    {
	      if (strcmp(version,buffer)==0)
		{
		  result=U61_PROTOCOL_DONE;
		  U61_LOG_DEBUG("Received version "<<buffer<<"");
		}
	      else
		{
		  result=U61_PROTOCOL_FAILED;
		  U61_MACRO_SPRINTF2(diagnostic,
				     "Wrong version, expecting \"%s\" and received \"%s\"",
				     version,
				     buffer);
		}
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * sends a ready "message" before the game starts
 */
int U61_Protocol::send_ready(U61_Connection *conn)
{
  int result=U61_PROTOCOL_LATER;

  U61_LOG_DEBUG("Sending ready message");

  if (conn!=NULL)
    {
      if (conn->send(U61_PROTOCOL_READY_MESSAGE,
		     strlen(U61_PROTOCOL_READY_MESSAGE)))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives a ready "message" before starting the game
 */
int U61_Protocol::recv_ready(U61_Connection *conn)
{
  int result=U61_PROTOCOL_LATER;
  char buffer[U61_CONST_STRING_SIZE];

  U61_LOG_DEBUG("Receiving ready message");

  if (conn!=NULL)
    {
      if (conn->peek_str())
	{
	  if (conn->recv(buffer,sizeof(buffer)-1))
	    {
	      if (strcmp(U61_PROTOCOL_READY_MESSAGE,buffer)==0)
		{
		  result=U61_PROTOCOL_DONE;
		  U61_LOG_DEBUG("Received ID "<<buffer<<"");
		}
	      else
		{
		  result=U61_PROTOCOL_FAILED;
		  U61_MACRO_SPRINTF2(diagnostic,
				     "Wrong message, expecting \"%s\" and received \"%s\"",
				     U61_PROTOCOL_READY_MESSAGE,
				     buffer);
		}
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the last error message
 */
char *U61_Protocol::get_diagnostic()
{
  diagnostic[sizeof(diagnostic)-1]='\0';

  return diagnostic;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns TRUE if the protocol is in a "connecting" stage
 */
bool U61_Protocol::is_connecting()
{
  return connecting;
}


