/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   score.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: this class holds global scores during a game. It can also
 *              dump the score on the disk and do a bunch of score related
 *              calculus.
 *              Basically the global scores are updated each time a game ends.
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "score.h"
#include "log.h"
#include "time.h"
#include "platform.h"
#include "log.h"
#include "macro.h"
#include "data.h"

/*--------------------------------------------------------------------------*/
/*
 * Constructor of the score class
 */
U61_Score::U61_Score()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * Destructor of the score class
 */
U61_Score::~U61_Score()
{

}

/*--------------------------------------------------------------------------*/
/*
 * Resets all the items of a score
 */
void U61_Score::reset()
{
  local.clear();
  remote.clear();
}

/*--------------------------------------------------------------------------*/
/*
 * Acknowledges the results of a game.
 */
void U61_Score::register_game(char *name, bool loc,
			      int score,
			      unsigned int time,
			      int curse_sent,
			      int curse_received)
{
  bool found;
  int size;
  int i;
  U61_ScoreLine new_line;
  char *local_str="";
  char *local_chr="";
  char buffer_time[U61_CONST_STRING_SIZE];
  char buffer_score[U61_CONST_STRING_SIZE];

  U61_LOG_DEBUG("registering "<<name<<" in score table");

  found=false;
  time/=U61_TIME_ONE_SECOND;

  if (loc)
    {
      local_str="Local player";
      local_chr="l";
      size=local.size();
      for (i=0;i<size && !found;++i)
	{
	  if (strcmp(name,local[i].get_name())==0)
	    {
	      found=true;
	      local[i].register_game(score,time,curse_sent,curse_received);
	    }
	}
      if (!found)
	{
	  new_line.set_name(name);
	  new_line.register_game(score,time,curse_sent,curse_received);
	  local.push_back(new_line);
	}
    }
  else
    {
      local_str="Remote player";
      local_chr="r";
      size=remote.size();
      for (i=0;i<size && !found;++i)
	{
	  if (strcmp(name,remote[i].get_name())==0)
	    {
	      found=true;
	      remote[i].register_game(score,time,curse_sent,curse_received);
	    }
	}
      if (!found)
	{
	  new_line.set_name(name);
	  new_line.register_game(score,time,curse_sent,curse_received);
	  remote.push_back(new_line);
	}
    }

  U61_MACRO_SPRINTF3(buffer_time,"%02d:%02d:%02d",time/3600,(time/60)%60,time%60);
  U61_LOG_MESSAGE(local_str
		  <<" \""<<name<<"\""
		  <<" score="<<score
		  <<" time="<<buffer_time
		  <<" sent="<<curse_sent
		  <<" received="<<curse_received
		  <<".");
  U61_MACRO_SPRINTF6(buffer_score,"%1s ; %-6s ; %8d ; %8s ; %4d ; %4d",
		      local_chr,name,score,buffer_time,curse_sent,curse_received);
  U61_Platform::dump_score(buffer_score);
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the number of different players who have played locally
 */
int U61_Score::get_local_size()
{
  return local.size();
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the information concerning a local player
 */
void U61_Score::get_local_info(int i,char **name,int *nb,int *sco,
			       unsigned int *h,
			       unsigned int *m,
			       unsigned int *s)
{
  *name=local[i].get_name();
  *nb=local[i].get_nb_games();
  *sco=local[i].get_total_score();
  *h=local[i].get_total_time()/3600;
  *m=(local[i].get_total_time()/60)%60;
  *s=local[i].get_total_time()%60;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the name and score of the player who has the highest score
 */
void U61_Score::get_best_max_score(char **name,int *val)
{
  int i;
  int n;

  *val=0;
  *name="";

  n=local.size();
  for (i=0;i<n;++i)
    {
      if (local[i].get_max_score()>*val)
	{
	  *val=local[i].get_max_score();
	  *name=local[i].get_name();
	}
    }
  n=remote.size();
  for (i=0;i<n;++i)
    {
      if (remote[i].get_max_score()>*val)
	{
	  *val=remote[i].get_max_score();
	  *name=remote[i].get_name();
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the name and avg score of the player who has the best avg score
 */
void U61_Score::get_best_avg_score(char **name,int *val)
{
  int i;
  int n;

  *val=0;
  *name="";

  n=local.size();
  for (i=0;i<n;++i)
    {
      if (local[i].get_avg_score()>*val)
	{
	  *val=local[i].get_avg_score();
	  *name=local[i].get_name();
	}
    }
  n=remote.size();
  for (i=0;i<n;++i)
    {
      if (remote[i].get_avg_score()>*val)
	{
	  *val=remote[i].get_avg_score();
	  *name=remote[i].get_name();
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the name and time of the player who has played the longest game
 */
void U61_Score::get_best_max_time(char **name,unsigned int *h,
				  unsigned int *m,
				  unsigned int *s)
{
  int i;
  int n;
  unsigned int time;

  time=0;
  *name="";

  n=local.size();
  for (i=0;i<n;++i)
    {
      if (local[i].get_max_time()>time)
	{
	  time=local[i].get_max_time();
	  *name=local[i].get_name();
	}
    }
  n=remote.size();
  for (i=0;i<n;++i)
    {
      if (remote[i].get_max_time()>time)
	{
	  time=remote[i].get_max_time();
	  *name=remote[i].get_name();
	}
    }
  *h=time/3600;
  *m=(time/60)%60;
  *s=time%60;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the name and average time of the player who has the best avg time
 */
void U61_Score::get_best_avg_time(char **name,unsigned int *h,
				  unsigned int *m,
				  unsigned int *s)
{
  int i;
  int n;
  unsigned int time;

  time=0;
  *name="";

  n=local.size();
  for (i=0;i<n;++i)
    {
      if (local[i].get_avg_time()>time)
	{
	  time=local[i].get_avg_time();
	  *name=local[i].get_name();
	}
    }
  n=remote.size();
  for (i=0;i<n;++i)
    {
      if (remote[i].get_avg_time()>time)
	{
	  time=remote[i].get_avg_time();
	  *name=remote[i].get_name();
	}
    }
  *h=time/3600;
  *m=(time/60)%60;
  *s=time%60;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the average number of curses sent and the name of the player
 */
void U61_Score::get_best_avg_curse_sent(char **name,int *val)
{
  int i;
  int n;

  *val=0;
  *name="";

  n=local.size();
  for (i=0;i<n;++i)
    {
      if (local[i].get_avg_curse_sent()>*val)
	{
	  *val=local[i].get_avg_curse_sent();
	  *name=local[i].get_name();
	}
    }
  n=remote.size();
  for (i=0;i<n;++i)
    {
      if (remote[i].get_avg_curse_sent()>*val)
	{
	  *val=remote[i].get_avg_curse_sent();
	  *name=remote[i].get_name();
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the average number of curses received and the name of the player
 */
void U61_Score::get_best_avg_curse_received(char **name,int *val)
{
  int i;
  int n;

  *val=0;
  *name="";

  n=local.size();
  for (i=0;i<n;++i)
    {
      if (local[i].get_avg_curse_received()>*val)
	{
	  *val=local[i].get_avg_curse_received();
	  *name=local[i].get_name();
	}
    }
  n=remote.size();
  for (i=0;i<n;++i)
    {
      if (remote[i].get_avg_curse_received()>*val)
	{
	  *val=remote[i].get_avg_curse_received();
	  *name=remote[i].get_name();
	}
    }
}




