/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   script.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: wraps the LUA api to C++ and vice-versa
 *              all the interaction with lua is contained in this file
 *              it might be splited some day since it's getting *big*
 *              lua scripting is fundamental in u61. in fact all the
 *              rules of the game are coded in lua. this allows a 
 *              player/hacker  to write his own rules and test them
 *              right away without recompiling, and be quite
 *              "protection fault" safe
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/core.h>
#include <stdio.h>
#include <string.h>

/*
 * This akward stuff is to handle the fact that Debian renamed lua lua40
 * and also moved the header files
 */
#ifdef U61_DEF_LUA40
extern "C" {
#include <lua40/lua.h>
#include <lua40/lualib.h>
}
#else
extern "C" {
#include <lua.h>
#include <lualib.h>
}
#endif

#include "script.h"
#include "const.h"
#include "utils.h"
#include "log.h"
#include "global.h"
#include "version.h"
#include "script.h"
#include "macro.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_SCRIPT_LUA_GET_PROGRAM "user_get_program"
#define U61_SCRIPT_LUA_GET_VERSION "user_get_version"
#define U61_SCRIPT_LUA_START "user_start"
#define U61_SCRIPT_LUA_NEW_SHAPE "user_new_shape"
#define U61_SCRIPT_LUA_DO_SHAPE "user_do_shape"
#define U61_SCRIPT_LUA_ROTATE_LEFT "user_rotate_left"
#define U61_SCRIPT_LUA_ROTATE_RIGHT "user_rotate_right"
#define U61_SCRIPT_LUA_MOVE_LEFT "user_move_left"
#define U61_SCRIPT_LUA_MOVE_RIGHT "user_move_right"
#define U61_SCRIPT_LUA_MOVE_DOWN "user_move_down"
#define U61_SCRIPT_LUA_LAND "user_land"
#define U61_SCRIPT_LUA_MATCH_PATTERN "user_match_pattern"
#define U61_SCRIPT_LUA_TIME_CALLBACK_1 "user_time_callback_1"
#define U61_SCRIPT_LUA_TIME_CALLBACK_10 "user_time_callback_10"
#define U61_SCRIPT_LUA_TIME_CALLBACK_100 "user_time_callback_100"
#define U61_SCRIPT_LUA_SQUARE_BLOWN_UP "user_square_blown_up"
#define U61_SCRIPT_LUA_NEW_CURSE "user_new_curse"
#define U61_SCRIPT_LUA_DO_CURSE "user_do_curse"
#define U61_SCRIPT_LUA_USE_ANTIDOTE "user_use_antidote"
#define U61_SCRIPT_LUA_GET_CURSE_NAME "user_get_curse_name"

#define U61_SCRIPT_C_CLEAR_BLOCK "u61_clear_block"
#define U61_SCRIPT_C_CENTER_BLOCK "u61_center_block"
#define U61_SCRIPT_C_GET_BLOCK_X "u61_get_block_x"
#define U61_SCRIPT_C_GET_BLOCK_Y "u61_get_block_y"
#define U61_SCRIPT_C_SET_BLOCK_X "u61_set_block_x"
#define U61_SCRIPT_C_SET_BLOCK_Y "u61_set_block_y"
#define U61_SCRIPT_C_ADD_ITEM "u61_add_item"
#define U61_SCRIPT_C_GET_NB_ITEMS "u61_get_nb_items"
#define U61_SCRIPT_C_GET_ITEM_X "u61_get_item_x"
#define U61_SCRIPT_C_GET_ITEM_Y "u61_get_item_y"
#define U61_SCRIPT_C_GET_ITEM_COLOR "u61_get_item_color"
#define U61_SCRIPT_C_SET_ITEM_X "u61_set_item_x"
#define U61_SCRIPT_C_SET_ITEM_Y "u61_set_item_y"
#define U61_SCRIPT_C_SET_ITEM_COLOR "u61_set_item_color"
#define U61_SCRIPT_C_GET_WIDTH "u61_get_width"
#define U61_SCRIPT_C_GET_HEIGHT "u61_get_height"
#define U61_SCRIPT_C_SET_WIDTH "u61_set_width"
#define U61_SCRIPT_C_SET_HEIGHT "u61_set_height"
#define U61_SCRIPT_C_GET_SQUARE_COLOR "u61_get_square_color"
#define U61_SCRIPT_C_SET_SQUARE_COLOR "u61_set_square_color"
#define U61_SCRIPT_C_GET_TIME "u61_get_time"
#define U61_SCRIPT_C_GET_SCORE "u61_get_score"
#define U61_SCRIPT_C_SET_SCORE "u61_set_score"
#define U61_SCRIPT_C_ADD_SCORE "u61_add_score"
#define U61_SCRIPT_C_GET_GLOBAL "u61_get_global"
#define U61_SCRIPT_C_SET_GLOBAL "u61_set_global"
#define U61_SCRIPT_C_GET_ANTICIPATION_STATE "u61_get_anticipation_state"
#define U61_SCRIPT_C_SET_ANTICIPATION_STATE "u61_set_anticipation_state"
#define U61_SCRIPT_C_GET_PREVIEW_STATE "u61_get_preview_state"
#define U61_SCRIPT_C_SET_PREVIEW_STATE "u61_set_preview_state"
#define U61_SCRIPT_C_BLOW_UP_SQUARE "u61_blow_up_square"
#define U61_SCRIPT_C_IS_SQUARE_EXPLODING "u61_is_square_exploding"
#define U61_SCRIPT_C_GET_CURSE_X "u61_get_curse_x"
#define U61_SCRIPT_C_GET_CURSE_Y "u61_get_curse_y"
#define U61_SCRIPT_C_GET_CURSE_STATE "u61_get_curse_state"
#define U61_SCRIPT_C_SET_CURSE_X "u61_set_curse_x"
#define U61_SCRIPT_C_SET_CURSE_Y "u61_set_curse_y"
#define U61_SCRIPT_C_SET_CURSE_STATE "u61_set_curse_state"
#define U61_SCRIPT_C_IS_CURSE_AVAILABLE "u61_is_curse_available"
#define U61_SCRIPT_C_SEND_CURSE "u61_send_curse"
#define U61_SCRIPT_C_GET_CURSE_AGE "u61_get_curse_age"
#define U61_SCRIPT_C_REGISTER_CURSE "u61_register_curse"
#define U61_SCRIPT_C_CANCEL_CURSE "u61_cancel_curse"
#define U61_SCRIPT_C_GET_OLDEST_CURSE "u61_get_oldest_curse"
#define U61_SCRIPT_C_GET_NB_CURSES "u61_get_nb_curses"
#define U61_SCRIPT_C_GET_NB_ANTIDOTES "u61_get_nb_antidotes"
#define U61_SCRIPT_C_ADD_ANTIDOTE "u61_add_antidote"
#define U61_SCRIPT_C_DELETE_ANTIDOTE "u61_delete_antidote"
#define U61_SCRIPT_C_IS_BLOCK_OK "u61_is_block_ok"
#define U61_SCRIPT_C_CLEAR_MAP "u61_clear_map"
#define U61_SCRIPT_C_SHIFT_MAP "u61_shift_map"

#define U61_SCRIPT_RANDOM_MAX 0x7FFFFFFF
#define U61_SCRIPT_STACK_SIZE 2048

/*--------------------------------------------------------------------------*/
/*
 * for now, we don't use ClanLib's Lua stuff since it's mostly about
 * bindings and we don't use bindings in Lua. Instead we use Lua's API
 * directly. I'll probably send a patch to ClanLib when I have time, so
 * that CL_Lua contains wrappers for all Lua API, but it's not my topmost
 * priority. So for now, we need to store a lua_State global value...
 */
static lua_State *lua_state=NULL;

/*--------------------------------------------------------------------------*/
/*
 * the following pointers allow tradionnal C functions, called by Lua,
 * to act directly on a specified object
 * this way, when one writes lua code, one only has to assume that
 * functions always operate on the right block, the right map etc...
 * they are not members of any class since they are used from
 * standard C function
 */
static U61_Block *current_block=NULL;
static U61_Map *current_map=NULL;

/*--------------------------------------------------------------------------*/
/*
 * the following booleans allows a full control of what exactly functions
 * called from lua have the right to read/write
 * for instance one may wish a function to be able to read the map state
 * but bot overwrite it
 * they are not members of any class since they are used from
 * standard C function
 */
static bool block_read=false;
static bool block_write=false;
static bool map_read=false;
static bool map_write=false;

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * initializes Lua. The reason we do not use the standard CL_Lua::init()
 * function is that in recent versions of ClanLib, CL_Lua::init() initializes
 * much more than Lua: it loads a bunch of libs including "lua bindings"
 * which would allow the scripts to get an access to the screen and so on.
 * And the use of these would represent a potential security hole for U61,
 * since the scripts acan be obtained over Internet automatically without
 * any check at all...
 */
void U61_Script::open()
{
  lua_state=lua_open(U61_SCRIPT_STACK_SIZE);

  /*
   * we close the lua bindings so that there's no security problem.
   * Indeed, ClanLib opens the bindings by default, and we don't use
   * them in U61, so it's just a pure nuisance for us.
   */
  // disabled since we don't use CL_Lua yet
  //tolua_clanbindings_close(CL_Lua::state);
}

/*--------------------------------------------------------------------------*/
/*
 * De-initializes Lua
 */
void U61_Script::close()
{
  lua_close(lua_state);
}

/*--------------------------------------------------------------------------*/
/*
 * initializes the script module with a string
 */
bool U61_Script::do_string(char *str)
{
  bool bSuccess;
  int iResult;

  bSuccess=false;

  iResult=lua_dostring(lua_state,str);
  switch (iResult)
    {
    case 0:
      bSuccess=true;
      break;
    default:
      U61_LOG_WARNING("Lua error reading script, code is "<<iResult<<".");
      break;
    }

  return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * initializes the script module with a file
 */
bool U61_Script::do_file(char *name)
{
  bool bSuccess;
  int iResult;
  
  U61_Global::data.chdir_to_initial_dir();

  bSuccess=false;

  if (!U61_Utils::file_exists(name))
    {
      U61_LOG_WARNING("Unable to load script from \""<<name<<"\" (file does not exist).");
    }
  else
    {
      U61_LOG_DEBUG("Loading script from \""<<name<<"\".");
      iResult=lua_dofile(lua_state,name);
      switch (iResult)
	{
	case 0:
	  bSuccess=true;
	  break;
	default:
	  U61_LOG_WARNING("Lua error reading script, code is "<<iResult<<".");
	  break;
	}    
    }

  U61_Global::data.chdir_to_data_dir();

  return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as a string.
 * One must pass a buffer with a size of U61_CONST_STRING_SIZE+1 to store
 * the result.
 */
char *U61_Script::get_global_str(char *name,char *def)
{
  static char result[U61_CONST_STRING_SIZE];
  int top;

  top=lua_gettop(lua_state);

  U61_MACRO_STRCPY(result,def);

  lua_getglobal(lua_state,name);
  if (lua_isstring(lua_state,-1))
    {
      U61_MACRO_STRCPY(result,lua_tostring(lua_state,-1));
    }    

  lua_settop(lua_state,top);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as an integer
 */
int U61_Script::get_global_int(char *name,int def)
{
  float result=def;
  int top;

  top=lua_gettop(lua_state);

  lua_getglobal(lua_state,name);
  if (lua_isnumber(lua_state,-1))
    {
      result=lua_tonumber(lua_state,-1);
    }    

  lua_settop(lua_state,top);

  return (int) result;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as a boolean
 */
bool U61_Script::get_global_bool(char *name,bool def)
{
  return (get_global_int(name,def) != 0);
}

/*--------------------------------------------------------------------------*/
/*
 * registers all C functions that can be used from Lua
 */
bool U61_Script::init()
{
  bool bSuccess=true;

  lua_baselibopen(lua_state);
  lua_mathlibopen(lua_state);
  lua_strlibopen(lua_state);
  record_all();

  return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * forces the lua garbage collector
 */
void U61_Script::collect_garbage()
{
  /*  lua_Object poped;

      while (!lua_isnil(poped=lua_pop()))
      {
      U61_LOG_DEBUG("Popping lua object");
      }*/

  //lua_collectgarbage(0);
}

/*--------------------------------------------------------------------------*/
/*
 * cancels previous calls to prepare_for_???
 * this way, if an lua script tries to access data it is not allowed to,
 * no protection fault will occur (at least that's what we hope...)
 */
void U61_Script::secure_pointers()
{
  current_block=NULL;
  current_map=NULL;
  block_read=false;
  block_write=false;
  map_read=false;
  map_write=false;
}



/*--------------------------------------------------------------------------*/
/*
 * checks that the script is a correct U61 script
 */
bool U61_Script::check_id()
{
  bool success=true;
  char *program_read;
  char *version_read;
  char version[U61_CONST_STRING_SIZE];

  U61_Version::get_version_text(version);

  program_read=get_program();
  version_read=get_version();

  if (program_read==NULL || strcmp(program_read,U61_CONST_PROGRAM))
    {
      success=false;
      U61_LOG_WARNING("The script file does not seem to be a valid U61 script file, you should check that U61 is correctly installed.");
    }
  else
    {  
      if (version_read==NULL || strcmp(version_read,version))
	{
	  success=false;
	  U61_LOG_WARNING("The script file version is \""<<version_read<<"\" but the game version is \""<<version<<"\", so you might get errors when playing. Maybe you should just get rid of it and/or reinstall U61.");
	}
    }

  return success;    
}

/*--------------------------------------------------------------------------*/
/*
 * calls a user defined function which takes no argument
 * and returns nothing
 */
bool U61_Script::execute_lua_void_func_void(char *func)
{
  bool success=false;
  int top;
  int err;

  top=lua_gettop(lua_state);

  lua_getglobal(lua_state,func);

  err=lua_call(lua_state,0,0);
  switch (err)
    {
    case 0:
      success=true;
      break;
    default:
      U61_LOG_WARNING("Lua error executing \""<<func<<"()\", code is "<<err<<".");
      break;
    }

  lua_settop(lua_state,top);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * calls a user defined function which takes an integer as argument
 * and returns nothing
 */
bool U61_Script::execute_lua_void_func_int(char *func, 
					   int par1)
{
  bool success=false;
  int top;
  int err;

  top=lua_gettop(lua_state);

  lua_getglobal(lua_state,func);
  lua_pushnumber(lua_state,par1);

  err=lua_call(lua_state,1,0);
  switch (err)
    {
    case 0:
      success=true;
      break;
    default:
      U61_LOG_WARNING("Lua error executing \""<<func<<"("<<par1<<")\", code is "<<err<<".");
      break;
    }

  lua_settop(lua_state,top);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * calls a user defined function which takes 2 integers as argument
 * and returns nothing
 */
bool U61_Script::execute_lua_void_func_int_int(char *func, 
					       int par1, 
					       int par2)
{
  bool success=false;
  int top;
  int err;

  top=lua_gettop(lua_state);

  lua_getglobal(lua_state,func);
  lua_pushnumber(lua_state,par1);
  lua_pushnumber(lua_state,par2);

  err=lua_call(lua_state,2,0);
  switch (err)
    {
    case 0:
      success=true;
      break;
    default:
      U61_LOG_WARNING("Lua error executing \""<<func<<"("<<par1<<","<<par2<<"), code is "<<err<<".");
      break;
    }

  lua_settop(lua_state,top);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * calls a user defined function which takes an integer as argument
 * and returns an integer
 */
bool U61_Script::execute_lua_int_func_int(int *result, 
					  char *func, 
					  int par1)
{
  bool success=false;
  int top;
  int err;

  top=lua_gettop(lua_state);

  (*result)=0;

  lua_getglobal(lua_state,func);
  lua_pushnumber(lua_state,par1);

  err=lua_call(lua_state,1,1);
  switch (err)
    {
    case 0:
      if (lua_isnumber(lua_state,-1))
	{
	  *result=(int) lua_tonumber(lua_state,-1);
	  success=true;
	}
      else
	{
	  U61_LOG_WARNING("Lua error, result of \""<<func<<"("<<par1<<")\" is not a number")
	}
      break;
    default:
      U61_LOG_WARNING("Lua error executing \""<<func<<"("<<par1<<")\", code is "<<err<<".");
      break;
    }

  lua_settop(lua_state,top);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * calls a user defined function which takes no argument
 * and returns a string
 */
bool U61_Script::execute_lua_string_func_void(char result[U61_CONST_STRING_SIZE], 
					      char *func)
{
  bool success=false;
  int top;
  int err;

  top=lua_gettop(lua_state);

  result[0]='\0';

  lua_getglobal(lua_state,func);

  err=lua_call(lua_state,0,1);
  switch (err)
    {
    case 0:
      if (lua_isstring(lua_state,-1))
	{
	  U61_MACRO_STRNCPY(result,lua_tostring(lua_state,-1),U61_CONST_STRING_SIZE);
	  success=true;
	}
      else
	{
	  U61_LOG_WARNING("Lua error, result of \""<<func<<"()\" is not a string")
	}
      break;
    default:
      U61_LOG_WARNING("Lua error executing \""<<func<<"()\", code is "<<err<<".");
      break;
    }

  lua_settop(lua_state,top);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * calls a user defined function which takes an integer as argument
 * and returns a string
 */
bool U61_Script::execute_lua_string_func_int(char result[U61_CONST_STRING_SIZE],
					     char *func, 
					     int par1)
{
  bool success=false;
  int top;
  int err;

  top=lua_gettop(lua_state);

  result[0]='\0';

  lua_getglobal(lua_state,func);
  lua_pushnumber(lua_state,par1);

  err=lua_call(lua_state,1,1);
  switch (err)
    {
    case 0:
      if (lua_isstring(lua_state,-1))
	{
	  U61_MACRO_STRNCPY(result,lua_tostring(lua_state,-1),U61_CONST_STRING_SIZE);
	  success=true;
	}
      else
	{
	  U61_LOG_WARNING("Lua error, result of \""<<func<<"("<<par1<<")\" is not a string")
	}
      break;
    default:
      U61_LOG_WARNING("Lua error executing \""<<func<<"("<<par1<<")\", code is "<<err<<".");
      break;
    }

  lua_settop(lua_state,top);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the name of the program.
 * It is used to check that the script is a valid Lua script
 */
char *U61_Script::get_program()
{
  static char result[U61_CONST_STRING_SIZE];

  secure_pointers();

  execute_lua_string_func_void(result,
			       U61_SCRIPT_LUA_GET_PROGRAM);

  return (result);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the name of the version.
 * It is used to check that the script is a valid Lua script
 */
char *U61_Script::get_version()
{
  static char result[U61_CONST_STRING_SIZE];

  secure_pointers();

  execute_lua_string_func_void(result,
			       U61_SCRIPT_LUA_GET_VERSION);

  return (result);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function which can performs various initializations
 * at the beginning of a game.
 */
void U61_Script::start(U61_Map *m, U61_Block *b)
{
  secure_pointers();

  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_START);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the code for the next block.
 * The value returned will be passed later to block_shape
 */
int U61_Script::new_shape(U61_Map *m,U61_Block *b)
{
  int result;

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;

  execute_lua_int_func_int(&result,
			   U61_SCRIPT_LUA_NEW_SHAPE,
			   U61_Utils::random(U61_SCRIPT_RANDOM_MAX));

  return (result);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that creates shapes
 * it does not perform clear nor center functions
 */
void U61_Script::do_shape(U61_Map *m,U61_Block *b, int type)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_int(U61_SCRIPT_LUA_DO_SHAPE,type);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the right
 * the Lua function may or may not do a geometrical rotation, it
 * may for instance be color cycling
 */
void U61_Script::rotate_right(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_ROTATE_RIGHT);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the left
 * the Lua function may or may not do a geometrical rotation, it
 * may for instance be color cycling
 */
void U61_Script::rotate_left(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_ROTATE_LEFT);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that moves a block on the right
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::move_right(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_MOVE_RIGHT);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the left
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::move_left(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_MOVE_LEFT);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block down
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::move_down(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_MOVE_DOWN);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function which is meant to be call before
 * the block is stabilized/included in the map
 * the Lua function may leave the block unchanged
 */
void U61_Script::land(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_LAND);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that finds patterns in a map
 * and deletes the approprated blocks
 * this function should also update the score
 */
bool U61_Script::match_pattern(U61_Map *m)
{
  int result;

  secure_pointers();
  current_map=m;
  map_read=true;
  map_write=true;

  execute_lua_int_func_int(&result,
			   U61_SCRIPT_LUA_MATCH_PATTERN,
			   m->get_match_count());

  return (result ? true : false);
}

/*--------------------------------------------------------------------------*/
/*
 * function called on a regular time basis
 * it can update or modify anything in the map or the block
 * usefull to implement some curses, and/or update the score
 * it is called about 10 times/sec, and the rate does not increase
 * with the global speed of the game
 */
void U61_Script::time_callback_1(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_TIME_CALLBACK_1);
}

/*--------------------------------------------------------------------------*/
/*
 * function called on a regular time basis
 * it can update or modify anything in the map or the block
 * usefull to implement some curses, and/or update the score
 * it is called about 10 times/sec, and the rate does not increase
 * with the global speed of the game
 */
void U61_Script::time_callback_10(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_TIME_CALLBACK_10);
}

/*--------------------------------------------------------------------------*/
/*
 * function called on a regular time basis
 * it can update or modify anything in the map or the block
 * usefull to implement some curses, and/or update the score
 * it is called about 10 times/sec, and the rate does not increase
 * with the global speed of the game
 */
void U61_Script::time_callback_100(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_TIME_CALLBACK_100);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that handles the end of an explosion
 * Usually it makes the square disappear and shifts the upper squares
 * of one row down, but it can do other things, for instance blow up
 * some other blocks!
 */
void U61_Script::square_blown_up(U61_Map *m,int x,int y)
{
  secure_pointers();
  current_map=m;
  map_read=true;
  map_write=true;

  execute_lua_void_func_int_int(U61_SCRIPT_LUA_SQUARE_BLOWN_UP,
				x,
				y);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the code for the next curse.
 * The value returned will be used by the next call to do_curse
 */
int U61_Script::new_curse(U61_Map *m,U61_Block *b)
{
  int result;

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;

  execute_lua_int_func_int(&result,
			   U61_SCRIPT_LUA_NEW_CURSE,
			   U61_Utils::random(U61_SCRIPT_RANDOM_MAX));

  return (result);
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the Lua script function taht performs the action linked to a curse
 * This is were most of the curse effects should be coded
 */
void U61_Script::do_curse(U61_Map *m,U61_Block *b,int id,bool sent)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_int_int(U61_SCRIPT_LUA_DO_CURSE,
				id,
				sent ? 1 : 0);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that should be called when using an 
 * antidote
 */
void U61_Script::use_antidote(U61_Map *m,U61_Block *b)
{
  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  execute_lua_void_func_void(U61_SCRIPT_LUA_USE_ANTIDOTE);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the name of a curse.
 * The value returned will be dispalyed in the player's status zone
 */
char *U61_Script::get_curse_name(U61_Map *m,int id)
{
  static char result[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  map_read=true;

  execute_lua_string_func_int(result,
			      U61_SCRIPT_LUA_GET_CURSE_NAME,
			      id);

  return (result);
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it completely clears a block, ie after it the block has no squares left...
 */
static int clear_block(lua_State *ls)
{
  int result=0;

  if (current_block!=NULL && block_write)
    {
      current_block->clear();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it centers a block, ie the square of coord 0,0 will be in the middle
 */
static int center_block(lua_State *ls)
{
  int result=0;

  if (current_block!=NULL && block_write)
    {
      current_block->center();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the x position of the block
 */
static int get_block_x(lua_State *ls)
{
  int result=0;

  if (current_block!=NULL && block_read)
    {
      lua_pushnumber(ls,current_block->get_x());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the y position of the block
 */
static int get_block_y(lua_State *ls)
{
  int result=0;

  if (current_block!=NULL && block_read)
    {
      lua_pushnumber(ls,current_block->get_y());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x position of the block
 */
static int set_block_x(lua_State *ls)
{
  int result=0;
  int x;

  if (current_block!=NULL && block_write)
    {
      x=(int) lua_tonumber(ls,-1);

      current_block->set_x(x);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the y position of the block
 */
static int set_block_y(lua_State *ls)
{
  int result=0;
  int y;

  if (current_block!=NULL && block_write)
    {
      y=(int) lua_tonumber(ls,-1);

      current_block->set_y(y);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it adds a block item (ie a square) to the current block
 */
static int add_item(lua_State *ls)
{
  int result=0;
  int type,x,y;

  if (current_block!=NULL && block_write)
    {
      x=(int) lua_tonumber(ls,-3);        
      y=(int) lua_tonumber(ls,-2);        
      type=(int) lua_tonumber(ls,-1);        

      current_block->add_item(x,y,type);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the number of items (ie of squares) in a block
 */
static int get_nb_items(lua_State *ls)
{
  int result=0;

  if (current_block!=NULL && block_read)
    {
      lua_pushnumber(ls,current_block->get_nb_items());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the x coordinate of a block item (a square)
 */
static int get_item_x(lua_State *ls)
{
  int result=0;
  int i;

  if (current_block!=NULL && block_read)
    {
      i=(int) lua_tonumber(ls,-1);

      lua_pushnumber(ls,current_block->get_item_x(i));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the y coordinate of a block item (a square)
 */
static int get_item_y(lua_State *ls)
{
  int result=0;
  int i;

  if (current_block!=NULL && block_read)
    {
      i=(int) lua_tonumber(ls,-1);

      lua_pushnumber(ls,current_block->get_item_y(i));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the color of a block item (a square)
 */
static int get_item_color(lua_State *ls)
{
  int result=0;

  int i;

  if (current_block!=NULL && block_read)
    {
      i=(int) lua_tonumber(ls,-1);

      lua_pushnumber(ls,current_block->get_item_color(i));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static int set_item_x(lua_State *ls)
{
  int result=0;
  int i,x;

  if (current_block!=NULL && block_write)
    {
      i=(int) lua_tonumber(ls,-2);
      x=(int) lua_tonumber(ls,-1);

      current_block->set_item_x(i,x);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static int set_item_y(lua_State *ls)
{
  int result=0;
  int i,y;

  if (current_block!=NULL && block_write)
    {
      i=(int) lua_tonumber(ls,-2);
      y=(int) lua_tonumber(ls,-1);

      current_block->set_item_y(i,y);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static int set_item_color(lua_State *ls)
{
  int result=0;
  int i,color;

  if (current_block!=NULL && block_write)
    {
      i=(int) lua_tonumber(ls,-2);
      color=(int) lua_tonumber(ls,-1);

      current_block->set_item_color(i,color);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the width of the map
 */
static int get_width(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_width());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the height of the map
 */
static int get_height(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_height());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the width of the map
 */
static int set_width(lua_State *ls)
{
  int result=0;
  int w;

  if (current_map!=NULL && map_write)
    {
      w=(int) lua_tonumber(ls,-1);

      current_map->set_width(w);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the height of the map
 */
static int set_height(lua_State *ls)
{
  int result=0;
  int h;

  if (current_map!=NULL && map_write)
    {
      h=(int) lua_tonumber(ls,-1);

      current_map->set_height(h);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it gets the color of a given square, color -1 means no square
 */
static int get_square_color(lua_State *ls)
{
  int result=0;
  int x,y;

  if (current_map!=NULL && map_read)
    {
      x=(int) lua_tonumber(ls,-2);
      y=(int) lua_tonumber(ls,-1);

      lua_pushnumber(ls,current_map->get_square_color(x,y));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the color of a given square, color -1 means no square
 */
static int set_square_color(lua_State *ls)
{
  int result=0;
  int x,y,color;

  if (current_map!=NULL && map_write)
    {
      x=(int) lua_tonumber(ls,-3);
      y=(int) lua_tonumber(ls,-2);
      color=(int) lua_tonumber(ls,-1);

      current_map->set_square_color(x,y,color);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the time associated to a map
 */
static int get_time(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_time());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the score associated to a map
 */
static int get_score(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_score());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the value of the score associated to a map
 */
static int set_score(lua_State *ls)
{
  int result=0;
  int score;

  if (current_map!=NULL && map_write)
    {
      score=(int) lua_tonumber(ls,-1);

      current_map->set_score(score);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it adds the value to the score associated to a map
 */
static int add_score(lua_State *ls)
{
  int result=0;
  int score;

  if (current_map!=NULL && map_write)
    {
      score=(int) lua_tonumber(ls,-1);

      current_map->add_score(score);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get global values associated to a map
 */
static int get_global(lua_State *ls)
{
  int result=0;
  int i;

  if (current_map!=NULL && map_read)
    {
      i=(int) lua_tonumber(ls,-1);

      lua_pushnumber(ls,current_map->get_global(i));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set global values associated to a map
 */
static int set_global(lua_State *ls)
{
  int result=0;
  int i,glob;

  if (current_map!=NULL && map_write)
    {
      i=(int) lua_tonumber(ls,-2);
      glob=(int) lua_tonumber(ls,-1);

      current_map->set_global(i,glob);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the anticipation state, ie if the anticipation is to be seen
 */
static int get_anticipation_state(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_anticipation_state() ? 1 : 0);
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the anticipation state, ie if the anticipation is to be seen
 */
static int set_anticipation_state(lua_State *ls)
{
  int result=0;
  bool state;

  if (current_map!=NULL && map_write)
    {
      state=(((float) lua_tonumber(ls,-1))!=0.);

      current_map->set_anticipation_state(state);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the preview state, ie if the preview is to be seen
 */
static int get_preview_state(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_preview_state() ? 1 : 0);
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the preview state, ie if the preview is to be seen
 */
static int set_preview_state(lua_State *ls)
{
  int result=0;

  bool state;

  if (current_map!=NULL && map_write)
    {
      state=(((float) lua_tonumber(ls,-1))!=0.);

      current_map->set_preview_state(state);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Blows up a square, ie puts this square in a exploding state
 */
static int blow_up_square(lua_State *ls)
{
  int result=0;
  int x,y;

  if (current_map!=NULL && map_write)
    {
      x=(int) lua_tonumber(ls,-2);
      y=(int) lua_tonumber(ls,-1);

      current_map->blow_up_square(x,y);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns != 0 if the square is exploding, and 0 otherwise
 */
static int is_square_exploding(lua_State *ls)
{
  int result=0;
  int x,y;

  if (current_map!=NULL && map_read)
    {
      x=(int) lua_tonumber(ls,-2);
      y=(int) lua_tonumber(ls,-1);

      lua_pushnumber(ls,
		     current_map->is_square_exploding(x,y)
		     ? 1 :0);
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the x coordinate of the curse square
 */
static int get_curse_x(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_curse_x());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the y coordinate of the curse square
 */
static int get_curse_y(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_curse_y());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the state of the curse square
 */
static int get_curse_state(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_curse_state());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set the x coordinate of the curse square
 */
static int set_curse_x(lua_State *ls)
{
  int result=0;
  int x;

  if (current_map!=NULL && map_write)
    {
      x=(int) lua_tonumber(ls,-1);

      current_map->set_curse_x(x);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set the y coordinate of the curse square
 */
static int set_curse_y(lua_State *ls)
{
  int result=0;
  int y;

  if (current_map!=NULL && map_write)
    {
      y=(int) lua_tonumber(ls,-1);

      current_map->set_curse_y(y);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set the state of the curse square
 */
static int set_curse_state(lua_State *ls)
{
  int result=0;
  bool state;

  if (current_map!=NULL && map_write)
    {
      state=(((float) lua_tonumber(ls,-1))!=0.);

      current_map->set_curse_state(state);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to know if the curse square is set and
 * is set on a valid square (ie a non-empty square)
 */
static int is_curse_available(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->is_curse_available());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to send a curse to the target player
 */
static int send_curse(lua_State *ls)
{
  int result=0;
  int par;

  if (current_map!=NULL && map_write)
    {
      par=(int) lua_tonumber(ls,-1);

      current_map->send_curse(par);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the age of a registered curse
 */
static int get_curse_age(lua_State *ls)
{
  int result=0;
  int i;

  if (current_map!=NULL && map_read)
    {
      i=(int) lua_tonumber(ls,-1);

      lua_pushnumber(ls,current_map->get_curse_age(i));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to register a curse
 */
static int register_curse(lua_State *ls)
{
  int result=0;
  int i;
  int length;
  bool good;

  if (current_map!=NULL && map_write)
    {
      i=(int) lua_tonumber(ls,-3);
      length=(int) lua_tonumber(ls,-2);
      good=(((float) lua_tonumber(ls,-1))!=0.);

      current_map->register_curse(i,length,good);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to cancel a registered curse
 */
static int cancel_curse(lua_State *ls)
{
  int result=0;
  int i;

  if (current_map!=NULL && map_write)
    {
      i=(int) lua_tonumber(ls,-1);

      current_map->cancel_curse(i);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the id of the oldest curse
 */
static int get_oldest_curse(lua_State *ls)
{
  int result=0;
  bool good;

  if (current_map!=NULL && map_read)
    {
      good=(((float) lua_tonumber(ls,-1))!=0.);

      lua_pushnumber(ls,current_map->get_oldest_curse(good));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the number of persistent curses activated
 */
static int get_nb_curses(lua_State *ls)
{
  int result=0;
  bool good;

  if (current_map!=NULL && map_read)
    {
      good=(((float) lua_tonumber(ls,-1))!=0.);

      lua_pushnumber(ls,current_map->get_nb_curse(good));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the number of antidotes available
 */
static int get_nb_antidotes(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read)
    {
      lua_pushnumber(ls,current_map->get_nb_antidote());
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it adds an antidote
 */
static int add_antidote(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_write)
    {
      current_map->add_antidote();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it deletes an antidote
 */
static int delete_antidote(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_write)
    {
      current_map->delete_antidote();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it checks if the current block has a correct position
 */
static int is_block_ok(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_read && current_block!=NULL && block_read)
    {
      lua_pushnumber(ls,current_map->is_block_ok(current_block));
      result=1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it clears a map from all its squares
 */
static int clear_map(lua_State *ls)
{
  int result=0;

  if (current_map!=NULL && map_write)
    {
      current_map->clear();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it shifts a map on both vertical and/or horizontal ways
 */
static int shift_map(lua_State *ls)
{
  int result=0;
  int x,y;

  if (current_map!=NULL && map_write)
    {
      x=(int) lua_tonumber(ls,-2);
      y=(int) lua_tonumber(ls,-1);

      current_map->shift(x,y);
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * registers all C functions that can be used from Lua
 */
void U61_Script::record_all()
{
  lua_register(lua_state,U61_SCRIPT_C_CLEAR_BLOCK,clear_block);
  lua_register(lua_state,U61_SCRIPT_C_CENTER_BLOCK,center_block);
  lua_register(lua_state,U61_SCRIPT_C_GET_BLOCK_X,get_block_x);
  lua_register(lua_state,U61_SCRIPT_C_GET_BLOCK_Y,get_block_y);
  lua_register(lua_state,U61_SCRIPT_C_SET_BLOCK_X,set_block_x);
  lua_register(lua_state,U61_SCRIPT_C_SET_BLOCK_Y,set_block_y);
  lua_register(lua_state,U61_SCRIPT_C_ADD_ITEM,add_item);
  lua_register(lua_state,U61_SCRIPT_C_GET_NB_ITEMS,get_nb_items);
  lua_register(lua_state,U61_SCRIPT_C_GET_ITEM_X,get_item_x);
  lua_register(lua_state,U61_SCRIPT_C_GET_ITEM_Y,get_item_y);
  lua_register(lua_state,U61_SCRIPT_C_GET_ITEM_COLOR,get_item_color);
  lua_register(lua_state,U61_SCRIPT_C_SET_ITEM_X,set_item_x);
  lua_register(lua_state,U61_SCRIPT_C_SET_ITEM_Y,set_item_y);
  lua_register(lua_state,U61_SCRIPT_C_SET_ITEM_COLOR,set_item_color);
  lua_register(lua_state,U61_SCRIPT_C_GET_WIDTH,get_width);
  lua_register(lua_state,U61_SCRIPT_C_GET_HEIGHT,get_height);
  lua_register(lua_state,U61_SCRIPT_C_SET_WIDTH,set_width);
  lua_register(lua_state,U61_SCRIPT_C_SET_HEIGHT,set_height);
  lua_register(lua_state,U61_SCRIPT_C_GET_SQUARE_COLOR,get_square_color);
  lua_register(lua_state,U61_SCRIPT_C_SET_SQUARE_COLOR,set_square_color);
  lua_register(lua_state,U61_SCRIPT_C_GET_TIME,get_time);
  lua_register(lua_state,U61_SCRIPT_C_GET_SCORE,get_score);
  lua_register(lua_state,U61_SCRIPT_C_SET_SCORE,set_score);
  lua_register(lua_state,U61_SCRIPT_C_ADD_SCORE,add_score);
  lua_register(lua_state,U61_SCRIPT_C_GET_GLOBAL,get_global);
  lua_register(lua_state,U61_SCRIPT_C_SET_GLOBAL,set_global);
  lua_register(lua_state,U61_SCRIPT_C_GET_ANTICIPATION_STATE,get_anticipation_state);
  lua_register(lua_state,U61_SCRIPT_C_SET_ANTICIPATION_STATE,set_anticipation_state);
  lua_register(lua_state,U61_SCRIPT_C_GET_PREVIEW_STATE,get_preview_state);
  lua_register(lua_state,U61_SCRIPT_C_SET_PREVIEW_STATE,set_preview_state);
  lua_register(lua_state,U61_SCRIPT_C_BLOW_UP_SQUARE,blow_up_square);
  lua_register(lua_state,U61_SCRIPT_C_IS_SQUARE_EXPLODING,is_square_exploding);
  lua_register(lua_state,U61_SCRIPT_C_GET_CURSE_X,get_curse_x);
  lua_register(lua_state,U61_SCRIPT_C_GET_CURSE_Y,get_curse_y);
  lua_register(lua_state,U61_SCRIPT_C_GET_CURSE_STATE,get_curse_state);
  lua_register(lua_state,U61_SCRIPT_C_SET_CURSE_X,set_curse_x);
  lua_register(lua_state,U61_SCRIPT_C_SET_CURSE_Y,set_curse_y);
  lua_register(lua_state,U61_SCRIPT_C_SET_CURSE_STATE,set_curse_state);
  lua_register(lua_state,U61_SCRIPT_C_IS_CURSE_AVAILABLE,is_curse_available);
  lua_register(lua_state,U61_SCRIPT_C_SEND_CURSE,send_curse);
  lua_register(lua_state,U61_SCRIPT_C_GET_CURSE_AGE,get_curse_age);
  lua_register(lua_state,U61_SCRIPT_C_REGISTER_CURSE,register_curse);
  lua_register(lua_state,U61_SCRIPT_C_CANCEL_CURSE,cancel_curse);
  lua_register(lua_state,U61_SCRIPT_C_GET_OLDEST_CURSE,get_oldest_curse);
  lua_register(lua_state,U61_SCRIPT_C_GET_NB_CURSES,get_nb_curses);
  lua_register(lua_state,U61_SCRIPT_C_GET_NB_ANTIDOTES,get_nb_antidotes);
  lua_register(lua_state,U61_SCRIPT_C_ADD_ANTIDOTE,add_antidote);
  lua_register(lua_state,U61_SCRIPT_C_DELETE_ANTIDOTE,delete_antidote);
  lua_register(lua_state,U61_SCRIPT_C_IS_BLOCK_OK,is_block_ok);
  lua_register(lua_state,U61_SCRIPT_C_CLEAR_MAP,clear_map);
  lua_register(lua_state,U61_SCRIPT_C_SHIFT_MAP,shift_map);
}





