/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   serverdispatcher.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the server dispatcher is the most complex of all dispatchers,
 *              it gets messages from all the clients, and sends them back
 *              to anyone who needs them
 */


/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include "serverdispatcher.h"
#include "log.h"

/*---------------------------------------------------------------------------
  variants
  ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a dispatcher
 */
U61_ServerDispatcher::U61_ServerDispatcher() : U61_Dispatcher()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * resets everything in the dispatcher (empty queues...)
 */
void U61_ServerDispatcher::reset()
{
  int i;

  U61_Dispatcher::reset();
  for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
    {
      history[i].reset();
      history[i].set_player_id(i);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Opens a connection, ie from now events will be send to and received
 * from this connection
 */
bool U61_ServerDispatcher::open(U61_Connection *c)
{
  int i,j;
  bool ok=true;
  U61_Event evt;
  int history_size;

  U61_LOG_MESSAGE("New client accepted.");
  connection.push_back(c);

  /*
   * Now that this connection has been added to the list
   * of active connections, we need to send a complete
   * history of what happened to other players
   */
  for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
    {
      /*
       * If a player is "authorized" on this connection, it means that it 
       * is "his" connection, therefore one must not send history to him 8-)
       */
      if (!c->is_authorized(i))
	{
	  U61_LOG_DEBUG("Sending history for player "<<i<<"");
	  history_size=(int) history[i].size();
	  if (history_size>0)
	    {
	      /*
	       * We get the first event in the history to prepare our
	       * HISTORY_BEGIN event. It's important, since we can not
	       * guess parameters such as time for instance.
	       */
	      evt=history[i].get(0);
	      evt.code=U61_EVENT_HISTORY_BEGIN;
	      evt.target=evt.author=i;
	      /*
	       * Now we send our HISTORY_BEGIN event
	       */
	      if (!c->send(evt))
		{
		  ok=false;
		}
	      /*
	       * Now we send all the events, including the one we had
	       * used to prepare the special HISTORY_BEGIN event.
	       */
	      for (j=0;j<history_size;++j)
		{
		  evt=history[i].get(j);
		  if (!c->send(evt))
		    {
		      ok=false;
		    }  
		}
	      /*
	       * Now we send the HISTORY_END event to mark the fact
	       * that the history is done. It's important, since during
	       * the transmission/interpretation of a long history,
	       * we do not want any display, but when the history is done,
	       * one must turn the map display back on.
	       */
	      evt.code=U61_EVENT_HISTORY_END;
	      evt.target=evt.author=i;
	      if (!c->send(evt))
		{
		  ok=false;
		}
	    }
	}
    }
  if (!ok)
    {
      U61_LOG_WARNING("Impossible to send complete history, too much data???");
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Closes a connection, ie no more events will be sent, received from there
 */
void U61_ServerDispatcher::close(U61_Connection *c)
{
  int i;
  int size;
  std::vector<U61_Connection *> temp;

  U61_LOG_DEBUG("Closing connection");

  /*
   * I could not find out how to remove an item from a vector,
   * if you know just send me a mail! (ufoot@ufoot.org)
   * so now this is an ugly hack to do it manually
   * BTW it's almost never called so who cares...
   */
  size=connection.size();
  for (i=0;i<size;++i)
    {
      temp.push_back(connection[i]);
    }
  connection.clear();
  for (i=0;i<size;++i)
    {
      if(temp[i]!=c)
	{
	  connection.push_back(temp[i]);
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends an event to all the connections
 */
bool U61_ServerDispatcher::send(U61_Event evt)
{
  int i;
  int size;
  bool ok=true;

  size=connection.size();
  for (i=0;i<size;++i)
    {
      U61_LOG_DEBUG("Sending event "<<evt<<" on connection "<<i<<"");
      if (!connection[i]->send(evt))
	{
	  ok=false;
	}
    }

  /*
   * Now this is a very important point, as an event is sent, we
   * need to bufferize it and store so that new players receive
   * a complete history of what happened to older players.
   * This is the safest way to garantee that there are no
   * network inconsistencies.
   */
  history[evt.target].put(evt);  

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * receives an event from a connection. We don't know which connection it is
 * we just know the event arrived. This is enough since we have the player
 * id in the event.
 */
bool U61_ServerDispatcher::recv(U61_Event *evt)
{
  int size;
  int i,j;
  bool ok=true;
  bool found=false;

  size=connection.size();
  for (i=0;i<size && !found;++i)
    {
      if (connection[i]->peek_event())
	{
          found=true;
          ok=connection[i]->recv(evt);
          /*
           * Now, as we get messages from foreign connections, we copy
           * them to other connections. This is a transparent process,
           * in fact all the connections will just receive the very
           * same things the server did.
           */
	  for (j=0;j<size;++j)
	    {
              /*
               * We do not send the message to the one who just
               * sent it to us!
               */
              if (i!=j)
		{
                  if (!connection[j]->send(*evt))
		    {
                      ok=false;
	            }
		}
	    }

          /*
           * We also copy everything in the history, this way
           * the server keeps a track of what happened to every client
           * and this is usefull when new clients arrive
           */
          history[evt->target].put(*evt);  
	}
    }
  if (found==false)
    {
      ok=false;
    }

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * returns true if there's at least an event available on one connection
 */
bool  U61_ServerDispatcher::peek()
{
  int size;
  int i;
  bool found=false;
  U61_Event evt;

  size=connection.size();
  for (i=0;i<size && !found;++i)
    {
      if (connection[i]->peek_event())
	{
          found=true;
 	}
    }
 
  return found;
} 



