/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   serverprotocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: welcomes a client, gets basic information about him
 *              and sends him all the informations about
 *              the players which are already playing
 */


/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include "serverprotocol.h"
#include "global.h"
#include "time.h"
#include "platform.h"
#include "log.h"
#include "macro.h"
#include "checksum.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

#define U61_SERVERPROTOCOL_LISTENING           10
#define U61_SERVERPROTOCOL_CLIENT_ACCEPTED     11
#define U61_SERVERPROTOCOL_PASSWORD_RECEIVED   30
#define U61_SERVERPROTOCOL_PASSWORD_OK_SENT    31
#define U61_SERVERPROTOCOL_SCRIPT_NAME_SENT    40
#define U61_SERVERPROTOCOL_SCRIPT_SENT         41
#define U61_SERVERPROTOCOL_ID0_SENT            50
#define U61_SERVERPROTOCOL_ID1_SENT            51
#define U61_SERVERPROTOCOL_INITIAL_SPEED_SENT  52
#define U61_SERVERPROTOCOL_ACCELERATION_SENT   53
#define U61_SERVERPROTOCOL_CURSE_DELAY_SENT    54
#define U61_SERVERPROTOCOL_TIME_SENT           55

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a server protocol
 */
U61_ServerProtocol::U61_ServerProtocol(U61_ServerDispatcher *d, int p)
  : U61_Protocol(p)
{
  int i,j;

  dispatcher=d;
  new_connection=-1;

  for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
    {
      connection[i]=NULL;
      for (j=0;j<U61_GAME_NB_PLAYERS_PER_NET_GROUP;++j)
	{
	  id[i][j]=-1;
	}	
    }
}

/*--------------------------------------------------------------------------*/
/*
 * deletes the protocol object
 */
U61_ServerProtocol::~U61_ServerProtocol()
{
  int i;

  for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
    {
      if (connection[i]!=NULL)
	{
	  delete connection[i];
          connection[i]=NULL;
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if there are some dead connections, and kills the players
 * if necessary
 */
void U61_ServerProtocol::check_connection()
{
  int i,j;
  U61_Player *p;
  U61_Event evt;  

  for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
    {
      if (connection[i]!=NULL)
	{
	  if (!connection[i]->is_alive())
	    {
	      if (dispatcher!=NULL)
		{
		  dispatcher->close(connection[i]);
		}
	      delete connection[i];
	      connection[i]=NULL;
	      
	      for (j=0;j<U61_GAME_NB_PLAYERS_PER_NET_GROUP;++j)
		{
		  evt.time=0;
		  evt.author=evt.target=id[i][j];
		  
		  evt.code=U61_EVENT_READY;
		  dispatcher->put_out(evt);   
		  
		  evt.code=U61_EVENT_KILL;
		  dispatcher->put_out(evt);   
		  
		  p=U61_Global::game.find_player_by_id(id[i][j]);
		  if (p!=NULL)
		    {
		      p->kill();
		    }
		  id[i][j]=-1;
		}		  
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to send to the client
 */
int U61_ServerProtocol::send_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_PROTOCOL_BEGIN:
      if ((result=start_listening())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_LISTENING;
	  U61_LOG_DEBUG("stage=LISTENING");
	}
      break;
    case U61_SERVERPROTOCOL_CLIENT_ACCEPTED:
      if ((result=new_connection>=0 ? send_program(connection[new_connection]) : U61_PROTOCOL_FAILED)==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_PROGRAM_SENT;
	  U61_LOG_DEBUG("stage=PROGRAM_SENT");
	}
      break;
    case U61_PROTOCOL_PROGRAM_RECEIVED:
      if ((result=new_connection>=0 ? send_version(connection[new_connection]) : U61_PROTOCOL_FAILED)==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_SENT;
	  U61_LOG_DEBUG("stage=VERSION_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_PASSWORD_RECEIVED:
      if ((result=send_password_ok())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_PASSWORD_OK_SENT;
	  U61_LOG_DEBUG("stage=PASSWORD_OK_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_PASSWORD_OK_SENT:
      if ((result=send_script_name())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_SCRIPT_NAME_SENT;
	  U61_LOG_DEBUG("stage=SCRIPT_NAME_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_SCRIPT_NAME_SENT:
      if ((result=send_script())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_SCRIPT_SENT;
	  U61_LOG_DEBUG("stage=SCRIPT_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_SCRIPT_SENT:
      if ((result=send_id0())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_ID0_SENT;
	  U61_LOG_DEBUG("stage=ID0_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_ID0_SENT:
      if ((result=send_id1())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_ID1_SENT;
	  U61_LOG_DEBUG("stage=ID1_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_ID1_SENT:
      if ((result=send_initial_speed())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_INITIAL_SPEED_SENT;
	  U61_LOG_DEBUG("stage=INITIAL_SPEED_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_INITIAL_SPEED_SENT:
      if ((result=send_acceleration())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_ACCELERATION_SENT;
	  U61_LOG_DEBUG("stage=ACCELERATION_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_ACCELERATION_SENT:
      if ((result=send_curse_delay())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_CURSE_DELAY_SENT;
	  U61_LOG_DEBUG("stage=CURSE_DELAY_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_CURSE_DELAY_SENT:
      if ((result=send_time())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_TIME_SENT;
	  U61_LOG_DEBUG("stage=TIME_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_TIME_SENT:
      if ((result=new_connection>=0 ? send_ready(connection[new_connection]) : U61_PROTOCOL_FAILED)==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_SENT;
	  U61_LOG_DEBUG("stage=READY_SENT");
	}
      break;
    case U61_PROTOCOL_READY_RECEIVED:
      if (new_connection>=0) 
	{
	  result=U61_PROTOCOL_DONE;
	  dispatcher->open(connection[new_connection]);
	  new_connection=-1;
	  U61_LOG_DEBUG("stage=LISTENING");
	  stage=U61_SERVERPROTOCOL_LISTENING;
	}
      else
	{
	  result=U61_PROTOCOL_FAILED;
	}

      break;
    }

  handle_error(&result);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to receive from the
 * client
 */
int U61_ServerProtocol::recv_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_SERVERPROTOCOL_LISTENING:
      if ((result=accept_client())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_CLIENT_ACCEPTED;
	  U61_LOG_DEBUG("stage=CLIENT_ACCEPTED");
	}
      break;
    case U61_PROTOCOL_PROGRAM_SENT:
      if ((result=new_connection>=0 ? recv_program(connection[new_connection]) : U61_PROTOCOL_FAILED)==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_PROGRAM_RECEIVED;
	  U61_LOG_DEBUG("stage=PROGRAM_RECEIVED");
	}
      break;
    case U61_PROTOCOL_VERSION_SENT:
      if ((result=new_connection>=0 ? recv_version(connection[new_connection]) : U61_PROTOCOL_FAILED)==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_RECEIVED;
	  U61_LOG_DEBUG("stage=VERSION_RECEIVED");
	}
      break;
    case U61_PROTOCOL_VERSION_RECEIVED:
      if ((result=new_connection>=0 ? recv_password() : U61_PROTOCOL_FAILED)==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_PASSWORD_RECEIVED;
	  U61_LOG_DEBUG("stage=PASSWORD_RECEIVED");
	}
      break;
    case U61_PROTOCOL_READY_SENT:
      if ((result=new_connection>=0 ? recv_ready(connection[new_connection]) : U61_PROTOCOL_FAILED)==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_RECEIVED;
	  U61_LOG_DEBUG("stage=READY_RECEIVED");
	}
      break;
    }

  handle_error(&result);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * starts listening what's happening on the network, in case a computer wishes
 * to connect itself
 */
int U61_ServerProtocol::start_listening()
{
  int result=U61_PROTOCOL_LATER;

  if (listening_sock.listen(port))
    {
      result=U61_PROTOCOL_DONE;
    }
  else
    {
      U61_MACRO_SPRINTF0(diagnostic,
			 "Unable to bind socket");

      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * detects a computer, ie accepts to dialog with it, it does not mean that
 * the players on this remote machine will actually be accepted in the
 * game but at least we'll dialog with them 
 */
int U61_ServerProtocol::accept_client()
{
  int result=U61_PROTOCOL_LATER;
  U61_BufferedSocket *accepted_sock;
  
  accepted_sock=listening_sock.accept();

  if (accepted_sock)
    {
      if ((new_connection=get_free_connection())<0)
	{
	  result=U61_PROTOCOL_FAILED;
	}
      else
	{
	  if ((connection[new_connection]=new U61_Connection(accepted_sock))!=NULL)
	    {	      
	      result=U61_PROTOCOL_DONE;
	    }
	  else
	    {
	      result=U61_PROTOCOL_FAILED;
	    }
	}

      if (result != U61_PROTOCOL_DONE)
	{
	  new_connection=-1;
	  delete accepted_sock;
	}
    }
  
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the index of a free connection in the connection array, if
 * no connection is available it just returns a <0 value
 */
int U61_ServerProtocol::get_free_connection()
{
  int i;
  int result=-1;

  for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
    {
      if (connection[i]==NULL)
	{
	  result=i;
	  break;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the password.
 */
int U61_ServerProtocol::recv_password()
{
  int result=U61_PROTOCOL_LATER;
  char ascii_sig_client[U61_CONST_STRING_SIZE];
  char *ascii_sig_server;
  U61_Checksum checksum;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      if (connection[new_connection]->peek_str())
	{
	  if (connection[new_connection]->recv(ascii_sig_client,
					       sizeof(ascii_sig_client)-1))
	    {
	      if (!U61_Global::config.use_password)
		{
		  result=U61_PROTOCOL_DONE;
		  U61_LOG_DEBUG("No password check.");
		}
	      else
		{
		  checksum.calc(U61_Global::config.password,
				strlen(U61_Global::config.password));
		  ascii_sig_server=checksum.get_ascii_sig();
		  
		  if (strcmp(ascii_sig_client,ascii_sig_server)==0)
		    {
		      result=U61_PROTOCOL_DONE;
		      U61_LOG_DEBUG("Password is OK");
		    }
		  else
		    {
		      result=U61_PROTOCOL_FAILED;
		      U61_LOG_WARNING("Wrong password.");
		    }
		}
	    }
	}      
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the message "password is OK"
 */
int U61_ServerProtocol::send_password_ok()
{
  int result=U61_PROTOCOL_LATER;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      if (connection[new_connection]->send(U61_PROTOCOL_OK_MESSAGE,
					   strlen(U61_PROTOCOL_OK_MESSAGE)))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the script name to the client, we remove the path from the
 * filename, for the full path does not interest the remote client
 */
int U61_ServerProtocol::send_script_name()
{
  int result=U61_PROTOCOL_LATER;
  char script_name[U61_CONST_STRING_SIZE];

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      U61_MACRO_STRCPY(script_name,
		       U61_Global::config.script_file);
      U61_Platform::strip_remote_script_path(script_name);
      if (connection[new_connection]->send(script_name,sizeof(script_name)-1))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the whole script to the client
 */
int U61_ServerProtocol::send_script()
{
  int result=U61_PROTOCOL_LATER;
  char script[U61_CONST_SCRIPT_SIZE];
  char script_name[U61_CONST_STRING_SIZE];
  FILE *f;
  int c;
  int pos=0;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      U61_Global::data.chdir_to_initial_dir();

      U61_MACRO_STRCPY(script_name,
		       U61_Global::config.script_file);

      f=fopen(script_name,"r");
      if (f!=NULL)
	{
	  while ((c=fgetc(f))!=EOF && pos<(int) sizeof(script)-1)
	    {
	      script[pos++]=c;
	    }
	  fclose(f);
	  if (connection[new_connection]->send(script,sizeof(script)-1))
	    {
	      result=U61_PROTOCOL_DONE;
	    }
	}
      else
	{
	  U61_LOG_DEBUG("Unable to open \""<<script_name<<"\".");
	  U61_MACRO_SPRINTF1(diagnostic,
			     "Could not open script file \"%s\"",
			     script_name);
	  result=U61_PROTOCOL_FAILED;
	}

      U61_Global::data.chdir_to_data_dir();
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the first id that will be usable by the client
 */
int U61_ServerProtocol::send_id0()
{
  int result=U61_PROTOCOL_LATER;
  int id0;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      id0=U61_Global::game.find_free_player_id0();
      if (connection[new_connection]->send(id0))
	{
	  id[new_connection][0]=id0;
	  connection[new_connection]->authorize_player_id(id0,true);
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the second id that will be usable by the client
 */
int U61_ServerProtocol::send_id1()
{
  int result=U61_PROTOCOL_LATER;
  int id1;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      id1=U61_Global::game.find_free_player_id1();
      if (connection[new_connection]->send(id1))
	{
	  id[new_connection][1]=id1;
	  connection[new_connection]->authorize_player_id(id1,true);
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the initial speed that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_initial_speed()
{
  int result=U61_PROTOCOL_LATER;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      if (connection[new_connection]->send(U61_Global::config.initial_speed))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the acceleration that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_acceleration()
{
  int result=U61_PROTOCOL_LATER;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      if (connection[new_connection]->send(U61_Global::config.acceleration))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the curse delay that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_curse_delay()
{
  int result=U61_PROTOCOL_LATER;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      if (connection[new_connection]->send(U61_Global::config.curse_delay))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the time that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_time()
{
  int result=U61_PROTOCOL_LATER;

  if (new_connection>=0 && connection[new_connection]!=NULL)
    {
      if (connection[new_connection]->send(U61_Time::for_event()))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Checks the return value of send & recv functions
 * stops everything if something's going wrong
 */
void U61_ServerProtocol::handle_error(int *result)
{
  if ((*result)==U61_PROTOCOL_FAILED)
    {
      U61_LOG_DEBUG("Network server protocol error on connection "
		    <<new_connection
		    <<" at stage "
		    <<stage);

      if (stage!=U61_PROTOCOL_BEGIN)
	{
	  /*
	   * For all the errors except the impossibility to bind the
	   * listening socket (which can only happen at stage 
	   * U61_PROTOCOL_BEGIN), we decide to mask the error and
	   * replace it by U61_PROTOCOL_LATER. This is because we do not
	   * want the whole server connections to be closed only because
	   * of a single little problem on one channel...
	   */
	  (*result)=U61_PROTOCOL_LATER;
	}

      if (new_connection>=0 && connection[new_connection]!=NULL)
	{
	  delete connection[new_connection];
	  connection[new_connection]=NULL;
	  new_connection=-1;
	}

      if (! listening_sock.is_alive())
	{
	  /*
	   * session is NULL, this means we have not successfully passed
           * the opening of the listening socket.
           * so we restart everything from scratch
           */    
	  stage=U61_PROTOCOL_BEGIN;
	}
      else
	{
	  /*
	   * session is not NULL, this means the protocol failed somewhere
	   * on an opened socket, therefore we go back to LISTENING mode
	   * ie accept new connection on the listening socket
	   */
	  stage=U61_SERVERPROTOCOL_LISTENING;
	}
    }
}


