/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   socket.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: wrapper on the ClanLib socket object, with error checks
 *              adapted to the way U61 works.
 */


/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include <string.h>

#ifdef U61_DEF_WIN32
#include <winsock.h>
#include "socketwindows.h"
#endif
#ifdef U61_DEF_UNIX
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#endif

#include "socket.h"
#include "log.h"
#include "macro.h"
#include "dns.h"
#include "system.h"
#include "global.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

#define U61_SOCKET_NB_BACKLOG    10
#define U61_SOCKET_ACCEPT_SEC     0
#define U61_SOCKET_ACCEPT_USEC    0           
#define U61_SOCKET_SEND_SEC       0
#define U61_SOCKET_SEND_USEC      0
#define U61_SOCKET_RECV_SEC       0
#define U61_SOCKET_RECV_USEC      0
#define U61_SOCKET_BUFFER_SIZE  512

#define U61_SOCKET_CONSOLE_DUMP_SIZE 20

/*---------------------------------------------------------------------------
  macros
  ---------------------------------------------------------------------------*/

#ifndef MIN
#define MIN(a, b)  (((a) < (b)) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a, b)  (((a) > (b)) ? (a) : (b))
#endif

/*---------------------------------------------------------------------------
  variants
  ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * Creates a socket
 */
U61_Socket::U61_Socket()
{
  sock = U61_SOCKET_INVALID;
  close_on_delete = true;

  close();
}

/*--------------------------------------------------------------------------*/
/*
 * Destroys a socket 
 */
U61_Socket::~U61_Socket()
{
  if (close_on_delete)
    {
      close();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Connects a socket on a server
 */
bool U61_Socket::connect(char *host, int port)
{
  bool result=false;
  struct sockaddr_in name;
  int enable=1;
  int disable=0;
  struct linger li;

  /*
   * We close the socket to clear stuff before opening
   * a new connection
   */
  close();

  sock=socket(AF_INET, SOCK_STREAM, 0);
  if (sock>=0)
    {
      name.sin_family = AF_INET;
      name.sin_addr.s_addr = INADDR_ANY;
      name.sin_port = 0;
      if (bind(sock, (struct sockaddr *) &name, sizeof name) >= 0) 
	{
	  name.sin_family = AF_INET;

	  U61_MACRO_STRCPY(sock_host,host);
	  if (U61_Dns::is_ip(host)) 
	    {
	      /* 
	       * host is an IP address, no DNS lookup
	       */
	      U61_MACRO_STRCPY(sock_ip,host);
	    } 
	  else 
	    {
	      /*
	       * DNS lookup, if it fails, the program will stop
	       * later when inet_aton will fail...
	       */
	      U61_Dns::get_ip_text(sock_ip, host);
	    }
	  sock_port=port;

	  if (inet_aton(sock_ip,&name.sin_addr)!=0)
	    {
	      name.sin_port = htons(port);
	      if (::connect(sock,(struct sockaddr *) &name, sizeof name)>=0)
		{
		  /*
		   * Added this code copied/paste from accept.
		   * don'tknow if it's usefull
		   */
		  li.l_onoff=0;
		  li.l_linger=0;
		  setsockopt(sock,SOL_SOCKET,SO_KEEPALIVE,
			     (char *) &enable,sizeof(int));
		  setsockopt(sock,SOL_SOCKET,SO_OOBINLINE,
			     (char *) &disable,sizeof(int));
		  setsockopt(sock,SOL_SOCKET,SO_LINGER,
			     (char *) &li,sizeof(struct linger));
		  
#ifdef WIN32
		  ioctlsocket(sock, FIONBIO, (unsigned long *) &enable);
#else
		  fcntl(sock, F_SETFL, O_NONBLOCK, 0);
#endif
		  mode = U61_SOCKET_MODE_CONNECTED;
		  result=true;
		}
	    }
	}
    }

  alive = result;
  if (result)
    {
      U61_LOG_MESSAGE("Connection on \""<<host<<":"<<sock_port<<"\".");
    }
  else
    {
      U61_LOG_WARNING("Unable to connect on \""<<host<<":"<<port<<"\".");
      sock = U61_SOCKET_INVALID;
      close();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Listens for connections on a given port
 */
bool U61_Socket::listen(int port)
{
  bool result = false;
  struct sockaddr_in name;
  int enable=1;

  /*
   * We close the socket to clear stuff before listening
   */
  close();

  sock=socket(AF_INET, SOCK_STREAM, 0);
  if (sock>=0)
    {
      setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,
		 (char *) &enable,sizeof(int));

      name.sin_family = AF_INET;
      name.sin_addr.s_addr = INADDR_ANY;
      name.sin_port = htons(port);
      if (bind(sock, (struct sockaddr *) &name, sizeof name)>=0) 
	{
	  if (::listen(sock,U61_SOCKET_NB_BACKLOG)>=0)
	    {
	      mode = U61_SOCKET_MODE_LISTENING;
	      U61_MACRO_STRCPY(sock_host,"");
	      U61_MACRO_STRCPY(sock_ip,"");
	      sock_port = port;	     

	      result=true;
	    }
	}
    }

  alive = result;
  if (result)
    {
      U61_LOG_MESSAGE("Listening on port "<<sock_port<<".");
    }
  else
    {
      U61_LOG_WARNING("Unable to listen on port "<<port<<".");
      sock = U61_SOCKET_INVALID;
      close();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns a valid socket object if a client has just connected. One
 * must call listen before.
 */
U61_Socket *U61_Socket::accept()
{
  U61_Socket *result=NULL;
  struct sockaddr_in name;
  socklen_t namelen=sizeof(struct sockaddr_in);
  fd_set read;
  struct timeval tv;
  int res;
  int enable=1;
  int disable=0;
  struct linger li;
  int accepted_sock;

  if (alive && sock>=0)
    {
      FD_ZERO(&read);
      FD_SET(sock,&read);
      tv.tv_sec=U61_SOCKET_ACCEPT_SEC;
      tv.tv_usec=U61_SOCKET_ACCEPT_USEC;
      res=select(sock+1,&read,NULL,NULL,&tv);
      if (res>=1)
	{
	  accepted_sock=::accept(sock,(struct sockaddr *) &name,&namelen);
	  if (accepted_sock>=0)
	    {
	      li.l_onoff=0;
	      li.l_linger=0;
	      setsockopt(accepted_sock,SOL_SOCKET,SO_KEEPALIVE,
			 (char *) &enable,sizeof(int));
	      setsockopt(accepted_sock,SOL_SOCKET,SO_OOBINLINE,
			 (char *) &disable,sizeof(int));
	      setsockopt(accepted_sock,SOL_SOCKET,SO_LINGER,
			 (char *) &li,sizeof(struct linger));

#ifdef WIN32
	      ioctlsocket(accepted_sock, FIONBIO, (unsigned long *) &enable);
#else
	      fcntl(accepted_sock, F_SETFL, O_NONBLOCK, 0);
#endif

	      result=new U61_Socket();
	      if (result) 
		{
		  result->alive = true;
		  result->mode = U61_SOCKET_MODE_CONNECTED;
		  result->sock = accepted_sock;
		  U61_MACRO_STRNCPY(result->sock_host, inet_ntoa(name.sin_addr), U61_CONST_STRING_SIZE);
		  U61_MACRO_STRNCPY(result->sock_ip, inet_ntoa(name.sin_addr), U61_CONST_STRING_SIZE);
		  result->sock_port = (int) ntohs(name.sin_port);
		}
	      else
		{
		  U61_LOG_WARNING("Unable to create socket object.");
		}
	    }
	}
    }

  if (result)
    {
      U61_LOG_MESSAGE("Connection on port "<<sock_port<<" from \""<<result->sock_ip<<":"<<result->sock_port<<"\".");
    }
  else
    {
      U61_LOG_DEBUG("No new connection on port "<<sock_port<<".");
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends a buffer on the network. Returns the number of bytes sent,
 * or -1 if the socket is closed.
 */
int U61_Socket::send_buf(char *buf, int size)
{
  int result=0;
  fd_set write;
  struct timeval tv;
  int res;
  int sent=0;
#ifdef WIN32
  int winerr;
#endif

  if (alive)
    {
      FD_ZERO(&write);
      FD_SET(sock,&write);
      tv.tv_sec=U61_SOCKET_SEND_SEC;
      tv.tv_usec=U61_SOCKET_SEND_USEC;
      
      errno=0;
      res=select(sock+1,NULL,&write,NULL,&tv);
      
      switch (res)
	{
	case -1:
#ifdef WIN32
	  winerr=WSAGetLastError();
	  if (winerr!=WSAEINTR && winerr!=WSAENOBUFS)
	    {
	      sent=-1;
	    }
#else
	  if (errno!=EINTR && errno!=ENOBUFS)
	    {
	      sent = -1;
	    }
#endif
	  break;
	case 1:
	  if (FD_ISSET(sock,&write))
	    {
	      sent=send(sock,
			buf,
			MIN(size,U61_SOCKET_BUFFER_SIZE),
			0);

	      if (sent == 0)
		{
		  sent = -1;
		}
	    }
	  break;
	default:
	  break;
	}
    }
  else
    {
      result=1;
    }

  if (sent<0)
    {
      result = -1;
      close();
    }
  else
    {
      U61_Global::system_info.register_bytes_out(sent);
      console_dump(buf,sent,"Network send ",".");

      result = sent;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives a buffer from the network. Returns the number of bytes received,
 * or -1 if the socket is closed.
 */
int U61_Socket::recv_buf(char *buf, int size, bool peek)
{
  int result=0;
  int res;
  fd_set read;
  struct timeval tv;
  int received=0;
#ifdef WIN32
  int winerr;
#endif

  memset(buf,0,size);

  if (alive)
    {
      FD_ZERO(&read);
      FD_SET(sock,&read);
      tv.tv_sec=U61_SOCKET_RECV_SEC;
      tv.tv_usec=U61_SOCKET_RECV_USEC;

      errno=0;
      res=select(sock+1,&read,NULL,NULL,&tv);
      
      switch (res)
	{
	case -1:
#ifdef WIN32
	  winerr=WSAGetLastError();
	  if (winerr!=WSAEINTR)
	    {
	      received=-1;
	    }
#else
	  if (errno!=EINTR)
	    {
	      received=-1;
	    }
#endif
	  break;
	case 1:
	  if (FD_ISSET(sock,&read))
	    {
	      received=recv(sock,
			    buf,
			    MIN(size, U61_SOCKET_BUFFER_SIZE),
			    peek ? MSG_PEEK : 0);
	      if (received == 0)
		{
		  received = -1;
		}
	    }
	  break;
	default:
	  break;
	}
    }
  else
    {
      result=-1;
    }

  if (received<0)
    {
      result = -1;
      close();
    }
  else
    {
      if (!peek)
	{
	  U61_Global::system_info.register_bytes_in(received);
	  console_dump(buf,received,"Network recv ",".");
	}

      result = received;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends a string on the network. Returns true if *all* the string has been sent.
 * This is a blocking call.
 */
bool U61_Socket::send_str(char *buf, int size)
{
  bool result = true;
  int total_sent = 0;
  int sent;

  size=MIN((int) strlen(buf),size);

  while (result && total_sent<size)
    {
      sent = send_buf(buf+total_sent,size-total_sent);
      result = result && (sent>=0);
      if (sent > 0)
	{
	  total_sent += sent;
	}
    }

  result = result && (total_sent == size);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives a string from the network. Returns true when *all* the string has
 * been received. This is a blocking call.  Buf must be of a size of size+1;
 */
bool U61_Socket::recv_str(char *buf, int size)
{
  bool result = true;
  int total_received = 0;
  int received;

  memset(buf,0,size+1);

  while (result)
    {
      received = recv_buf(buf+total_received,size-total_received);
      result = result && (received>=0);

      if (received >=0 )
	{
	  total_received += received;
	}
    }

  result = (total_received > 0);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if a socket is alive (not closed)
 */
bool U61_Socket::is_alive()
{
  U61_LOG_DEBUG("socket::alive = "<<alive<<".");

  return alive;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the connection mode of the socket
 */
int U61_Socket::get_mode()
{
  int result=0;

  switch (mode) 
    {
    case U61_SOCKET_MODE_LISTENING:
    case U61_SOCKET_MODE_CONNECTED:
      result = mode;
      break;
    default:
      result = U61_SOCKET_MODE_CLOSED;
      break;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the socket handler
 */
int U61_Socket::get_sock()
{
  return sock;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the host the socket is connected to
 */
char *U61_Socket::get_host()
{
  return sock_host;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the IP address the socket is connected to
 */
char *U61_Socket::get_ip()
{
  return sock_ip;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the port the socket is connected to
 */
int U61_Socket::get_port()
{
  return sock_port;
}

/*--------------------------------------------------------------------------*/
/*
 * Closes a socket
 */
void U61_Socket::close()
{
  if (sock>=0)
    {
      if (alive)
	{
	  switch (mode)
	    {
	    case U61_SOCKET_MODE_LISTENING:
	      U61_LOG_MESSAGE("Stopping listen on port "<<sock_port<<".");
	      break;
	    case U61_SOCKET_MODE_CONNECTED:
	      U61_LOG_MESSAGE("Closing connection with \""<<sock_host<<":"<<sock_port<<"\".");
	      break;
	    }
	}

      if (shutdown(sock,2)!=0)
	{
	  /*
	   * We could print a warning here.
	   */
	}

#ifdef WIN32
      if (closesocket(sock)!=0)
#else
	if (::close(sock)!=0)
#endif
	  {
	    /*
	     * Strange error, was the socket really opened?
	     */
	  }
    }

  alive = false;
  mode = U61_SOCKET_MODE_CLOSED;
  sock = U61_SOCKET_INVALID;
  U61_MACRO_MEMSET0(sock_host);
  U61_MACRO_MEMSET0(sock_ip);
  sock_port=0;
}

/*--------------------------------------------------------------------------*/
/*
 * Disable the feature "close socket automatically when object is destroyed"
 * This function is usefull for the accept method of the bufferedsocket obj.
 */
void U61_Socket::disable_close_on_delete()
{
  close_on_delete=false;
}

/*--------------------------------------------------------------------------*/
/*
 * Dumps traffic on the console.
 */
void U61_Socket::console_dump(char *data, int size, char *prefix, char *suffix)
{
  int total_written;
  int written;
  int to_write;
  char hex_buffer[U61_SOCKET_CONSOLE_DUMP_SIZE*2+1];
  char str_buffer[U61_SOCKET_CONSOLE_DUMP_SIZE+1];
  bool do_dump=false;

#ifdef U61_DEF_DEBUG
  do_dump = true;
#endif
  if (U61_Global::config.network_console_dump)
    {
      do_dump = true;
    }

  if (do_dump)
    {
      for  (total_written=0; 
	    total_written < size; 
	    total_written+=U61_SOCKET_CONSOLE_DUMP_SIZE)
	{
	  U61_MACRO_MEMSET0(hex_buffer);
	  U61_MACRO_MEMSET0(str_buffer);
	  
	  to_write=size-total_written;
	  if (to_write>U61_SOCKET_CONSOLE_DUMP_SIZE)
	    {
	      to_write=U61_SOCKET_CONSOLE_DUMP_SIZE;
	    }
	  for (written=0; written<to_write; ++written)
	    {
	      unsigned int hex;
	      unsigned char c;
	  
	      c=(unsigned char) data[total_written+written];
	      hex=c;
	      if (c<32)
		{
		  c=32;
		}
	      
	      sprintf(hex_buffer+written*2,"%02x",hex);
	      str_buffer[written]=c;
	    }

	  if (U61_Global::config.network_console_dump)
	    {
	      U61_LOG_MESSAGE(prefix<<hex_buffer<<",\""<<str_buffer<<"\""<<suffix); 
	    }
	  else
	    {
	      U61_LOG_DEBUG(prefix<<hex_buffer<<",\""<<str_buffer<<"\""<<suffix); 
	    }
	}
    }  
}
