/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   square.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: a square is the basic atomic object in u61. it's about
 *              the very squares you see on the screen. a square can be
 *              contained in a block (when falling) or in a map (when stable)
 *              it provides an API to be drawn and is automatically 
 *              animated.
 *              the main attribute of a square is its color
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "square.h"
#include "global.h"
#include "time.h"
#include "utils.h"
#include "serial.h"
#include "log.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/
#define U61_SQUARE_EXPLOSION_LENGTH 50
#define U61_SQUARE_NB_BITMAP 16
#define U61_SQUARE_TIME_FACTOR 3

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * this integer is used to animate the squares
 */
int U61_Square::cycle=0;

/*--------------------------------------------------------------------------*/
/*
 * this boolean is used to animate the squares
 */
bool U61_Square::blink=false;
 

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default square
 */ 
U61_Square::U61_Square()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * frees the square
 */ 
U61_Square::~U61_Square()
{
    
}

/*--------------------------------------------------------------------------*/
/* 
 * sets square values to default
 */ 
void U61_Square::reset()
{
  color=0;
  state=false;
  cycle_offset=0;
  exploding=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * set the square as active
 */ 
void U61_Square::enable()
{
  state=true;
}

/*--------------------------------------------------------------------------*/
/* 
 * set the square as inactive
 */ 
void U61_Square::disable()
{
  state=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * returns true if the square is active
 */ 
bool U61_Square::is_enabled()
{
  return state;
}

/*--------------------------------------------------------------------------*/
/* 
 * returns true if the square is inactive
 */ 
bool U61_Square::is_disabled()
{
  return !state;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the type of the square
 */
void U61_Square::set_color(int c)
{
  if (c>=0 && c<U61_SQUARE_NB_COLORS)
    {
      color=c;
    }
  else
    {
      color=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the type of the square
 */
int U61_Square::get_color()
{
  return color;
}

/*--------------------------------------------------------------------------*/
/*
 * updates the cycle integer, which depends on current time 
 */
void U61_Square::update_cycle()
{
  int time;

  time=U61_Time::for_effect();
  cycle=(time/U61_SQUARE_TIME_FACTOR)%U61_SQUARE_NB_BITMAP;
  blink=((time/U61_SQUARE_TIME_FACTOR)/U61_SQUARE_NB_BITMAP)%2 ? true : false;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true or false with a rate that is synchronized with the "cycle"
 * Usefull when you want something to blink (even not a square)
 */
bool U61_Square::should_blink()
{
  return blink;
}

/*--------------------------------------------------------------------------*/
/*
 * Starts an explosion. From now the square will return when questionned
 * about if it's exploding or not, and it will be drawn with the
 * explosion bitmap.
 * It only leaves this state when the explosion is finished
 */ 
void U61_Square::begin_explosion()
{
  exploding=true;
  explosion_counter=false;
}

/*--------------------------------------------------------------------------*/
/*
 * This function forces an explosion to end, generating an explosion_ended
 * "event"
 */
void U61_Square::end_explosion()
{
  explosion_counter=U61_SQUARE_EXPLOSION_LENGTH-1;
}

/*--------------------------------------------------------------------------*/
/*
 * An exploding square has to be updated at each game cycle, for the
 * explosion to end when it's needed. It's a little tricky to manage
 * since graphical animations are usually based on the for_effect()
 * value, but here we want to use the for_logic() time.
 * It is assumed that this function is called for each square
 * at each game cycle.
 */
void U61_Square::update_explosion()
{
  if (exploding)
    {  
      explosion_counter++;
      /*
       * We test if the explosion is finished. A very important point
       * is to test if explosion_counter is greater than
       * EXPLOSION_LENGTH, and end the explosion if necessary.
       * Indeed, there *must* be a game cycle when explosion_counter
       * is equal to EXPLOSION_LENGTH-1 it's at this condition only
       * that Lua scripts callbacks we'll be able to handle explosions
       * in a correct manner.
       */
      if (explosion_counter>=U61_SQUARE_EXPLOSION_LENGTH)
	{
	  exploding=false;
	  explosion_counter=0;
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the explosion is just ended. This condition should be
 * checked for every square at every round, or one may miss the end
 * of an explosion, and some Lua scripts can be based on such an event
 */
bool U61_Square::is_explosion_ended()
{
  /*
   * We use EXPLOSION_LENGTH-1 as a test value, for we need this
   * situation to happen once and only once, and a test with
   * EXPLOSION_LENGTH would always return false (see update_explosion())
   */
  return explosion_counter==U61_SQUARE_EXPLOSION_LENGTH-1;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the explosion is just new. This condition should be
 * checked for every square at every round, or one may miss the beginning
 * of an explosion, and some Lua scripts can be based on such an event
 */
bool U61_Square::is_explosion_new()
{
  return explosion_counter==1;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the square is exploding
 */
bool U61_Square::is_exploding()
{
  return exploding;
}


/*--------------------------------------------------------------------------*/
/*
 * draws the square
 */
void U61_Square::draw(int x, int y,int size)
{
  if (state)
    {
      if (!exploding)
	{
	  U61_Global::data.square[color][size]->put_screen
	    (x,y,(cycle+cycle_offset)%U61_SQUARE_NB_BITMAP);    
	}
      else
	{
	  U61_Global::data.explosion[size]->put_screen
	    (x,y,(explosion_counter*U61_SQUARE_NB_BITMAP)
	     /U61_SQUARE_EXPLOSION_LENGTH);
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * draws a square with an anticipation look
 */
void U61_Square::draw_anticipation(int x, int y,int size)
{
  if (state)
    {
      U61_Global::data.anticipation[size]->put_screen
	(x,y,(cycle+cycle_offset)%U61_SQUARE_NB_BITMAP);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * draws a square with a curse look
 */
void U61_Square::draw_curse(int x, int y,int size)
{
  if (state)
    {
      U61_Global::data.curse[size]->put_screen
	(x,y,(cycle+cycle_offset)%U61_SQUARE_NB_BITMAP);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * draws a filler square
 */
void U61_Square::draw_filler(int x, int y,int size)
{
  if (state)
    {
      U61_Global::data.filler[size]->put_screen
	(x,y,(cycle+cycle_offset)%U61_SQUARE_NB_BITMAP);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Serializes a square into a byte strem
 */
bool U61_Square::serialize(unsigned char **buffer, unsigned char *buffer_limit)
{
  bool result=true;
  unsigned char *buffer_start=*buffer;

  result=result && U61_Serial::serialize_int(explosion_counter,buffer,buffer_limit);
  result=result && U61_Serial::serialize_int(color,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(state,buffer,buffer_limit);
  result=result && U61_Serial::serialize_bool(exploding,buffer,buffer_limit);

  if ((*buffer) - buffer_start != U61_SQUARE_SERIALIZED_SIZE)
    {
      U61_LOG_WARNING("Wrong buffer size for square serialization (diff="<<(buffer_start - (*buffer) + U61_SQUARE_SERIALIZED_SIZE)<<")!");
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Un-serializes a square from a byte strem
 */
bool U61_Square::unserialize(unsigned char **buffer, unsigned char *buffer_limit)
{
  bool result=true;
  unsigned char *buffer_start=*buffer;

  result=result && U61_Serial::unserialize_int(&explosion_counter,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_int(&color,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&state,buffer,buffer_limit);
  result=result && U61_Serial::unserialize_bool(&exploding,buffer,buffer_limit);

  if ((*buffer) - buffer_start != U61_SQUARE_SERIALIZED_SIZE)
    {
      U61_LOG_WARNING("Wrong buffer size for square unserialization (diff="<<(buffer_start - (*buffer) + U61_SQUARE_SERIALIZED_SIZE)<<")!");
    }

  return result;
}
