/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   systeminfo.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: used to calculate and display system information such
 *              as the number of frames per sec for instance
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/core.h>
#include <ClanLib/display.h>

#include "systeminfo.h"
#include "global.h"
#include "macro.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_SYSTEMINFO_DELAY U61_TIME_ONE_SECOND

#define U61_SYSTEMINFO_MAX_NETWORK_FLOW 10000

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a system info object
 */
U61_SystemInfo::U61_SystemInfo()
{
  start_time=0;

  idle_count=0;
  logic_count=0;
  display_count=0;

  last_idle_rate=0.;
  last_logic_per_sec=0;
  last_display_per_sec=0;
}

/*--------------------------------------------------------------------------*/
/*
 * register a call to sleep() to calculate idle time
 */
void U61_SystemInfo::register_idle()
{
  idle_count++;  
}

/*--------------------------------------------------------------------------*/
/*
 * registers a call to the logic() function
 */
void U61_SystemInfo::register_logic()
{
  logic_count++;
}

/*--------------------------------------------------------------------------*/
/*
 * registers a call to the display() function
 */
void U61_SystemInfo::register_display()
{
  display_count++;

  reset_if_needed();
}

/*--------------------------------------------------------------------------*/
/*
 * registers a call to the recv() function with n bytes received
 */
void U61_SystemInfo::register_bytes_in(int bytes)
{
  bytes_in_count+=bytes;
}

/*--------------------------------------------------------------------------*/
/*
 * registers a call to the send() function with n bytes sent
 */
void U61_SystemInfo::register_bytes_out(int bytes)
{
  bytes_out_count+=bytes;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the rate of idle loops in the game
 */
float U61_SystemInfo::get_idle_rate()
{
  return last_idle_rate;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of calls to logic() per second
 */
int U61_SystemInfo::get_logic_per_sec()
{

  return last_logic_per_sec;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of calls to display() per second
 */
int U61_SystemInfo::get_display_per_sec()
{
  return last_display_per_sec;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of bytes received per second
 */
int U61_SystemInfo::get_bytes_in_per_sec()
{
  int result;

  result=last_bytes_in_per_sec;

  if (result>=U61_SYSTEMINFO_MAX_NETWORK_FLOW)
    {
      result=U61_SYSTEMINFO_MAX_NETWORK_FLOW-1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of bytes sent per second
 */
int U61_SystemInfo::get_bytes_out_per_sec()
{
  int result;

  result=last_bytes_out_per_sec;

  if (result>=U61_SYSTEMINFO_MAX_NETWORK_FLOW)
    {
      result=U61_SYSTEMINFO_MAX_NETWORK_FLOW-1;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * registers a call to the display() function
 */
void U61_SystemInfo::reset_if_needed()
{
  unsigned int current_time;
  unsigned int elapsed_time;

  current_time=U61_Time::for_effect();
  elapsed_time=current_time-start_time;
  if (elapsed_time>U61_SYSTEMINFO_DELAY)
    {
      last_idle_rate=
	((float) idle_count*U61_TIME_IDLE_DELAY)/((float) elapsed_time);
      last_logic_per_sec=
	(2*logic_count*U61_TIME_ONE_SECOND+1)/(elapsed_time*2);
      last_display_per_sec=
	(2*display_count*U61_TIME_ONE_SECOND+1)/(elapsed_time*2);
      last_bytes_in_per_sec=
	(2*bytes_in_count*U61_TIME_ONE_SECOND+1)/(elapsed_time*2);
      last_bytes_out_per_sec=
	(2*bytes_out_count*U61_TIME_ONE_SECOND+1)/(elapsed_time*2);

      idle_count=0;
      logic_count=0;
      display_count=0;
      bytes_in_count=0;
      bytes_out_count=0;

      start_time=current_time;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * displays the information on the screen
 */
void U61_SystemInfo::draw()
{
  CL_Font *font;
  int height;
  int y;
  char buffer[U61_CONST_STRING_SIZE];

  font=U61_Global::data.font_info[U61_DATA_SIZE_SMALL];
  height=font->get_height();

  y=0;
  U61_MACRO_SPRINTF1(buffer,"frames/sec %4d",get_display_per_sec());
  font->print_right(U61_Global::data.screen_w,y,buffer);
  y+=height;
  U61_MACRO_SPRINTF1(buffer,"idle rate %4d",(int) (get_idle_rate()*100.));
  font->print_right(U61_Global::data.screen_w,y,buffer);
  y+=height;
  U61_MACRO_SPRINTF1(buffer,"bytes in/sec %4d",get_bytes_in_per_sec());
  font->print_right(U61_Global::data.screen_w,y,buffer);
  y+=height;
  U61_MACRO_SPRINTF1(buffer,"bytes out/sec %4d",get_bytes_out_per_sec());
  font->print_right(U61_Global::data.screen_w,y,buffer);
  y+=height;
  U61_MACRO_SPRINTF1(buffer,"rounds/sec %4d",get_logic_per_sec());
  font->print_right(U61_Global::data.screen_w,y,buffer);
}

