/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   textdisplayer.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: class used to display text on a bitmap. It knows how to
 *              wrap text, align it, and is heavily used by menu items.
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#ifdef U61_DEF_MSVC
#include <minmax.h>
#endif

#include "textdisplayer.h"
#include "letter.h"
#include "log.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * generation of a default menu item
 */ 
U61_TextDisplayer::U61_TextDisplayer(CL_Font *fnt)
{
  font=fnt;
  set_text("");
}

/*--------------------------------------------------------------------------*/
/* 
 * generation of a default menu item
 */ 
U61_TextDisplayer::U61_TextDisplayer(CL_Font *fnt, char *str)
{
  font=fnt;
  set_text(str);
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of the menu-item
 */ 
U61_TextDisplayer::~U61_TextDisplayer()
{

}

/*--------------------------------------------------------------------------*/
/*
 * draws text (for a menu item for instance)
 */
void U61_TextDisplayer::draw(int x, 
			     int y, 
			     int align, 
			     int w, 
			     int h)
{
  int dummy=0;

  draw_ex(true, &dummy,
	  x,y,
	  align,
	  w,h);
 }

/*--------------------------------------------------------------------------*/
/*
 * draws a menu item
 */
void U61_TextDisplayer::test_mouse(int x, 
			     int y, 
			     int align, 
			     int w, 
			     int h,
			     int mouse_x,
			     int mouse_y,
			     bool *mouse_in,
			     int *mouse_char)
{
  int dummy=0;

  draw_ex(false, &dummy,
	  x,y,
	  align,
	  w,h,
	  mouse_x,mouse_y,
	  mouse_in,
	  mouse_char);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the height of the menu item (the display is wrapped)
 */
int U61_TextDisplayer::get_height(int w)
{
  int result=0;

  draw_ex(false, &result,
	  0,0,
	  U61_TEXTDISPLAYER_ALIGN_LEFT,
	  w,U61_TEXTDISPLAYER_MAX_H);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Internal function used to draw the text. It's also used to calculate the
 * height of the text, this way we're sure the same algorithm is used when
 * pre-calculating height and when drawing for good.
 */
void U61_TextDisplayer::draw_ex(bool do_draw,
				int *total_height,
				int x, 
				int y, 
				int align, 
				int w, 
				int h,
				int mouse_x,
				int mouse_y,
				bool *mouse_in,
				int *mouse_char)
{
  int i=0;
  int nb_letter;
  int height=0;
  int height_char=0;
  int height_line=0;
  int width_char;
  int width_line=0;
  bool height_exceeded=false;
  int j0=0;
  int jn=0;
  int j0n=0;
  int j;
  bool draw_line;
  int dx=0;
  bool space_found;

  if (mouse_in)
    {
      *mouse_in = false;
    }
  if (mouse_char)
    {
      *mouse_char = -1;
    }

  nb_letter=get_text_size();
  while (i<nb_letter && !height_exceeded)
    {
      draw_line=false;

      height_char=text[i].get_height();
      width_char=text[i].get_width();
      if (width_line+width_char>w)
	{
	  if (height+height_line>h)
	    {
	      height_exceeded=true;
	    }
	  else
	    {
	      /*
	       * We go to the next line and draw the
	       * letter on this next line
	       */
	      draw_line=true;
	      j0n=jn=i;

	      /*
	       * Here's an ugly algorithm to "cut" lines
	       * where there are spaces preferrably to
	       * anywhere else.
	       */
	      space_found=false;
	      /*
	       * The j>j0 instead of j>=j0 is important here,
	       * for if they are only spaces at the beginning
	       * of the line, we want at least one of them
	       * to be drawn, otherwise we would risk an
	       * infinite loop.
	       */
	      for (j=jn-1;j>j0 && !space_found;--j)
		{
		  if (text[j].get_ascii()==' ')
		    {
		      jn=j;
		      j0n=j+1;
		      space_found=true;
		    }
		}
	    }
	}

      /*
       * If it's the last letter, we draw it too
       */
      if (i==nb_letter-1 && !height_exceeded)
	{
	  draw_line=true;
	  j0n=jn=nb_letter;
	}

      if (draw_line)
	{
	  /*
	   * We remove spaces at the beginning of the line
	   * 
	   * Note the j0<jn-1 for if there are only
	   * spaces on the line we want at least one
	   * space to be drawn for padding.
	   */
	  while (j0<jn-1 && text[j0].get_ascii()==' ')
	    {
	      j0++;
	    }

	  /*
	   * If it's the last letter, we draw it anyway,
	   * for if there are spaces at the end of the
	   * string, there must be a good reason for it.
	   */
	  if (i<nb_letter-1)
	    {	  
	      /*
	       * We remove spaces at the end of the line
	       * 
	       * Note the jn>j0=1 for if there are only
	       * spaces on the line we want at least one
	       * space to be drawn for padding.
	       */
	      while (jn>j0+1 && text[jn-1].get_ascii()==' ')
		{
		  jn--;
		}
	    }
	  
	  /*
	   * We recalculate the width of the line for
	   * removing spaces might have changed things
	   */
	  width_line=0;
	  for (j=j0;j<jn;++j)
	    {
	      width_line+=text[j].get_width();
	      height_line=max(height_line,height_char);
	    }
	  
	  switch (align)
	    {
	    case U61_TEXTDISPLAYER_ALIGN_LEFT:
	      dx=0;
	      break;
	    case U61_TEXTDISPLAYER_ALIGN_CENTER:
	      dx=(w-width_line)/2;
	      break;
	    case U61_TEXTDISPLAYER_ALIGN_RIGHT:
	      dx=w-width_line;
	      break;
	    default:
	      U61_LOG_DEBUG("Bad alignement: "<<align);
	      dx=0;
	      break;
	    }
	  
	  for (j=j0;j<jn;++j)
	    {
	      /*
	       * We try and check if mouse is in the zone
	       * being drawn
	       */
	      if (mouse_in && mouse_char)
		{
		  if (mouse_x >= x+dx &&
		      mouse_x < x+dx+text[j].get_width() &&
		      mouse_y >= y+height &&
		      mouse_y < y+height+text[j].get_height())
		    {
		      U61_LOG_DEBUG("mouse in char, mouse_x="<<mouse_x<<", mouse_y="<<mouse_y<<", char="<<((char) text[j].get_ascii()));
		      *mouse_in = true;
		      *mouse_char = j;
		    } 							    
		}

	      /*
	       * Only draw stuff if do_draw is set, this is usefull
	       * for this way we can also use the function to
	       * calculate the height of the displayer only.
	       */
	      if (do_draw)
		{
		  text[j].draw(x+dx,y+height);
		}
	      dx+=text[j].get_width();
	    }	      
	  
	  height+=height_line;

	  j0=j0n;
	  height_line=0;
	  width_line=0;

	  /*
	   * We recalculate line width/height for all the chars we finally
	   * did not draw (mostly because there was a space between)
	   */
	  for (j=j0;j<i;++j)
	    {
	      width_line += text[j].get_width();
	      height_line = max(height_line,text[j].get_height());
	    }
	}

      width_line+=width_char;

      ++i;
    }

  (*total_height)=height;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the width of the menu item
 */
int U61_TextDisplayer::get_width(int w)
{
  int i;
  int nb_letter;
  int width=0;

  nb_letter=get_text_size();
  for (i=0;i<nb_letter && width<w;++i)
    {
      width+=text[i].get_width();
    }

  width=min(w,width);

  return width;
}

/*--------------------------------------------------------------------------*/
/*
 * makes a letter enter dancing state
 */
void U61_TextDisplayer::enable_dance(int i)
{
  if (i>=0 && i<get_text_size())
    {
      text[i].enable_dance();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * makes a letter leave dancing state
 */
void U61_TextDisplayer::disable_dance(int i)
{
  if (i>=0 && i<get_text_size())
    {
      text[i].disable_dance();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * makes a letter enter blinking state
 */
void U61_TextDisplayer::enable_blink(int i)
{
  if (i>=0 && i<get_text_size())
    {
      text[i].enable_blink();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * makes a letter leave blinking state
 */
void U61_TextDisplayer::disable_blink(int i)
{
  if (i>=0 && i<get_text_size())
    {
      text[i].disable_blink();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the text of the menu item
 */
void U61_TextDisplayer::set_text(char *str)
{
  text.clear();

  /*
   * we fill the vector of letters with the chars of the string
   * passed in arg.
   */
  while(*str)
    {
      text.push_back(U61_Letter(font,*str));
      str++;
    } 
}

/*--------------------------------------------------------------------------*/
/*
 * puts the text in a string buffer. buf must contain at least max+1
 * characters to store the trailing '\0'.
 */
void U61_TextDisplayer::put_text(char *buf, int max)
{
  int i;
  int nb_letter;
  char c;
    
  nb_letter=get_text_size();
  for (i=0;i<nb_letter && i<max;++i)
    {
      c=text[i].get_ascii();
      buf[i]=c;
    }
  buf[i]=0;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the size of the text
 */
int U61_TextDisplayer::get_text_size()
{
  int size=0;

  size=text.size();

  return size;
}

