/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   wwwlist.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: this class allows U61 clients to query the meta-server
 *              and get the list of available servers.
 */



/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include "wwwlist.h"
#include "http.h"
#include "macro.h"
#include "time.h"
#include "log.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

#define U61_WWWLIST_DELAY_IN_SECONDS 150

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a 'www list' instance
 */ 
U61_WwwList::U61_WwwList(char *g, char *v, char *h, char *u, int p)
{
  searching = false;
  thread = NULL;

  reset();

  set_game(g);
  set_version(v);
  set_metaserver_host(h);
  set_metaserver_url(u);
  set_metaserver_port(p);
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a 'www list' instance
 */ 
U61_WwwList::~U61_WwwList()
{
  stop_thread();
}

/*--------------------------------------------------------------------------*/
/* 
 * resets a 'www list' instance
 */ 
void U61_WwwList::reset()
{
  stop_thread();
  remove_servers();

  /*
   * We fill all these strings with zeros, for we can't garantee that
   * they won't be written while another thread is reading them.
   * Filling with zeros will avoid runaway string functions, even
   * if it can't prevent some errors - a few errors here is not that
   * bad, since registering on the meta-server is not a critical task.
   */
  U61_MACRO_MEMSET0(game);
  U61_MACRO_MEMSET0(version);
  U61_MACRO_MEMSET0(metaserver_host);
  U61_MACRO_MEMSET0(metaserver_url);

  set_game(U61_CONST_PROGRAM);
  set_version(U61_DEF_VERSION);
  set_metaserver_host(U61_WWWREGISTER_METASERVER_DEFAULT_HOST);
  set_metaserver_url(U61_WWWREGISTER_METASERVER_DEFAULT_URL);
  set_metaserver_port(U61_WWWREGISTER_METASERVER_DEFAULT_PORT);

  last_execute_time=0;
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "game" parameter
 */ 
void U61_WwwList::set_game(char *g)
{
  U61_MACRO_STRCPY(game,g);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "version" parameter
 */ 
void U61_WwwList::set_version(char *v)
{
  U61_MACRO_STRCPY(version,v);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "metaserver host" parameter
 */ 
void U61_WwwList::set_metaserver_host(char *h)
{
  U61_MACRO_STRCPY(metaserver_host,h);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "metaserver url" parameter
 */ 
void U61_WwwList::set_metaserver_url(char *u)
{
  U61_MACRO_STRCPY(metaserver_url,u);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "metaserver port" parameter
 */ 
void U61_WwwList::set_metaserver_port(int p)
{
  metaserver_port=p;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns true if a search is active in a separate thread.
 */ 
bool U61_WwwList::is_searching()
{
  return searching;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the number of server founds
 */ 
int U61_WwwList::get_size()
{
  int size=0;

  /*
   * We test searching which might be changed in a separate thread but
   * we shouldm't rick anything since the other thread can only change
   * the value to false after everything is done - this garantees
   * the servers vector won't be modified after we do the test.
   */
  if (!searching)
    {
      size=servers.size();
    }

  return size;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns informations about a given server
 */ 
U61_WwwListItem *U61_WwwList::get_item(int i)
{
  U61_WwwListItem *item=NULL;

  /*
   * We test searching which might be changed in a separate thread but
   * we shouldm't rick anything since the other thread can only change
   * the value to false after everything is done - this garantees
   * the servers vector won't be modified after we do the test.
   */
  if (!searching && i>=0 && i<get_size())
    {
      item=servers[i];
    }

  return item;
}

/*--------------------------------------------------------------------------*/
/* 
 * Function which launches a search for servers in a separate thread
 */ 
bool U61_WwwList::search(bool force)
{
  bool result=false;
  unsigned int current_time = U61_Time::for_effect()/U61_TIME_ONE_SECOND;

  if (!searching)
    {
      U61_LOG_DEBUG("Getting server list from meta-server, current_time="<<
		    current_time<<
		    ", last_execute_time="<<
		    last_execute_time<<
		    ", force="<<
		    force);
      
      /*
       * We stop the old thread which is useless now...
       */
      stop_thread();

      if (force || last_execute_time + U61_WWWLIST_DELAY_IN_SECONDS < current_time)
	{
	  try 
	    {
	      remove_servers();

	      /*
	       * we must return true if we have cleared the server list for
	       * this will inform the calling menu that items have been freed.
	       */
	      result = true;
	      searching = true;

	      thread=CL_Thread::create(search_callback, (void *) this);
	      if (thread)
		{
		  thread->start();
		  U61_LOG_DEBUG("U61_WwwList thread started");
		  last_execute_time = current_time;
		}
	      else
		{
		  U61_LOG_ERROR("Could not create CL_Thread object");
		  stop_thread();
		}
	    }
	  catch (CL_Error e)
	    {
	      U61_LOG_WARNING("Could not launch U61_WwwList thread");
	      stop_thread();
	    }
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Function called in a separate threads and queries the meta-server
 * about existing servers.
 */ 
int U61_WwwList::search_callback(void *data)
{
  int result;
  bool ok=false;
  U61_WwwList *www_list;
  char request[U61_CONST_STRING_SIZE];
  char buffer[U61_CONST_STRING_SIZE];

  U61_LOG_DEBUG("Game list thread started");

  www_list=(U61_WwwList *) data;

  U61_LOG_MESSAGE("Listing games on \""<<www_list->metaserver_host<<"\".");

  U61_MACRO_SPRINTF4(request,
		     "%s:%d%slist.php3?protocol=%s",
		     www_list->metaserver_host,
		     www_list->metaserver_port,
		     www_list->metaserver_url,
		     U61_WWWREGISTER_METASERVER_PROTOCOL);

  U61_MACRO_SPRINTF2(buffer,
		     "&game=%s&version=%s",
		     www_list->game,
		     www_list->version);
  U61_MACRO_STRCAT(request,buffer);

  U61_MACRO_STRCAT(request,"\n");

  U61_Http http(www_list->metaserver_host,www_list->metaserver_port,request);

  U61_LOG_DEBUG("HTTP request = "<<request);

  if (http.execute())
    {
      U61_LOG_DEBUG("HTTP answer = "<<http.get_answer());

      char *str=http.get_answer();
      
      ok = www_list->parser_read_page(&str);
    }

  result=ok ? 0 : 1;

  www_list->searching = false;

  U61_LOG_DEBUG("Game list thread finished");

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * stops the thread launched by the instance
 */
void U61_WwwList::stop_thread()
{
  if (thread)
    {
      U61_LOG_DEBUG("Stopping meta-server search thread");

      //#ifdef U61_DEF_WIN32
      /*
       * Threads suck on Win98, so we disable the "force terminate"
       * feature on this platform and wait for the thread to
       * terminate alone instead
       */
      //      thread->wait();
      //#else
      thread->terminate();
      //#endif

      U61_LOG_DEBUG("Game list thread stopped");

      delete thread;
      thread = NULL;
    }

  searching = false;
}

/*--------------------------------------------------------------------------*/
/* 
 * Cancels the current search
 */ 
void U61_WwwList::cancel()
{
  stop_thread();
  remove_servers();
}

/*--------------------------------------------------------------------------*/
/* 
 * Removes all the servers from the list
 */
void U61_WwwList::remove_servers()
{
  int i;
  int nb_server;

  nb_server=servers.size();
  for (i=0;i<nb_server;++i)
    {
      delete servers[i];
    }

  servers.clear();
}

/*--------------------------------------------------------------------------*/
/* 
 * Parses a page received from the meta-server
 */
bool U61_WwwList::parser_read_page(char **str)
{
  bool ok = true;
  char *tmp;

  tmp=*str;

  while (U61_Http::parser_read_eol(str));

  while (strlen(*str)>0)
    {
      tmp=*str;
      if (U61_Http::parser_read_eol(&tmp))
	{
	  /*
	   * This was just an empty line...
	   */
	  *str=tmp;
	}
      else
	{
	  /*
	   * We parse the line since there's something on it
	   */
	  ok = ok && parser_read_line(str);
	}
    }

  return ok;
}
 
/*--------------------------------------------------------------------------*/
/* 
 * Parses a line of a page received from the meta-server
 */
bool U61_WwwList::parser_read_line(char **str)
{
  bool ok = false;

  char temp[U61_CONST_STRING_SIZE];

  char address[U61_CONST_STRING_SIZE];
  int port;
  int uptime;
  int busy_players;
  int max_players;
  int password;
  char comment[U61_CONST_STRING_SIZE];

  if (U61_Http::parser_read_string(address,str,sizeof(address)) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_int(&port,str) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_string(temp,str,sizeof(temp)) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_string(temp,str,sizeof(temp)) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_int(&uptime,str) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_int(&busy_players,str) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_int(&max_players,str) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_int(&password,str) &&
      U61_Http::parser_read_separator(str) &&
      U61_Http::parser_read_string(comment,str,sizeof(comment)))
    {
      /*
       * We skip full servers for they are of no use since we can't
       * connect on them
       */
      if (busy_players<max_players)
	{
	  /*
	   * Here we create a www list item instance to hold the data
	   * we've just parsed
	   */
	  U61_WwwListItem *item;

	  item = new U61_WwwListItem(address,
				     port,
				     uptime,
				     busy_players,
				     max_players,
				     password ? true : false,
				     comment); 
	  if (item)
	    {
	      servers.push_back(item);
	    }
	  else
	    {
	      U61_LOG_ERROR("Unable to allocate memory for U61_WwwListItem object");
	    }
	}

      ok = true;
    }

  U61_Http::parser_next_line(str);

  return ok;
}
