/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   wwwregister.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: this class allows U61 servers to register themselves on
 *              the meta-server.
 */



/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include <string.h>

#include "wwwregister.h"
#include "layout.h"
#include "macro.h"
#include "log.h"
#include "time.h"
#include "http.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

#define U61_WWWREGISTER_DELAY_IN_SECONDS 90

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a 'www register' instance
 */ 
U61_WwwRegister::U61_WwwRegister(char *g, char *v, char *h, char *u, int p)
{
  threads.clear();

  reset();

  set_game(g);
  set_version(v);
  set_metaserver_host(h);
  set_metaserver_url(u);
  set_metaserver_port(p);
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a 'www register' instance
 */ 
U61_WwwRegister::~U61_WwwRegister()
{
  stop_threads();
}

/*--------------------------------------------------------------------------*/
/* 
 * resets a 'www register' instance
 */ 
void U61_WwwRegister::reset()
{
  stop_threads();

  /*
   * We fill all these strings with zeros, for we can't garantee that
   * they won't be written while another thread is reading them.
   * Filling with zeros will avoid runaway string functions, even
   * if it can't prevent some errors - a few errors here is not that
   * bad, since registering on the meta-server is not a critical task.
   */
  U61_MACRO_MEMSET0(game);
  U61_MACRO_MEMSET0(version);
  U61_MACRO_MEMSET0(metaserver_host);
  U61_MACRO_MEMSET0(metaserver_url);
  U61_MACRO_MEMSET0(comment);

  set_game(U61_CONST_PROGRAM);
  set_version(U61_DEF_VERSION);
  set_metaserver_host(U61_WWWREGISTER_METASERVER_DEFAULT_HOST);
  set_metaserver_url(U61_WWWREGISTER_METASERVER_DEFAULT_URL);
  set_metaserver_port(U61_WWWREGISTER_METASERVER_DEFAULT_PORT);
  set_privacy(U61_WWWREGISTER_PRIVACY_PUBLIC);
  set_port(0);
  set_busy_players(0);
  set_max_players(U61_LAYOUT_NB_PLAYERS_TOTAL);
  set_password(false);
  set_comment("");

  last_execute_time=0;
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "game" parameter
 */ 
void U61_WwwRegister::set_game(char *g)
{
  U61_MACRO_STRCPY(game,g);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "version" parameter
 */ 
void U61_WwwRegister::set_version(char *v)
{
  U61_MACRO_STRCPY(version,v);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "metaserver host" parameter
 */ 
void U61_WwwRegister::set_metaserver_host(char *h)
{
  U61_MACRO_STRCPY(metaserver_host,h);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "metaserver url" parameter
 */ 
void U61_WwwRegister::set_metaserver_url(char *u)
{
  U61_MACRO_STRCPY(metaserver_url,u);
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "metaserver port" parameter
 */ 
void U61_WwwRegister::set_metaserver_port(int p)
{
  metaserver_port=p;
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "privacy" parameter
 */ 
void U61_WwwRegister::set_privacy(int p)
{
  switch (p)
    {
    case U61_WWWREGISTER_PRIVACY_PRIVATE:
    case U61_WWWREGISTER_PRIVACY_PUBLIC:
      privacy=p;
      break;
    default:
      privacy=U61_WWWREGISTER_PRIVACY_PUBLIC;
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "port" parameter
 */ 
void U61_WwwRegister::set_port(int p)
{
  port=p;
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "busy players" parameter
 */ 
void U61_WwwRegister::set_busy_players(int bp)
{
  busy_players=bp;
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "max players" parameter
 */ 
void U61_WwwRegister::set_max_players(int mp)
{
  max_players=mp;
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "password" parameter
 */ 
void U61_WwwRegister::set_password(bool p)
{
  password=p;
}

/*--------------------------------------------------------------------------*/
/* 
 * sets the "comment" parameter
 */ 
void U61_WwwRegister::set_comment(char *c)
{
  U61_MACRO_STRCPY(comment,c);
}

/*--------------------------------------------------------------------------*/
/* 
 * Registers the game on the meta-server
 */ 
bool U61_WwwRegister::execute(bool force)
{
  bool result=false;
  unsigned int current_time = U61_Time::for_effect()/U61_TIME_ONE_SECOND;

  U61_LOG_DEBUG("Registering on meta-server, current_time="<<
		current_time<<
		", last_execute_time="<<
		last_execute_time);

  if (force || last_execute_time + U61_WWWREGISTER_DELAY_IN_SECONDS < current_time)
    {
      try 
	{
	  CL_Thread *thread=CL_Thread::create(execute_callback, (void *) this);
	  if (thread)
	    {
	      thread->start();
	      threads.push_back(thread);
	      U61_LOG_DEBUG("U61_WwwRegister thread started");
	      result=true;
	      last_execute_time = current_time;
	    }
	  else
	    {
	      U61_LOG_ERROR("Could not create CL_Thread object");
	    }
	}
      catch (CL_Error e)
	{
	  U61_LOG_WARNING("Could not launch U61_WwwRegister thread");
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Callback called in a separate thread by the "execute" function
 */ 
int U61_WwwRegister::execute_callback(void *data)
{
  int result;
  bool ok=false;
  U61_WwwRegister *www_register;
  char request[U61_CONST_STRING_SIZE];
  char buffer[U61_CONST_STRING_SIZE];
  char comment[U61_CONST_STRING_SIZE];

  U61_LOG_DEBUG("Game register thread started");

  www_register=(U61_WwwRegister *) data;

  U61_LOG_MESSAGE("Registering game on \""<<www_register->metaserver_host<<"\".");

  U61_MACRO_SPRINTF4(request,
		     "%s:%d%sregister.php3?protocol=%s",
		     www_register->metaserver_host,
		     www_register->metaserver_port,
		     www_register->metaserver_url,
		     U61_WWWREGISTER_METASERVER_PROTOCOL);

  U61_MACRO_SPRINTF6(buffer,
		     "&game=%s&version=%s&port=%d&busy_players=%d&max_players=%d&password=%d",
		     www_register->game,
		     www_register->version,
		     www_register->port,
		     www_register->busy_players,
		     www_register->max_players,
		     www_register->password);
  U61_MACRO_STRCAT(request,buffer);

  U61_Http::format_url_param(comment,www_register->comment);
  U61_MACRO_SPRINTF1(buffer,
		     "&comment=%s",
		     comment)
  U61_MACRO_STRCAT(request,buffer);

  U61_MACRO_STRCAT(request,"\n");

  U61_Http http(www_register->metaserver_host,www_register->metaserver_port,request);

  U61_LOG_DEBUG("HTTP request = "<<request);

  if (http.execute())
    {
      U61_LOG_DEBUG("HTTP answer = "<<http.get_answer());

      if (strstr(http.get_answer(),U61_HTTP_ANSWER_OK))
	{
	  ok=true;
	}
    }

  result=ok ? 0 : 1;

  U61_LOG_DEBUG("Game register thread finished");

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * stops all the threads launched by the instance
 */
void U61_WwwRegister::stop_threads()
{
  CL_Thread *thread;

  U61_LOG_DEBUG("Stopping meta-server register threads ("<<threads.size()<<" items)");

  while (!threads.empty())
    {
      thread=threads.front();
      threads.pop_front();
      //#ifdef U61_DEF_WIN32
      /*
       * Threads suck on Win98, so we disable the "force terminate"
       * feature on this platform and wait for the thread to
       * terminate alone instead
       */
      //      thread->wait();
      //#else
      thread->terminate();
      //#endif

      U61_LOG_DEBUG("Game register thread stopped");

      delete thread;
    }
}


