
/*
 * cache.c                                                      (jh,05.02.2006)
 */

/*
 *  unpackfs: filesystem with transparent unpacking of archive files
 *  Copyright (C) 2005, 2006  Jochen Hepp <jochen.hepp@gmx.de>
 *
 *  This file is part of unpackfs.
 *
 *  unpackfs is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  unpackfs is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#define _GNU_SOURCE
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif
#include <unpackfs.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif
#ifdef HAVE_GLOB_H
#	include <glob.h>
#endif
#ifdef HAVE_MAGIC_H
#	include <magic.h>
#endif


/*
 * cache_magic - unpack magicly from root_path.* or cache_path.* to cache_path
 *
 * in_cache: zero (false) to search for magic files with glob() in root_path
 *           -1   (true)  to search for magic files with glob() in cache_path
 */
static int cache_magic(const char *root_path, const char *cache_path,
                       int in_cache)
{
	int res, found;
	struct unpackfs_magic_unpack_info *packer_info;
	magic_t magic;
	char *cachefile, **filev, *file, *tmp;
	size_t root_path_len, cache_path_len, path_len, ext_len_max, ext_len;
	glob_t search;
	struct stat statbuf;

	res = 0;

	packer_info = unpackfs_packer_info_get();
	magic = unpackfs_magic_get();

	/* constant 2 is strlen(".d") and constant 7 is strlen(".unpack") */

	root_path_len = strlen(root_path);

	if (root_path_len > 2 && *(root_path + root_path_len - 2 - 1) != '/' &&
	    strncmp(root_path + root_path_len - 2, ".d", 2) == 0)
		root_path_len -= 2;

	if (root_path_len > 7 && *(root_path + root_path_len - 7 - 1) != '/' &&
	    strncmp(root_path + root_path_len - 7, ".unpack", 7) == 0)
		root_path_len -= 7;

	cache_path_len = strlen(cache_path);

	if (cache_path_len > 2 && *(cache_path + cache_path_len - 2 - 1) != '/' &&
	    strncmp(cache_path + cache_path_len - 2, ".d", 2) == 0)
		cache_path_len -= 2;

	if (cache_path_len > 7 && *(cache_path + cache_path_len - 7 - 1) != '/' &&
	    strncmp(cache_path + cache_path_len - 7, ".unpack", 7) == 0)
		cache_path_len -= 7;

	/* variable cachefile is used as a pattern buffer for glob() with
	   root_path* or cache_path* - afterwards for cache_path.extension */
	/* constant 1 is strlen("*") */
	ext_len_max = (UNPACKFS_CACHE_EXTENSION_LEN_ADD > 1) ?
	              UNPACKFS_CACHE_EXTENSION_LEN_ADD : 1;
	/* if searching with glob() is in root_path
	   extend memorysize for pattern if needed */
	if (!in_cache && root_path_len + 1 > cache_path_len + ext_len_max)
			ext_len_max = root_path_len + 1 - cache_path_len;

	cachefile = (char *) xmalloc(cache_path_len + ext_len_max + 1);
	if (cachefile == NULL) {
		res = -1;
		goto err_none;
	}

	if (!in_cache) {
		/* cachefile has content of root_path only to search there with glob() */
		strncpy(cachefile, root_path, root_path_len);
		path_len = root_path_len;
	}
	else {
		strncpy(cachefile, cache_path, cache_path_len);
		path_len = cache_path_len;
	}
	*(cachefile + path_len) = '*';
	*(cachefile + path_len + 1) = '\0';

	memset(&search, 0, sizeof(glob_t));
	glob(cachefile, GLOB_NOSORT|GLOB_NOESCAPE, NULL, &search);
	if (search.gl_pathv == NULL) {
		res = -1;
		goto err_free_search;
	}

	/* cachefile now has always cache_path after the glob for root_path */
	if (!in_cache)
		strncpy(cachefile, cache_path, cache_path_len);

	found = 0;

	for(filev = search.gl_pathv; (file = *filev) != NULL; filev++)
		if (lstat(file, &statbuf) == 0 && S_ISREG(statbuf.st_mode)) {
			ext_len = strlen(file + path_len);
			if (ext_len > ext_len_max) {
				ext_len_max = ext_len + UNPACKFS_CACHE_EXTENSION_LEN_ADD;
				tmp = (char *) xrealloc(cachefile,
				                        cache_path_len + ext_len_max + 1);
				if (tmp == NULL) {
					res = -1;
					break;
				}
				cachefile = tmp;
			}
			strcpy(cachefile + cache_path_len, file + path_len);

			if (unpackfs_magic_file(magic, packer_info, file, cachefile) == 0 &&
			    lstat(cache_path, &statbuf) == 0) {
				found = -1;
				break;
			}
		}

	if (found == 0)
		res = -1;

 err_free_search:
	globfree(&search);
	free(cachefile);

 err_none:
	return res;
}


/*
 * unpackfs_cache - return path of file to work with and optionally other path
 *
 * return values:
 * - work:   existing path of file (the path is prefixed with root or cache)
 *           if both or neither files exist prefer prefix root
 * - other:  the respectively other path of file (e.g. prefixed with cache if
 *           work is prefixed with root)
 *           (nothing is returned if other was NULL when calling the function)
 * - cached: zero (false) if work is in root
 *           -1   (true)  if work is in cache
 *           (nothing is returned if cached was NULL when calling the function)
 */
int unpackfs_cache(const char *path, const char *root, const char *cache,
                   enum unpackfs_cache_mode mode,
                   char **work, char **other, int *cached)
{
	size_t path_len, len;
	const char *p;
	char *root_buf, *rb, *cache_buf, *cb;
	int in_cache;
	struct stat statbuf;
	mode_t type;

	in_cache = 0;
	*work = NULL;
	if (other != NULL)
		*other = NULL;

	path_len = strlen(path);
	len = strlen(root);
	root_buf = (char *) xmalloc(len + path_len + 1);
	if (root_buf == NULL)
		goto err_none;

	strcpy(root_buf, root);
	rb = root_buf + len;

	len = strlen(cache);
	cache_buf = (char *) xmalloc(len + path_len + 1);
	if (cache_buf == NULL)
		goto err_free_root;

	strcpy(cache_buf, cache);
	cb = cache_buf + len;

	if (path_len == 1 && *path == '/')
		goto ok_root;

	for (p = path; *p != '\0'; path = p) {
		p = strchrnul(++p, '/');
		len = p - path;

		strncpy(cb, path, len);
		cb += len;
		*cb = '\0';

		strncpy(rb, path, len);
		rb += len;
		*rb = '\0';

		/* if to create something new
		   then return path of existing node if present */
		if ( *p == '\0' && mode != UFS_EXISTS) {
			if (lstat(root_buf, &statbuf) == 0)
				in_cache = 0;
			else if (lstat(cache_buf, &statbuf) == 0)
				in_cache = -1;
		}
		/* found in root */
		else if (lstat(root_buf, &statbuf) == 0) {
			in_cache = 0;
			/* break if no subdirectory is possible */
			if (*p != '\0') {
				type = statbuf.st_mode & S_IFMT;
				if (type != S_IFDIR && type != S_IFLNK) {
					strcpy(rb, p);
					if (other != NULL)
						strcpy(cb, p);
					break;
				}
			}
		}
		/* found in cache */
		else if (lstat(cache_buf, &statbuf) == 0) {
			in_cache = -1;
			/* break if no subdirectory is possible */
			if (*p != '\0') {
				type = statbuf.st_mode & S_IFMT;
				if (type != S_IFDIR && type != S_IFLNK) {
					strcpy(cb, p);
					if (other != NULL)
						strcpy(rb, p);
					break;
				}
			}
		}
		/* not found, unpack in a magic way if possible */
		else {
			/* break if unpacking is not possible */
			if (cache_magic(root_buf, cache_buf, in_cache) != 0 ||
			    lstat(cache_buf, &statbuf) != 0) {
				in_cache = -1;
				strcpy(cb, p);
				if (other != NULL)
					strcpy(rb, p);
				break;
			}

			in_cache = -1;
			/* break if no subdirectory is possible */
			if (*p != '\0') {
				type = statbuf.st_mode & S_IFMT;
				if (type != S_IFDIR && type != S_IFLNK) {
					strcpy(cb, p);
					if (other != NULL)
						strcpy(rb, p);
					break;
				}
			}
		}
	}

 ok_root:
	*work = (in_cache != 0) ? cache_buf : root_buf;

	if (other != NULL)
		*other = (in_cache != 0) ? root_buf  : cache_buf;
	else
		free((in_cache != 0) ? root_buf  : cache_buf);

	if (cached != NULL)
		*cached = in_cache;

	return 0;

 err_free_root:
	free(root_buf);

 err_none:
	return -1;
}


/* --- end --- */

