
/*
 * config.c                                                     (jh,09.02.2006)
 */

/*
 *  unpackfs: filesystem with transparent unpacking of archive files
 *  Copyright (C) 2005, 2006  Jochen Hepp <jochen.hepp@gmx.de>
 *
 *  This file is part of unpackfs.
 *
 *  unpackfs is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  unpackfs is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif
#include <unpackfs.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#	include <strings.h>
#endif
#ifdef HAVE_CTYPE_H
#	include <ctype.h>
#endif
#ifdef HAVE_ERRNO_H
#	include <errno.h>
#endif


/*
 * config_error - prints error message
 */
static void config_error(const char *configfile, int line,
                         const char *message, const char *buffer) {
	extern char* program;

	if (buffer != NULL)
		fprintf(stderr, "%s: error in config file ``%s'' at line %d:\n%s%s\n",
		        program, configfile, line, message, buffer);
	else
		fprintf(stderr, "%s: error in config file ``%s'' at line %d:\n%s",
		        program, configfile, line, message);
}


/*
 * unpackfs_config_packer - read configfile for packers
 */
int unpackfs_config_packer(const char *configfile,
                           struct unpackfs_magic_unpack_info **packer_info_p) {
	int res;
	FILE *file;
	int line, argc, in_arg, in_str;
	int argc_max = UNPACKFS_CONFIG_ARGC_MAX;
	char *buffer, *buf, *buf_arg, *buf_to, c, in_str_char, *argv[argc_max];
	size_t memory, len, argv_len[argc_max];
	struct unpackfs_magic_unpack_info *packer_info;
	struct unpackfs_magic_unpack *packer, *packer_iter, **cache;
	size_t cache_key;
	int cache_key_used;
	int argc_unpack, argnr_srcfile, extc, i;
	enum unpackfs_magic_unpack_type type;
	struct stat statbuf;

	res = 0;

	*packer_info_p = NULL;
	packer = NULL;

	buffer = (char *) xmalloc(UNPACKFS_CONFIG_BUFFERSIZE);
	if (buffer == NULL) {
		res = -ENOMEM;
		goto err_none;
	}

	if (stat(configfile, &statbuf) == 0 && S_ISDIR(statbuf.st_mode)) {
		errno = EISDIR;
		res = -EISDIR;
		goto err_free_buffer;
	}

	file = fopen(configfile, "r");
	if (file == NULL) {
		res = -errno;
		goto err_free_buffer;
	}

	packer_info = (struct unpackfs_magic_unpack_info *)
	              xcalloc(1, sizeof(struct unpackfs_magic_unpack_info));
	if (packer_info == NULL) {
		res = -ENOMEM;
		goto err_close;
	}

	cache = (struct unpackfs_magic_unpack **)
	        xcalloc(UNPACKFS_PACKER_CACHE_SIZE,
	                sizeof(struct unpackfs_magic_unpack *));
	if (cache == NULL) {
		res = -ENOMEM;
		goto err_free_info;
	}
	packer_info->ufsmui_packers_cache = cache;
	packer_info->ufsmui_cache_size = UNPACKFS_PACKER_CACHE_SIZE;

	line = 0;

	while (fgets(buffer, UNPACKFS_CONFIG_BUFFERSIZE, file) != NULL) {
		line++;

		/* split line in arguments */

		memory = len = 0;
		argc = in_arg = in_str = 0;

 		for (buf = buffer; (c = *buf) != '\0'; buf++) {
			if ((in_str == 0 && (c == ' ' || c == '\t')) || c == '\n') {
				if (in_arg == 0)
					continue;

				in_arg = 0;
				*buf_to = '\0';
				memory += len + 1;
				argv[argc] = buf_arg;
				argv_len[argc] = len;
				len = 0;

				if (++argc == argc_max) {
					for (buf++; (c = *buf) != '\0'; buf++)
						if (c != ' ' && c != '\t' && c != '\n') {
							config_error(configfile, line, "too much arguments: ",
							             buf);
							break;
						}
					break;
				}
			}
			else {
				if (in_arg == 0) {
					if (c == '#') {
						*buf = '\0';
						break;
					}
					in_arg = 1;
					buf_arg = buf_to = buf;
				}

				if (in_str == 0 && (c == '\"' || c == '\'')) {
					in_str = -1;
					in_str_char = c;
				}
				else if (in_str != 0 && c == in_str_char) {
					in_str = 0;
				}
				else {
					*(buf_to++) = c;
					len++;
				}
			}
		}

		/* parse found arguments */

		if (argc == 0)
			continue;

		if (argc < 3) {
			config_error(configfile, line, "too few parameters for packer ",
			             argv[0]);
			continue;
		}

		if (strcasecmp(argv[1], "none") == 0) {
			if (argc > 4) {
				config_error(configfile, line, "too many parameters for type "
				             "``none'' packer ", argv[0]);
				continue;
			}

			/* packer "none" found */
			type = UFSMU_NONE;
			argc_unpack = 0;
		}
		else {
			if (strcmp(argv[1], "-") == 0)
				type = UFSMU_FILE_STDOUT;
			else if (strcasecmp(argv[1], "file") == 0)
				type = UFSMU_FILE;
			else if (strcasecmp(argv[1], "dir") == 0)
				type = UFSMU_DIR;
			else {
				config_error(configfile, line, "unknown packer type: ", argv[1]);
				continue;
			}

			if (argc < 6) {
				config_error(configfile, line, "too few parameters for packer ",
				             argv[0]);
				continue;
			}

			/* real packer found */				
			argc_unpack = argc - 4;
		}

		/* parse packer extensions */
		if (argc < 4)
			extc = 0;
		else {
			extc = (*argv[3] == '\0') ? 0 : 1;
			for (buf_arg = argv[3]; (c = *buf_arg) != '\0'; buf_arg++)
				if (c == '|' || c == ',' || c == ' ' || c == '\t')
					extc++;

			if (extc == 0) {
				config_error(configfile, line, "no packer extensions found for "
				             "packer ", argv[0]);
				continue;
			}
		}

		/* parse command for packer with arguments */
		if (argc < 5)
			argnr_srcfile = 0;
		else {
			if (*argv[4] == '\0') {
				config_error(configfile, line, "no command for packer ", argv[0]);
				continue;
			}
			else if (stat(argv[4], &statbuf) != 0) {
				config_error(configfile, line, "packer command not found: ",
				             argv[4]);
				continue;
			}

			for (i = 4 + 1, argnr_srcfile = 0; i < argc; i++)
				if (strcasecmp(argv[i], "$source") == 0) {
					argnr_srcfile = i - 4;
					break;
				}

			if (argnr_srcfile == 0) {
				config_error(configfile, line,
				             "no argument ``$source'' found for packer ", argv[0]);
				continue;
			}
		}
		
		memory += sizeof(struct unpackfs_magic_unpack) - (argv_len[1] + 1) +
		          (extc + 1) * (sizeof(char *) + sizeof(size_t)) +
		          (argc_unpack + 1) * sizeof(char *);
		packer = (struct unpackfs_magic_unpack *) xmalloc(memory);
		if (packer == NULL) {
			res = -ENOMEM;
			break;
		}

		/* fill data in structure */
		buf = (char *) packer + sizeof(struct unpackfs_magic_unpack);

		packer->ufsmu_extension = (const char **) buf;
		buf += (extc + 1) * sizeof(char *);
		packer->ufsmu_cmd_argv = (const char **) buf;
		buf += (argc_unpack + 1) * sizeof(char *);
		packer->ufsmu_extension_len = (const size_t *) buf;
		buf += (extc + 1) * sizeof(size_t);

		packer->ufsmu_next_magic = NULL;
		packer->ufsmu_next_unpack = NULL;

		packer->ufsmu_name = buf;
		strcpy(buf, argv[0]);
		buf += argv_len[0] + 1;

		packer->ufsmu_type = type;

		packer->ufsmu_magic_type = buf;
		strcpy(buf, argv[2]);
		buf += argv_len[2] + 1;
		packer->ufsmu_magic_type_len = argv_len[2];

		/* store packer extensions */
		if (extc > 0) {
			extc = 0;
			len = 0;
			for (buf_arg = argv[3]; (c = *buf_arg) != '\0'; buf_arg++) {
				if (c == '|' || c == ',' || c == ' ' || c == '\t') {
					*((char **)  packer->ufsmu_extension + extc) = buf - len;
					*((size_t *) packer->ufsmu_extension_len + extc) = len;
					extc++;
					*(buf++) = '\0';
					len = 0;
				}
				else {
					*(buf++) = tolower(c);
					len++;
				}
			}
			*((char **)  packer->ufsmu_extension + extc) = buf - len;
			*((size_t *) packer->ufsmu_extension_len + extc) = len;
			extc++;
			*(buf++) = '\0';
		}
		*((char **)  packer->ufsmu_extension + extc) = NULL;
		*((size_t *) packer->ufsmu_extension_len + extc) = 0;

		/* store command for packer with arguments */
		for (i = 4; i < argc; i++) {
			*(packer->ufsmu_cmd_argv + i - 4) = buf;
			strcpy(buf, argv[i]);
			buf += argv_len[i] + 1;
		}
		*(packer->ufsmu_cmd_argv + i - 4) = NULL;

		packer->ufsmu_cmd_file = *(packer->ufsmu_cmd_argv);

		packer->ufsmu_cmd_argnr_srcfile = argnr_srcfile;

		if ((void *) packer != buf - memory)
			fprintf(stderr, "config.c: internal memory error!\n"
			        "packer: %s, memory: %d Bytes, difference: %d Bytes\n",
			        argv[0], memory, (void *)(buf - memory) - ((void *) packer));


		/* update next pointer of already available packers and cache */

		cache_key = unpackfs_cache_key(packer->ufsmu_magic_type,
		                               UNPACKFS_PACKER_CACHE_SIZE);

		if (packer_info->ufsmui_packers == NULL) {
			packer_info->ufsmui_packers = packer;
			if (packer_info->ufsmui_packers_cache[cache_key] == NULL)
				packer_info->ufsmui_packers_cache[cache_key] = packer;
		}
		else {
			if (packer_info->ufsmui_packers_cache[cache_key] != NULL) {
				packer_iter = packer_info->ufsmui_packers_cache[cache_key];
				cache_key_used = 1;
			}
			else {
				packer_iter = packer_info->ufsmui_packers;
				cache_key_used = 0;
			}

			while (packer_iter != NULL) {
				if (strcmp(packer->ufsmu_magic_type,
				           packer_iter->ufsmu_magic_type) == 0) {
					for ( ; packer_iter->ufsmu_next_unpack != NULL;
					     packer_iter = packer_iter->ufsmu_next_unpack)
						;
					packer_iter->ufsmu_next_unpack = packer;
					break;
				}
				else {
					if (packer_iter->ufsmu_next_magic != NULL) {
						/* insert packer before other packer with not matching key */
						if (cache_key_used == 1 &&
						    unpackfs_cache_key(
						       packer_iter->ufsmu_next_magic->ufsmu_magic_type,
						       UNPACKFS_PACKER_CACHE_SIZE) != cache_key) {
							packer->ufsmu_next_magic = packer_iter->ufsmu_next_magic;
							packer_iter->ufsmu_next_magic = packer;
							break;
						}
						packer_iter = packer_iter->ufsmu_next_magic;
					}
					else {
						packer_iter->ufsmu_next_magic = packer;
						if (packer_info->ufsmui_packers_cache[cache_key] == NULL)
							packer_info->ufsmui_packers_cache[cache_key] = packer;
						break;
					}
				}
			}
		}
	}

 err_free_info:
	if (res != 0) {
		unpackfs_config_packer_free(packer_info);
		packer_info = NULL;
	}
	*packer_info_p = packer_info;

 err_close:
	if (res != 0)
		fclose(file);
	else {
		res = fclose(file);
		if (res != 0)
			res = -errno;
	}

 err_free_buffer:
	free(buffer);

 err_none:
	return res;
}


/*
 * unpackfs_configfile_free - free allocated packers
 */
void unpackfs_config_packer_free(struct unpackfs_magic_unpack_info
                                        *packer_info) {
	struct unpackfs_magic_unpack *packer, *packer_next, *packer_iter;

	packer_next = packer_info->ufsmui_packers;
	if (packer_info->ufsmui_packers_cache != NULL)
		free(packer_info->ufsmui_packers_cache);
	free(packer_info);

	while (packer_next != NULL) {
		packer = packer_next;
		packer_next = packer->ufsmu_next_magic;
		packer_iter = packer->ufsmu_next_unpack;
		free(packer);
		while (packer_iter != NULL) {
			packer = packer_iter;
			packer_iter = packer->ufsmu_next_unpack;
			free(packer);
		}
	}
}


/* --- end --- */

