
/*
 * fuse_operation.c                                             (jh,05.02.2006)
 */

/*
 *  unpackfs: filesystem with transparent unpacking of archive files
 *  Copyright (C) 2005, 2006  Jochen Hepp <jochen.hepp@gmx.de>
 *
 *  This file is part of unpackfs.
 *
 *  unpackfs is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  unpackfs is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
	code is used from fuse/example/fusexmp_fh.c:

	FUSE: Filesystem in Userspace
	Copyright (C) 2001-2005  Miklos Szeredi <miklos@szeredi.hu>

	fusexmp.c can be distributed under the terms of the GNU GPL.
	See the file COPYING.
*/

#define _GNU_SOURCE
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif
#include <unpackfs.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
#	include <fcntl.h>
#endif
#ifdef HAVE_DIRENT_H
#	include <dirent.h>
#endif
#ifdef HAVE_NDIR_H
#	include <ndir.h>
#endif
#ifdef HAVE_ERRNO_H
#	include <errno.h>
#endif
#ifdef HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif
#ifdef HAVE_SYS_STATFS_H
#	include <sys/statfs.h>
#endif
#ifdef HAVE_SETXATTR
#	include <sys/xattr.h>
#endif
#ifdef HAVE_FUSE_H
#	include <fuse.h>
#endif

#ifndef HAVE_LIBFUSE
#	error library libfuse missing - this program does not work without it!
#endif

static struct unpackfs_magic_unpack_info *unpackfs_packer_info;
static magic_t unpackfs_magic_handle;
static const char *unpackfs_root_dir;
static const char *unpackfs_cache_dir;


/*
 * unpackfs_packer_info_set - store packers in extern variable
 */
void unpackfs_packer_info_set(struct unpackfs_magic_unpack_info *packer_info)
{
	extern struct unpackfs_magic_unpack_info *unpackfs_packer_info;
	unpackfs_packer_info = packer_info;
}

/*
 * unpackfs_packer_info_get - get packer info from extern variable
 */
struct unpackfs_magic_unpack_info *unpackfs_packer_info_get(void)
{
	extern struct unpackfs_magic_unpack_info *unpackfs_packer_info;
	return unpackfs_packer_info;
}

/*
 * unpackfs_magic_set - store magic in extern variable
 */
void unpackfs_magic_set(magic_t magic)
{
	extern magic_t unpackfs_magic_handle;
	unpackfs_magic_handle = magic;
}

/*
 * unpackfs_magic_get - get magic from extern variable
 */
magic_t unpackfs_magic_get(void)
{
	extern magic_t unpackfs_magic_handle;
	return unpackfs_magic_handle;
}


/*
 * unpackfs_root_set - store root directory in extern variable
 */
void unpackfs_root_set(const char *root)
{
	extern const char *unpackfs_root_dir;
	unpackfs_root_dir = root;
}


/*
 * unpackfs_cache_set - store cache directory in extern variable
 */
void unpackfs_cache_set(const char *cache)
{
	extern const char *unpackfs_cache_dir;
	unpackfs_cache_dir = cache;
}


/*
 * cache - return path of file to work with and optionally other path
 */
static int cache(const char *path, int mode, char **work, char **other,
                 int *cached)
{
	extern const char *unpackfs_root_dir;
	extern const char *unpackfs_cache_dir;

	return unpackfs_cache(path, unpackfs_root_dir, unpackfs_cache_dir,
	                      mode, work, other, cached);
}


/*
 * unpackfs_getattr - get file status
 */
static int unpackfs_getattr(const char *path, struct stat *statbuf)
{
	int res;
	char *workpath, *otherpath;
	int cached;
	nlink_t nlink;

	res = cache(path, UFS_EXISTS, &workpath, &otherpath, &cached);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	nlink = 0;

	res = lstat(otherpath, statbuf);
	if (res == 0 && S_ISDIR(statbuf->st_mode))
		nlink = statbuf->st_nlink - 2;

	res = lstat(workpath, statbuf);
	if (res != 0) {
		res = -errno;
		goto err_free_paths;
	}

	statbuf->st_nlink += nlink;

 err_free_paths:
	free(workpath);
	free(otherpath);

 err_none:
	return res;
}


/*
 * unpackfs_readlink - read value of a symbolic link
 */
static int unpackfs_readlink(const char *path, char *buf, size_t size)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = readlink(workpath, buf, size - 1);
	if (res == -1) {
		res = -errno;
		goto err_free_workpath;
	}

	buf[res] = '\0';
	res = 0;

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_opendir - open a directory
 */
static int unpackfs_opendir(const char *path, struct fuse_file_info *fi)
{
	(void) path;
	(void) fi;

	return 0;
}


/*
 * unpackfs_readdir - read a directory
 */
static int unpackfs_readdir(const char *path, void *buf,
                            fuse_fill_dir_t filler, off_t offset,
                            struct fuse_file_info *fi)
{
	int res;
	char *workpath, *otherpath;
	char *workpathname, *otherpathname;
	size_t workpath_len, otherpath_len, d_name_len;
	int cached;
	char **entries, **entry, *name;
	size_t entries_avail, entries_max, entries_count;
	DIR *dp;
	struct dirent *de;
	struct stat statbuf;
	(void) offset;
	(void) fi;

	res = cache(path, UFS_EXISTS, &workpath, &otherpath, &cached);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}


	/* read directory in workpath to find magic files */

	dp = opendir(workpath);
	if (dp == NULL) {
		res = -errno;
		goto err_free_paths;
	}

	workpath_len = strlen(workpath);
	otherpath_len = strlen(otherpath);

	while ((de = readdir(dp)) != NULL) {
		d_name_len = strlen(de->d_name);

		workpathname = (char *) xmalloc(workpath_len + 1 + d_name_len + 1);
		if (workpathname == NULL) {
			res = -ENOMEM;
			break;
		}
		strcpy(workpathname, workpath);
		*(workpathname + workpath_len) = '/';
		strcpy(workpathname + workpath_len + 1, de->d_name);

		if (lstat(workpathname, &statbuf) == 0 &&
		    S_ISREG(statbuf.st_mode)) {
			/* if cached then workpath is already in cache_dir */
			if (cached)
				unpackfs_magic_file(unpackfs_magic_get(),
				                    unpackfs_packer_info_get(),
				                    workpathname, workpathname);

			/* if not cached then workpath is in root_dir and otherpath is in
			   cache_dir so prepare otherpathname for magic_file */
			else {
				otherpathname = (char *)
				                xmalloc(otherpath_len + 1 + d_name_len + 1);
				if (otherpathname == NULL) {
					free(workpathname);
					res = -ENOMEM;
					break;
				}
				strcpy(otherpathname, otherpath);
				*(otherpathname + otherpath_len) = '/';
				strcpy(otherpathname + otherpath_len + 1, de->d_name);

				unpackfs_magic_file(unpackfs_magic_get(),
				                    unpackfs_packer_info_get(),
				                    workpathname, otherpathname);
				free(otherpathname);
			}
		}

		free(workpathname);
	}
	if (res != 0) {
		closedir(dp);
		goto err_free_paths;
	}
#ifdef CLOSEDIR_VOID
	closedir(dp);
#else
	res = closedir(dp);
	if (res != 0) {
		res = -errno;
		goto err_free_paths;
	}
#endif


	entries_max = UNPACKFS_DIR_ENTRIES_ADD;
	entries = (char **) xmalloc(entries_max * sizeof(char *));
	if (entries == NULL) {
		res = -ENOMEM;
		goto err_free_paths;
	}

	entries_avail = entries_max;
	entry = entries;

	/* if not cached then workpath is in root_dir and otherpath is in cache_dir
	   so read directory in other path if it exists */

	if (!cached) {
		dp = opendir(otherpath);
		if (dp != NULL) {
			while ((de = readdir(dp)) != NULL) {
				memset(&statbuf, 0, sizeof(statbuf));
				statbuf.st_ino = de->d_ino;
				statbuf.st_mode = de->d_type << 12;
				if (filler(buf, de->d_name, &statbuf, 0)) {
					res = -ENOMEM;
					break;
				}

				if (entries_avail == 0) {
					entries_max += UNPACKFS_DIR_ENTRIES_ADD;
					entry = (char **)
					        xrealloc(entries, entries_max * sizeof(char *));
					if (entry == NULL) {
						res = -ENOMEM;
						break;
					}
					entries_avail = UNPACKFS_DIR_ENTRIES_ADD;
					entries = entry;
					entry = entries + (entries_max - entries_avail);
				}
				name = xstrdup(de->d_name);
				if (name == NULL) {
					res = -ENOMEM;
					break;
				}
				*(entry++) = name;
				entries_avail--;
			}

			if (res != 0) {
				closedir(dp);
				goto err_free_entries;
			}
#ifdef CLOSEDIR_VOID
			closedir(dp);
#else
			res = closedir(dp);
			if (res != 0) {
				res = -errno;
				goto err_free_entries;
			}
#endif
		}
	}

	entries_count = entries_max - entries_avail;
	qsort(entries, entries_count, sizeof(char *), unpackfs_compare_strings);


	/* read directory in workpath */

	dp = opendir(workpath);
	if (dp == NULL) {
		res = -errno;
		goto err_free_entries;
	}

	while ((de = readdir(dp)) != NULL) {
		entry = (char **) &(de->d_name);
		if (bsearch(&entry, entries, entries_count, sizeof(char *),
		            unpackfs_compare_strings) == NULL) {
			memset(&statbuf, 0, sizeof(statbuf));
			statbuf.st_ino = de->d_ino;
			statbuf.st_mode = de->d_type << 12;
			if (filler(buf, de->d_name, &statbuf, 0)) {
				res = -ENOMEM;
				break;
			}
		}
	}
	if (res != 0) {
		closedir(dp);
		goto err_free_entries;
	}
#ifdef CLOSEDIR_VOID
	closedir(dp);
#else
	res = closedir(dp);
	if (res != 0) {
		res = -errno;
		goto err_free_entries;
	}
#endif

 err_free_entries:
	entries_max -= entries_avail;
	for (entry = entries; entries_max-- != 0; entry++)
		free(*entry);
	free(entries);

 err_free_paths:
	free(workpath);
	free(otherpath);

 err_none:
	return res;
}


/*
 * unpackfs_releasedir - close a directory
 */
static int unpackfs_releasedir(const char *path, struct fuse_file_info *fi)
{
	(void) path;
	(void) fi;

	return 0;
}


/*
 * unpackfs_mknod - create a special or ordinary file
 */
static int unpackfs_mknod(const char *path, mode_t mode, dev_t rdev)
{
	int res;
	char *workpath;

	res = cache(path, UFS_NEW, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = mknod(workpath, mode, rdev);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_mkdir - create a directory
 */
static int unpackfs_mkdir(const char *path, mode_t mode)
{
	int res;
	char *workpath;

	res = cache(path, UFS_NEW, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = mkdir(workpath, mode);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_unlink - delete a name and possibly the file it refers to
 */
static int unpackfs_unlink(const char *path)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = unlink(workpath);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_rmdir - delete a directory
 */
static int unpackfs_rmdir(const char *path)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = rmdir(workpath);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_symlink - make a new name for a file
 */
static int unpackfs_symlink(const char *from, const char *to)
{
	int res;
	char *workto;

	res = cache(to, UFS_NEW, &workto, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = symlink(from, workto);
	if (res != 0) {
		res = -errno;
		goto err_free_workto;
	}

 err_free_workto:
	free(workto);

 err_none:
	return res;
}


/*
 * unpackfs_rename - change the name or location of a file
 */
static int unpackfs_rename(const char *from, const char *to)
{
	int res;
	char *workfrom, *workto;

	res = cache(from, UFS_EXISTS, &workfrom, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = cache(to, UFS_NEW, &workto, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_free_workfrom;
	}

	res = rename(workfrom, workto);
	if (res != 0) {
		res = -errno;
		goto err_free_workto;
	}

 err_free_workto:
	free(workto);

 err_free_workfrom:
	free(workfrom);

 err_none:
	return res;
}


/*
 * unpackfs_link - make a new name for a file
 */
static int unpackfs_link(const char *from, const char *to)
{
	int res;
	char *workfrom, *workto;

	res = cache(from, UFS_EXISTS, &workfrom, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = cache(to, UFS_NEW, &workto, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_free_workfrom;
	}

	res = link(workfrom, workto);
	if (res != 0) {
		res = -errno;
		goto err_free_workto;
	}

 err_free_workto:
	free(workto);

 err_free_workfrom:
	free(workfrom);

 err_none:
	return res;
}


/*
 * unpackfs_chmod - change permissions of a file
 */
static int unpackfs_chmod(const char *path, mode_t mode)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = chmod(workpath, mode);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_chown - change ownership of a file
 */
static int unpackfs_chown(const char *path, uid_t uid, gid_t gid)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

#ifdef HAVE_LCHOWN
	res = lchown(workpath, uid, gid);
#else
	res = chown(workpath, uid, gid);
#endif
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_truncate - truncate a file to a specified length
 */
static int unpackfs_truncate(const char *path, off_t size)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = truncate(workpath, size);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


#ifdef HAVE_UTIME
/*
 * unpackfs_utime - change access and/or modification times of an inode
 */
static int unpackfs_utime(const char *path, struct utimbuf *buf)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = utime(workpath, buf);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}
#endif


/*
 * unpackfs_open - open and possibly create a file or device
 */
static int unpackfs_open(const char *path, struct fuse_file_info *fi)
{
	int res;
	char *workpath;
	int fd;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	fd = open(workpath, fi->flags);
	if (fd == -1) {
		res = -errno;
		goto err_free_workpath;
	}

	fi->fh = fd;

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_read - read from a file descriptor
 */
static int unpackfs_read(const char *path, char *buf, size_t size,
                         off_t offset, struct fuse_file_info *fi)
{
	int res;
	(void) path;

	res = pread(fi->fh, buf, size, offset);
	if (res == -1) {
		res = -errno;
		goto err_none;
	}

 err_none:
	return res;
}


/*
 * unpackfs_write - write to a file descriptor
 */
static int unpackfs_write(const char *path, const char *buf, size_t size,
                          off_t offset, struct fuse_file_info *fi)
{
	int res;
	(void) path;

	res = pwrite(fi->fh, buf, size, offset);
	if (res == -1) {
		res = -errno;
		goto err_none;
	}

 err_none:
	return res;
}


/*
 * unpackfs_statfs - get file system statistics
 */
static int unpackfs_statfs(const char *path, struct statfs *statbuf)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = statfs(workpath, statbuf);
	if (res != 0) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_release - close filehandle
 */
static int unpackfs_release(const char *path, struct fuse_file_info *fi)
{
	(void) path;

	close(fi->fh);

	return 0;
}


/*
 * unpackfs_fsync - synchronize a file's in-core state with that on disk
 */
static int unpackfs_fsync(const char *path, int isdatasync,
                          struct fuse_file_info *fi)
{
	int res;
	(void) path;

	res = 0;

	if (isdatasync)
#ifdef HAVE_FDATASYNC
		res = fdatasync(fi->fh);
#else
		res = fsync(fi->fh);
#endif
	else
		res = fsync(fi->fh);

	if (res != 0) {
		res = -errno;
		goto err_none;
	}

 err_none:
	return res;
}


#ifdef HAVE_SETXATTR
/* xattr operations are optional and can safely be left unimplemented */

/*
 * unpackfs_setxattr
 */
static int unpackfs_setxattr(const char *path, const char *name,
                           const char *value, size_t size, int flags)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = lsetxattr(workpath, name, value, size, flags);
	if (res == -1) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_getxattr
 */
static int unpackfs_getxattr(const char *path, const char *name, char *value,
                           size_t size)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = lgetxattr(workpath, name, value, size);
	if (res == -1) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_listxattr
 */
static int unpackfs_listxattr(const char *path, char *list, size_t size)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = llistxattr(workpath, list, size);
	if (res == -1) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}


/*
 * unpackfs_removexattr
 */
static int unpackfs_removexattr(const char *path, const char *name)
{
	int res;
	char *workpath;

	res = cache(path, UFS_EXISTS, &workpath, NULL, NULL);
	if (res != 0) {
		res = -ENOMEM;
		goto err_none;
	}

	res = lremovexattr(workpath, name);
	if (res == -1) {
		res = -errno;
		goto err_free_workpath;
	}

 err_free_workpath:
	free(workpath);

 err_none:
	return res;
}
#endif /* HAVE_SETXATTR */


/*
 *  unpackfs_operations - table of filesystem functions for fuse
 */
struct fuse_operations unpackfs_operations = {
	.getattr    = unpackfs_getattr,
	.readlink   = unpackfs_readlink,
	.opendir    = unpackfs_opendir,
	.readdir    = unpackfs_readdir,
	.releasedir = unpackfs_releasedir,
	.mknod      = unpackfs_mknod,
	.mkdir      = unpackfs_mkdir,
	.symlink    = unpackfs_symlink,
	.unlink     = unpackfs_unlink,
	.rmdir      = unpackfs_rmdir,
	.rename     = unpackfs_rename,
	.link       = unpackfs_link,
	.chmod      = unpackfs_chmod,
	.chown      = unpackfs_chown,
	.truncate   = unpackfs_truncate,
#ifdef HAVE_UTIME
	.utime      = unpackfs_utime,
#endif
	.open       = unpackfs_open,
	.read       = unpackfs_read,
	.write      = unpackfs_write,
	.statfs     = unpackfs_statfs,
	.release    = unpackfs_release,
	.fsync      = unpackfs_fsync,
#ifdef HAVE_SETXATTR
	.setxattr   = unpackfs_setxattr,
	.getxattr   = unpackfs_getxattr,
	.listxattr  = unpackfs_listxattr,
	.removexattr= unpackfs_removexattr,
#endif
};


/* --- end --- */

