
/*
 * fuse_setup.c                                                 (jh,15.12.2005)
 */

/*
 *  unpackfs: filesystem with transparent unpacking of archive files
 *  Copyright (C) 2005  Jochen Hepp <jochen.hepp@gmx.de>
 *
 *  This file is part of unpackfs.
 *
 *  unpackfs is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  unpackfs is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
	code is used from fuse/lib/helper.c:

	FUSE: Filesystem in Userspace
	Copyright (C) 2001-2005  Miklos Szeredi <miklos@szeredi.hu>

	libfuse can be distributed under the terms of the GNU LGPL.
	See the file COPYING.LIB.
*/

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif
#include <unpackfs.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_SIGNAL_H
#	include <signal.h>
#endif
#ifdef HAVE_FUSE_H
#	include <fuse.h>
#endif

#ifndef HAVE_LIBFUSE
#	error library libfuse missing - this program does not work without it!
#endif

static struct unpackfs_fuse *unpackfs_fuse_data = NULL;


/*
 * exit_handler - signal handler exits from fuse event loop
 */
static void exit_handler(int sig)
{
	extern struct unpackfs_fuse *unpackfs_fuse_data;
	struct fuse *fuse;
	(void) sig;

	if (unpackfs_fuse_data != NULL) {
		fuse = unpackfs_fuse_data->ufsf_fuse_handle;
		if (fuse != NULL)
			fuse_exit(fuse);
	}
}


/*
 * set_signal_handler - set a signal handler
 */
static int set_signal_handler(int sig, void (*handler)(int))
{
	struct sigaction sa;
	struct sigaction old_sa;

	memset(&sa, 0, sizeof(struct sigaction));
	sa.sa_handler = handler;
	sigemptyset(&(sa.sa_mask));
	sa.sa_flags = 0;

	if (sigaction(sig, NULL, &old_sa) == -1) {
		perror("fuse: cannot get old signal handler");
		return -1;
	}

	if (old_sa.sa_handler == SIG_DFL &&
	    sigaction(sig, &sa, NULL) == -1) {
		perror("fuse: cannot set signal handler");
		return -1;
	}
	return 0;
}


/*
 * set_signal_handler - set handlers for signals HUP, INT, TERM and ignore PIPE
 */
static int set_signal_handlers(void)
{
	if (set_signal_handler(SIGHUP, exit_handler) == -1 ||
	    set_signal_handler(SIGINT, exit_handler) == -1 ||
	    set_signal_handler(SIGTERM, exit_handler) == -1 ||
	    set_signal_handler(SIGPIPE, SIG_IGN) == -1)
	return -1;

	return 0;
}


/*
 * unpack_fuse_mount - mount fuse filesystem
 */
struct unpackfs_fuse *
unpackfs_fuse_mount(struct fuse_operations *op, size_t op_size,
                    char *mountpoint, char *mount_opts,
                    char *fuselib_opts, int background)
{
	extern const char *program;
	extern struct unpackfs_fuse *unpackfs_fuse_data;
	struct fuse *fuse;
	int fd;
	char *mp;
	int res;

	if (unpackfs_fuse_data != NULL) {
		fprintf(stderr, "%s: unpack_fuse_mount() called twice\n", program);
		goto err_none;
	}

	unpackfs_fuse_data = (struct unpackfs_fuse *)
	                     xmalloc(sizeof(struct unpackfs_fuse));
	if (unpackfs_fuse_data == NULL)
		goto err_none;

	unpackfs_fuse_data->ufsf_fuse_handle = NULL;
	unpackfs_fuse_data->ufsf_control_fd = -1;
	unpackfs_fuse_data->ufsf_mountpoint = NULL;

	mp = (char *) xmalloc(strlen(mountpoint) + 1);
	if (mp == NULL)
		goto err_free_unpackfs_fuse;

	strcpy(mp, mountpoint);
	unpackfs_fuse_data->ufsf_mountpoint = mp;

	fd = fuse_mount(mp, mount_opts);
	if (fd == -1)
		goto err_free_mountpoint;

	unpackfs_fuse_data->ufsf_control_fd = fd;

	fuse = fuse_new(fd, fuselib_opts, op, op_size);
	if (fuse == NULL)
		goto err_unmount;

	unpackfs_fuse_data->ufsf_fuse_handle = fuse;

	if (background) {
		res = daemon(0, 0);
		if (res == -1) {
			perror("fuse: failed to daemonize program");
			goto err_destroy;
		}
	}

	res = set_signal_handlers();
	if (res == -1)
		goto err_destroy;

	return unpackfs_fuse_data;

 err_destroy:
	unpackfs_fuse_data->ufsf_fuse_handle = NULL;
	fuse_destroy(fuse);
 err_unmount:
	fuse_unmount(mp);
	close(fd);
	unpackfs_fuse_data->ufsf_control_fd = -1;
 err_free_mountpoint:
	free(mp);
	unpackfs_fuse_data->ufsf_mountpoint = NULL;
 err_free_unpackfs_fuse:
	free(unpackfs_fuse_data);
	unpackfs_fuse_data = NULL;
 err_none:
	return NULL;
}


/*
 * unpackfs_fuse_unmount - unmount fuse filesystem
 */
int unpackfs_fuse_unmount(struct unpackfs_fuse *unpackfs_fuse)
{
	struct fuse *fuse;
	int fd;
	char *mp;
	int res;

	if (unpackfs_fuse == NULL)
		return 0;

	fuse = unpackfs_fuse->ufsf_fuse_handle;
	unpackfs_fuse->ufsf_fuse_handle = NULL;
	fuse_destroy(fuse);

	mp = unpackfs_fuse->ufsf_mountpoint;
	fuse_unmount(mp);
	fd = unpackfs_fuse->ufsf_control_fd;
	res = close(fd);
	unpackfs_fuse->ufsf_control_fd = -1;

	free(mp);
	unpackfs_fuse->ufsf_mountpoint = NULL;

	free(unpackfs_fuse);
	unpackfs_fuse = NULL;

	if (res == -1)
		return 1;

	return 0;
}


/* --- end --- */

