
/*
 * magic.c                                                      (jh,09.02.2006)
 */

/*
 *  unpackfs: filesystem with transparent unpacking of archive files
 *  Copyright (C) 2005, 2006  Jochen Hepp <jochen.hepp@gmx.de>
 *
 *  This file is part of unpackfs.
 *
 *  unpackfs is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  unpackfs is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#define _GNU_SOURCE
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif
#include <unpackfs.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#	include <strings.h>
#endif
#ifdef HAVE_LIBGEN_H
#	include <libgen.h>
#endif
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
#	include <errno.h>
#endif
#ifdef HAVE_FCNTL_H
#	include <fcntl.h>
#endif
#ifdef HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#	include <sys/wait.h>
#endif
#ifdef HAVE_UTIME_H
#	include <utime.h>
#endif
#ifdef HAVE_MAGIC_H
#	include <magic.h>
#endif


#ifndef HAVE_LIBMAGIC
#	error library libmagic missing - this program does not work without magic!
#endif

#ifndef HAVE_FORK
#	error fork is not available - this program needs a working fork!
#endif
#ifndef HAVE_WORKING_FORK
#	error fork is not working - this program needs a working fork!
#endif


/*
 * unpackfs_magic_open - open magic library (libmagic, file)
 */
magic_t unpackfs_magic_open(void)
{
	extern const char *program;
	magic_t magic;
	char *magicfile;
	char *usermagic;
	char *home;
	size_t home_len;
	struct stat statbuf;

	magic=magic_open(MAGIC_RAW|MAGIC_ERROR);
	if (magic == NULL)
		goto err_none;

	if (magic_setflags(magic, MAGIC_PRESERVE_ATIME) != 0)
		fprintf(stderr, "%s: %s\n", program, "atime is not preserved");

	usermagic = NULL;
	if ((magicfile = getenv("MAGIC")) == NULL) {
		if ((home = getenv("HOME")) != NULL) {
			home_len = strlen(home);
			if ((usermagic = (char *) xmalloc(home_len + 7 + 1)) != NULL) {
				strcpy(usermagic, home);
				strcpy(usermagic + home_len, "/.magic");
				if (stat(usermagic, &statbuf) == 0) {
					magicfile = usermagic;
				}
				else {
					free(usermagic);
					usermagic = NULL;
				}
			}
		}
		if (magicfile == NULL)
			magicfile = UNPACKFS_MAGICFILE;
	}

	if (magic_load(magic, magicfile) == -1) {
		fprintf(stderr, "%s: %s\n", program, magic_error(magic));
		goto err_magic_close;
	}

	if (usermagic)
		free(usermagic);

	return magic;

 err_magic_close:
	if (usermagic)
		free(usermagic);
	magic_close(magic);
 err_none:
	return NULL;
}


/*
 * unpackfs_magic_close - close magic library (libmagic, file)
 */
void unpackfs_magic_close(magic_t magic)
{
	magic_close(magic);
}


/*
 * unpackfs_magic_file - is file magic and an archive then unpack it
 */
int unpackfs_magic_file(magic_t magic,
                        struct unpackfs_magic_unpack_info *packer_info,
                        const char *srcfile, const char *dstfile) {
	extern const char *program;
	extern int debug;
	int res;
	const char *type, **exts, *ext, **argv;
	const size_t *lens;
	char *srctest, *dsttest, *s;
	size_t srcfile_len, dstfile_len, dsttest_len, dstfile_basename_len, len;
	struct unpackfs_magic_unpack *packer, *packer_first;
	size_t cache_key;
	int magic_found, ext_append;
	pid_t pid;
	int status, fd;
	struct stat statbuf;
	struct utimbuf utimebuf;

	res = 0;

	type = magic_file(magic, srcfile);
	if (type == NULL) {
		res = -errno;
		goto err_none;
	}

	/* constant 7 is max(strlen(".d"), strlen(".unpack")), see ext = . below */
	srcfile_len = strlen(srcfile);
	srctest = (char *) xmalloc(srcfile_len + 7 + 1);
	if (srctest == NULL) {
		res = -ENOMEM;
		goto err_none;
	}

	/* constant 7 is max(strlen(".d"), strlen(".unpack")), see ext = . below */
	dstfile_len = strlen(dstfile);
	dsttest = (char *) xmalloc(dstfile_len + 7 + 1);
	if (dsttest == NULL) {
		res = -ENOMEM;
		goto err_free_srctest;
	}
	s = memrchr(dstfile, '/', dstfile_len);
	if (s == NULL)
		dstfile_basename_len = dstfile_len;
	else
		dstfile_basename_len = dstfile_len - ((s + 1) - dstfile);


	packer_first = NULL;
	magic_found = ext_append = 0;

	cache_key = unpackfs_cache_key(type, UNPACKFS_PACKER_CACHE_SIZE);
	packer = packer_info->ufsmui_packers_cache[cache_key];

	while (packer != NULL) {
		res = 0;

		/* if packer type found */
		if (magic_found ||
		    strncmp(type, packer->ufsmu_magic_type,
		            packer->ufsmu_magic_type_len) == 0) {
			magic_found = 1;
			packer_first = packer;

			/* if packer is no real program and has no extensions
			   then quit searching */
			if (packer->ufsmu_type == UFSMU_NONE &&
			    *(packer->ufsmu_extension) == NULL)
				break;

			/* shorten filename from extension */
			if (ext_append == 0) {
				exts = packer->ufsmu_extension;
				lens = packer->ufsmu_extension_len;
				for( ; len = *lens, (ext = *exts) != NULL; exts++, lens++) {
					if (dstfile_basename_len > len &&
					    strcasecmp(ext, dstfile + dstfile_len - len) == 0) {
						strncpy(srctest, srcfile, srcfile_len - len);
						*(srctest + srcfile_len - len) = '\0';
						dsttest_len = dstfile_len - len;
						strncpy(dsttest, dstfile, dsttest_len);
						*(dsttest + dsttest_len) = '\0';
						break;
					}
				}
			}
			/* append ".d" or ".unpack" to filename */
			else {
				/* remind to adjust constant length 7 added at xmalloc above too
				   if you change the value of the variables ext here */
				if (packer->ufsmu_type == UFSMU_DIR) {
					ext = ".d";   /* constant 2 is strlen(".d") */
					dsttest_len = dstfile_len + 2;
				}
				else {
					ext = ".unpack";   /* constant 7 is strlen(".unpack") */
					dsttest_len = dstfile_len + 7;
				}
				strncpy(srctest, srcfile, srcfile_len);
				strcpy(srctest + srcfile_len, ext);
				strncpy(dsttest, dstfile, dstfile_len);
				strcpy(dsttest + dstfile_len, ext);
			}

			if (ext != NULL) {
				/* if packer is no real program then quit searching */
				if (packer->ufsmu_type == UFSMU_NONE)
					break;

				/* if filename exists already in source or destination directory */
				if (lstat(srctest, &statbuf) != -1) {
					res = -1;
					break;
				}
				if (lstat(dsttest, &statbuf) != -1)
					break;

				/* create directory for destination file if it does not exist */
				for(len = dsttest_len;
				    len > 0 && (s = memrchr(dsttest, '/', len)) != NULL;
				    len = s - dsttest) {
					*s = '\0';
					if (lstat(dsttest, &statbuf) != 0)
						*s = '/';
					else {
						*s = '/';
						if (!S_ISDIR(statbuf.st_mode)) {
							res = -1;
							break;
						}
						for(s++ ; (s = strchr(s, '/')) != NULL; s++) {
							*s = '\0';
							res = mkdir(dsttest, S_IRWXU|S_IRWXG|S_IRWXO);
							*s = '/';
							if (res != 0)
								break;
						}
						break;
					}
				}
				if (res != 0)
					break;

				if (debug != 0)
					fprintf(stdout, "magic: unpacker: %s, ext: %s, src: %s, dst: %s"
					        "\n", packer->ufsmu_name, ext, srcfile, dsttest);

				pid = fork();
				if (pid == 0) {
					/* child */
					if (packer->ufsmu_type == UFSMU_FILE_STDOUT) {
						fd = open(dsttest, O_CREAT|O_EXCL|O_WRONLY|O_NOFOLLOW,
						          S_IRUSR|S_IWRITE|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);
						if (fd == -1)
							goto err_fork_none;

						if (dup2(fd, 1) != 1) {
							close(fd);
							goto err_fork_none;
						}
						close(fd);
					}
					else if (packer->ufsmu_type == UFSMU_DIR) {
						if (mkdir(dsttest, S_IRWXU|S_IRWXG|S_IRWXO) != 0)
							goto err_fork_none;

						if (chdir(dsttest) != 0)
							goto err_fork_none;
					}
					else if (packer->ufsmu_type == UFSMU_FILE) {
						if (chdir(dirname(dsttest)) != 0)
							goto err_fork_none;
					}

					*(packer->ufsmu_cmd_argv +
					  packer->ufsmu_cmd_argnr_srcfile) = srcfile;
					execv(packer->ufsmu_cmd_file, (char **) packer->ufsmu_cmd_argv);

					fprintf(stderr, "%s: packer %s: exec failed!\ncommand line:",
					        program, packer->ufsmu_name);
					for (argv = packer->ufsmu_cmd_argv; *argv != NULL; argv++)
						fprintf(stderr, " %s", *argv);
					fprintf(stderr, "\n%s: %s\n", program, strerror(errno));

				 err_fork_none:
					_exit(EXIT_FAILURE);
				}
				else if (pid == -1) {
					/* fork failed */
					fprintf(stderr, "%s: packer %s: fork failed!\ncommand line:",
					        program, packer->ufsmu_name);
					for (argv = packer->ufsmu_cmd_argv; *argv != NULL; argv++)
						fprintf(stderr, " %s", *argv);
					fprintf(stderr, "\n");
					res = -1;
					break;
				}

				/* forked, now wait for child and break if error happens */
				if (waitpid(pid, &status, 0) != pid) {
					res = -1;
					break;
				}

				/* break if child terminated not normally with exit() or _exit() */
				if (!WIFEXITED(status)) {
					res = -1;
					break;
				}

				res = WEXITSTATUS(status);
				if (res != EXIT_SUCCESS) {
					/* unpacker failed, if exists try to remove file or directory */
					if (lstat(dsttest, &statbuf) == 0 &&
					    ((packer->ufsmu_type == UFSMU_FILE_STDOUT ||
					      packer->ufsmu_type == UFSMU_FILE) &&
					     unlink(dsttest) == -1) ||
					    (packer->ufsmu_type == UFSMU_DIR &&
					     rmdir(dsttest) == -1)) {
						/* it was not possible to remove file or directory */
						res = -1;
						break;
					}
				}
				else {
					/* if a new file was created then try unpacking again */
					if ((packer->ufsmu_type == UFSMU_FILE_STDOUT ||
					     packer->ufsmu_type == UFSMU_FILE) &&
					    lstat(dsttest, &statbuf) == 0 &&
					    S_ISREG(statbuf.st_mode)) {
						if (packer->ufsmu_type == UFSMU_FILE_STDOUT &&
						    lstat(srcfile, &statbuf) == 0) {
							utimebuf.actime  = statbuf.st_atime;
							utimebuf.modtime = statbuf.st_mtime;
							utime(dsttest, &utimebuf);
						}
						unpackfs_magic_file(magic, packer_info, dsttest, dsttest);
					}

					break;
				}
			}
		}

		if (magic_found)
			packer = packer->ufsmu_next_unpack;
		else
			packer = packer->ufsmu_next_magic;

		/* if no packer has a matching extension then append an extension */
		if (packer == NULL && ext_append == 0) {
			packer = packer_first;
			ext_append = 1;
		}
	}

	/* no matching packer found */
	if (packer == NULL)
		res = -1;

	free(dsttest);

 err_free_srctest:
	free(srctest);

 err_none:
	return res;
}


/* --- end --- */

