
/*
 * util.c                                                       (jh,08.02.2006)
 */

/*
 *  unpackfs: filesystem with transparent unpacking of archive files
 *  Copyright (C) 2005, 2006  Jochen Hepp <jochen.hepp@gmx.de>
 *
 *  This file is part of unpackfs.
 *
 *  unpackfs is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  unpackfs is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif
#include <unpackfs.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif
#ifdef HAVE_DIRENT_H
#	include <dirent.h>
#endif
#ifdef HAVE_NDIR_H
#	include <ndir.h>
#endif
#ifdef HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif


/*
 * xmalloc - allocates memory with malloc and prints error if not possible
 *
 * based on the code from edition 0.10 of the GNU C Library Reference Manual
 */
void *xmalloc(size_t size)
{
	extern const char *program;
	register void *value;

	value = malloc(size);
	if (value == 0)
		fprintf(stderr, "%s: %s\n", program, "virtual memory exhausted");

	return value;
}


/*
 * xcalloc - allocates memory with calloc and prints error if not possible
 *
 * based on the code from edition 0.10 of the GNU C Library Reference Manual
 */
void *xcalloc(size_t nmemb, size_t size)
{
	extern const char *program;
	register void *value;

	value = calloc(nmemb, size);
	if (value == 0)
		fprintf(stderr, "%s: %s\n", program, "virtual memory exhausted");

	return value;
}


/*
 * xrealloc - reallocates memory with realloc and prints error if not possible
 *
 * based on the code from edition 0.10 of the GNU C Library Reference Manual
 */
void *xrealloc(void *ptr, size_t size)
{
	extern const char *program;
	register void *value;

	value = realloc(ptr, size);
	if (value == 0)
		fprintf(stderr, "%s: %s\n", program, "virtual memory exhausted");

	return value;
}


/*
 * xstrdup - duplicate a string and prints error if not possible
 */
char *xstrdup(const char *s)
{
	extern const char *program;
	register char *value;

	value = strdup(s);
	if (value == 0)
		fprintf(stderr, "%s: %s\n", program, "virtual memory exhausted");

	return value;
}


/*
 * unpackfs_compare_strings - compare two strings for e.g. qsort
 */
int unpackfs_compare_strings(const void *a, const void *b)
{
	return strcmp(*((const char **) a), *((const char **) b));
}


/*
 * unpackfs_string_cat - concatenates two strings and allocates memory for them
 */
char *unpackfs_string_cat(const char *str1, const char *str2)
{
	char *buffer;
	size_t len1;

	len1 = strlen(str1);
	buffer = (char *) xmalloc(len1 + strlen(str2) + 1);
	if (buffer == NULL)
		goto err_none;

	strcpy(buffer, str1);
	strcpy(buffer + len1, str2);

 err_none:
	return buffer;
}


/*
 * unpackfs_tmp_template - create a template for a temporary file or directory
 */
char *unpackfs_tmp_template(const char *prefix)
{
	char *tmpdir;
	const char *random;
	char *buffer;
	size_t tmpdir_len, prefix_len;
	struct stat statbuf;

	tmpdir = getenv("TMPDIR");
	if (tmpdir != NULL)
		if (*tmpdir == '\0' ||
		    stat(tmpdir, &statbuf) != 0 ||
		    !S_ISDIR(statbuf.st_mode))
			tmpdir = NULL;

	if (tmpdir == NULL)
		tmpdir = "/tmp";

	tmpdir_len = strlen(tmpdir);
	prefix_len = strlen(prefix);
	random = ".XXXXXX";
	buffer = (char *) xmalloc(tmpdir_len + 1 + prefix_len + strlen(random) + 1);
	if (buffer == NULL)
		goto err_none;

	strcpy(buffer, tmpdir);
	*(buffer + tmpdir_len) = '/';
	strcpy(buffer + tmpdir_len + 1, prefix);
	strcpy(buffer + tmpdir_len + 1 + prefix_len, random);

 err_none:
	return buffer;
}


/*
 * unpackfs_rmdir_recursive - remove directory and all files and links in it
 */
int unpackfs_rmdir_recursive(const char *dir)
{
	int res;
	char *file, *tmp;
	size_t dir_len, file_len, file_len_max;
	DIR *dp;
	struct dirent *de;
	struct stat statbuf;

	res = 0;

	dir_len = strlen(dir);
	file_len_max = UNPACKFS_UTIL_FILENAME_ADD;
	file = (char *) xmalloc(dir_len + 1 + file_len_max + 1);
	if (file == NULL)
		goto err_none;

	strcpy(file, dir);
	*(file + dir_len) = '/';

	dp = opendir(dir);
	if (dir == NULL) {
		res = -1;
		goto err_free_file;
	}

	while ((de = readdir(dp)) != NULL) {
		file_len = strlen(de->d_name);

		if (file_len > file_len_max) {
			file_len_max = file_len + UNPACKFS_UTIL_FILENAME_ADD;
			tmp = (char *) xrealloc(file, dir_len + 1 + file_len_max + 1);
			if (tmp == NULL) {
				res = -1;
				break;
			}
			file = tmp;
		}

		if ((file_len == 1 && *(de->d_name) == '.') ||
		    (file_len == 2 && *(de->d_name) == '.' && *(de->d_name+1) == '.'))
			continue;

		strcpy(file + dir_len + 1, de->d_name);

		if (lstat(file, &statbuf) == 0) {
			if (S_ISDIR(statbuf.st_mode)) {
				res |= chmod(file, S_IRWXU);
				res |= unpackfs_rmdir_recursive(file);
			}
			else
				res |= unlink(file);
		}
	}

	res |= closedir(dp);

	res |= rmdir(dir);

 err_free_file:
	free(file);

 err_none:
	return res;
}


/*
 * unpackfs_hashkey - make a hash key from a string
 */
size_t unpackfs_cache_key(const char *string, const size_t size)
{
	char c;
	size_t key;

	c = *string;
	key = (c) ? (c - 'A' + 1) & 0x1f : c;
	if (*string != '\0') {
		c = *(string+1);
		key |= (c) ? ((c - 'A' + 1) & 0x1f) << 5 : c;
	}
	key &= size - 1;
	return key;
}


/* --- end --- */

