
/*
 * unpackfs.c                                                   (jh,05.02.2006)
 */

/*
 *  unpackfs: filesystem with transparent unpacking of archive files
 *  Copyright (C) 2005, 2006  Jochen Hepp <jochen.hepp@gmx.de>
 *
 *  This file is part of unpackfs.
 *
 *  unpackfs is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  unpackfs is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
	code is used from fuse/lib/helper.c:

	FUSE: Filesystem in Userspace
	Copyright (C) 2001-2005  Miklos Szeredi <miklos@szeredi.hu>

	libfuse can be distributed under the terms of the GNU LGPL.
	See the file COPYING.LIB.
*/

#define _GNU_SOURCE
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif
#include <unpackfs.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#	include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif
#ifdef HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif
#ifdef HAVE_ERRNO_H
#	include <errno.h>
#endif
#ifdef HAVE_MAGIC_H
#	include <magic.h>
#endif
#ifdef HAVE_FUSE_H
#	include <fuse.h>
#endif


const char *program;   /* argv[0] */
int debug;


/*
 * usage - shows how to use this program
 */
static void usage(int error)
{
	extern const char *program;
	FILE *out;

	out = (error == 0) ? stdout : stderr;

	fprintf(out, "Usage: %s mountpoint "
	        "[unpackfs opts] [fuse opts] [-o mount opts[,...]]\n\n",
	        (program != NULL) ? program : "unpackfs");

	fprintf(out,
		"Unpackfs options:\n"
		"    -u unpackdir           show contents of unpackdir at mountpoint\n"
		"    -t cachedir            use cachedir for temporary files\n"
		"    -c config              read configuration for magic unpacking from "
		                           "config\n"
		"    -l                     enable logging of magic unpacking "
		                           "(implies -f)\n"
		"    -V                     display version number\n"
		"    -h                     display this help and exit\n"
		"\n"
		"Fuse options:\n"
		"    -d                     enable debug output (implies -f)\n"
		"    -f                     foreground operation\n"
/*		"    -s                     disable multi-threaded operation\n" */
		"    -r                     mount read only (equivalent to '-o ro')\n"
		"\n"
	);
	fprintf(out,
		"Mount options:\n"
		"    default_permissions    enable permission checking\n"
		"    allow_other            allow access to other users\n"
		"    allow_root             allow access to root\n"
		"    kernel_cache           cache files in kernel\n"
		"    large_read             issue large read requests (2.4 only)\n"
		"    direct_io              use direct I/O\n"
		"    max_read=N             set maximum size of read requests\n"
		"    hard_remove            immediate removal (don't hide files)\n"
	);
	fprintf(out,
		"    debug                  enable debug output\n"
		"    fsname=NAME            set filesystem name in mtab\n"
		"    use_ino                let filesystem set inode numbers\n"
		"    readdir_ino            try to fill in d_ino in readdir\n"
		"    nonempty               allow mounts over non-empty file/dir\n"
		"    umask=M                set file permissions (octal)\n"
		"    uid=N                  set file owner\n"
		"    gid=N                  set file group\n"
	);
}


/*
 * invalid_option - print error for a not supported option
 */
static void invalid_option(const char *option)
{
	extern const char *program;

	fprintf(stderr, "%s: invalid option: %s\n"
	                "see `%s -h' for usage\n", program, option, program);
}


/*
 * missing_option_after - print error for a forgotten argument to an option
 */
static void missing_option_after(const char option)
{
	extern const char *program;

	fprintf(stderr, "%s: missing option after -%c\n"
	                "see `%s -h' for usage\n", program, option, program);

}


/*
 * add_option_to - add option opt to optp
 */
static int add_option_to(const char *opt, char **optp)
{
	size_t len, oldlen;
	char *buf;

	len = strlen(opt);

	if (*optp == NULL) {
		buf = (char *) xmalloc(len + 1);
		if (buf == NULL)
			return -1;
		strcpy(buf, opt);
	}
	else {
		oldlen = strlen(*optp);
		buf = (char *) xrealloc(*optp, oldlen + 1 + len + 1);
		if (buf == NULL) {
			free(*optp);
			*optp = NULL;
			return -1;
		}
		*(buf + oldlen) = ',';
		strcpy(buf + oldlen + 1, opt);
	}
	*optp = buf;
	return 0;
}


/*
 * add_option - add option to fuselib or to mount options
 */
static int add_options(char **fuselib_optp, char **mount_optp,
                       const char *opts)
{
	extern const char *program;
	int res;
	char *xopts, *s, *opt;
	int has_allow_other, has_allow_root;

	res = 0;

	xopts = xstrdup(opts);
	if (xopts == NULL) {
		res = -1;
		goto err_none;
	}

	s = xopts;
	has_allow_other = has_allow_root = 0;

	while((opt = strsep(&s, ",")) != NULL) {
		if (fuse_is_lib_option(opt)) {
			res = add_option_to(opt, fuselib_optp);
			/* Compatibility hack */
			if (strcmp(opt, "allow_root") == 0 && res != -1) {
				has_allow_root = 1;
				res = add_option_to("allow_other", mount_optp);
			}
		}
		else {
			res = add_option_to(opt, mount_optp);
			if (strcmp(opt, "allow_other") == 0)
				has_allow_other = 1;
		}
		if (res == -1)
			break;
	}

	if (has_allow_other && has_allow_root) {
		fprintf(stderr, "%s: 'allow_other' and 'allow_root' options are "
		                "mutually exclusive\n", program);
		res = -1;
	}

	free(xopts);

 err_none:
	return res;
}


/*
 * parse_cmdline - parse the options given to the program
 */
static int parse_cmdline(int argc, const char *argv[], char **mountpoint,
                         char **rootdir, char **cachedir, char **configfile,
                         char **mount_opts, char **fuselib_opts,
                         int *background, int *multithreaded)
{
	extern int debug;
	int res;
	const char *program;
	char *fsname_opt;
	int argctr;

	*mountpoint = *rootdir = *cachedir = *configfile = NULL;
	*mount_opts = *fuselib_opts = NULL;
	*background = 1;
	*multithreaded = 1;

	res = 0;

	program = basename(argv[0]);

	fsname_opt = unpackfs_string_cat("fsname=", program);
	if (fsname_opt == NULL) {
		res = -1;
		goto err_none;
	}

	res = add_options(fuselib_opts, mount_opts, fsname_opt);
	free(fsname_opt);
	if (res == -1)
		goto err;

	for (argctr = 1; argctr < argc; argctr ++) {
		if (argv[argctr][0] == '-') {
			if (strlen(argv[argctr]) == 2)
				switch (argv[argctr][1]) {
				case 'o':
					if (argctr + 1 == argc || argv[argctr+1][0] == '-') {
						missing_option_after('o');
						res = -1;
						goto err;
					}
					argctr ++;
					res = add_options(fuselib_opts, mount_opts, argv[argctr]);
					if (res == -1)
						goto err;
					break;

				case 'd':
					*background = 0;
					res = add_options(fuselib_opts, mount_opts, "debug");
					if (res == -1)
						goto err;
					break;

				case 'r':
					res = add_options(fuselib_opts, mount_opts, "ro");
					if (res == -1)
						goto err;
					break;

				case 'f':
					*background = 0;
					break;

				case 's':
					*multithreaded = 0;
					break;

				case 'u':
					if (argctr + 1 == argc || argv[argctr+1][0] == '-') {
						missing_option_after('u');
						res = -1;
						goto err;
					}
					*rootdir = xstrdup(argv[++argctr]);
					if (*rootdir == NULL) {
						res = -1;
						goto err;
					}
					break;

				case 't':
					if (argctr + 1 == argc || argv[argctr+1][0] == '-') {
						missing_option_after('t');
						res = -1;
						goto err;
					}
					*cachedir = xstrdup(argv[++argctr]);
					if (*cachedir == NULL) {
						res = -1;
						goto err;
					}
					break;

				case 'c':
					if (argctr + 1 == argc || argv[argctr+1][0] == '-') {
						missing_option_after('c');
						res = -1;
						goto err;
					}
					*configfile = xstrdup(argv[++argctr]);
					if (*configfile == NULL) {
						res = -1;
						goto err;
					}
					break;

				case 'l':
					*background = 0;
					debug = 1;
					break;

				case 'V':
					fprintf(stdout, "%s\n", PACKAGE_STRING);
					res = 1;
					goto err;

				case 'h':
					usage(0);
					res = 1;
					goto err;

				default:
					invalid_option(argv[argctr]);
					res = -1;
					goto err;
				}
			else {
				if (argv[argctr][1] == 'o') {
					res = add_options(fuselib_opts, mount_opts, &argv[argctr][2]);
					if (res == -1)
						goto err;
				} else if(strcmp(argv[argctr], "-ho") == 0) {
					usage(1);
					res = -1;
					goto err;
				} else {
					invalid_option(argv[argctr]);
					res = -1;
					goto err;
				}
			}
		}
		else if (*mountpoint == NULL) {
			*mountpoint = xstrdup(argv[argctr]);
			if (*mountpoint == NULL) {
				res = -1;
				goto err;
			}
		}
		else {
			invalid_option(argv[argctr]);
			res = -1;
			goto err;
		}
	}

	if (*mountpoint == NULL) {
		fprintf(stderr, "%s missing mountpoint\n"
		                "see `%s -h' for usage\n", program, program);
		res = -1;
		goto err;
	}

	return res;

 err:
	if (*mountpoint != NULL)
		free(*mountpoint);
	if (*rootdir != NULL)
		free(*rootdir);
	if (*cachedir != NULL)
		free(*cachedir);
	if (*configfile != NULL)
		free(*configfile);
	if (*mount_opts != NULL)
		free(*mount_opts);
	if (*fuselib_opts != NULL)
		free(*fuselib_opts);
	*mountpoint = *rootdir = *cachedir = *configfile = NULL;
	*mount_opts = *fuselib_opts = NULL;

 err_none:
	return res;
}


/*
 * main - main program
 */
int main(int argc, const char *argv[])
{
	extern const char *program;
	extern int debug;
	extern struct fuse_operations unpackfs_operations;
	int res, r;
	char *mountpoint, *rootdir, *cachedir, *config;
	char *mount_opts, *fuselib_opts, *tmp;
	int background, multithreaded, cachedir_temporary;
	struct unpackfs_magic_unpack_info *packer_info;
	magic_t magic;
	struct unpackfs_fuse *unpackfs_fuse_data;
	struct fuse *fuse;
	struct stat statbuf;

	program = basename(argv[0]);
	debug = 0;

	umask(umask(0) & 0077);


	res = parse_cmdline(argc, argv, &mountpoint, &rootdir, &cachedir, &config,
	                    &mount_opts, &fuselib_opts,
	                    &background, &multithreaded);
	if (res == -1)
		goto err_none;

	/* option for help or version */
	if (res == 1) {
		res = 0;
		goto err_none;
	}


	if (rootdir == NULL || *rootdir == '\0') {
		tmp = xstrdup("/");
		if (tmp == NULL) {
			res = -1;
			goto err_free_cmdline;
		}
		if (rootdir != NULL)
			free(rootdir);
		rootdir = tmp;
	}
	else if (stat(rootdir, &statbuf) != 0 || !S_ISDIR(statbuf.st_mode)) {
		fprintf(stderr, "%s: %s: No such directory\n", program, rootdir);
		res = -1;
		goto err_free_cmdline;
	}

	tmp = canonicalize_file_name(rootdir);
	if (tmp == NULL) {
		fprintf(stderr, "%s: %s: unable to canonicalize unpackdir\n"
		                "%s: %s\n", program, rootdir, program, strerror(errno));
		res = -1;
		goto err_free_cmdline;
	}
	if (rootdir != NULL)
		free(rootdir);
	rootdir = tmp;

	unpackfs_root_set(rootdir);


	cachedir_temporary = 0;
	if (cachedir == NULL || *cachedir == '\0') {
		tmp = unpackfs_tmp_template(program);
		if (tmp == NULL) {
			res = -1;
			goto err_free_cmdline;
		}
		if (cachedir != NULL)
			free(cachedir);
		cachedir = tmp;

		tmp = mkdtemp(cachedir);
		if (tmp == NULL) {
			fprintf(stderr, "%s: %s: unable to create temporary directory\n"
			                "%s: %s\n",
			                program, cachedir, program, strerror(errno));
			res = -1;
			goto err_free_cmdline;
		}
		cachedir = tmp;
		cachedir_temporary = -1;
	}
	else if (stat(cachedir, &statbuf) != 0 || !S_ISDIR(statbuf.st_mode)) {
		fprintf(stderr, "%s: %s: No such directory\n", program, cachedir);
		res = -1;
		goto err_free_cmdline;
	}

	tmp = canonicalize_file_name(cachedir);
	if (tmp == NULL) {
		fprintf(stderr, "%s: %s: unable to canonicalize cachedir\n"
		                "%s: %s\n", program, cachedir, program, strerror(errno));
		res = -1;
		goto err_remove_cachedir;
	}
	if (cachedir != NULL)
		free(cachedir);
	cachedir = tmp;

	unpackfs_cache_set(cachedir);


	if (config == NULL) {
		tmp = xstrdup(SYSCONFDIR "/unpackfs");
		if (tmp == NULL) {
			res = -1;
			goto err_remove_cachedir;
		}
		config = tmp;
	}


	res = unpackfs_config_packer(config, &packer_info);
	if (res != 0) {
		fprintf(stderr, "%s: unable to read configfile for magic unpacking\n"
		                "%s: %s: %s\n",
		                program, program, config, strerror(errno));
		goto err_remove_cachedir;
	}

	unpackfs_packer_info_set(packer_info);


	magic = unpackfs_magic_open();
	if (magic == NULL) {
		res = -1;
		goto err_free_packers;
	}

	unpackfs_magic_set(magic);


	unpackfs_fuse_data = unpackfs_fuse_mount(&unpackfs_operations,
	                                         sizeof(struct fuse_operations),
	                                         mountpoint, mount_opts,
	                                         fuselib_opts, background);
	if (unpackfs_fuse_data == NULL) {
		fprintf(stderr, "%s: unable to mount the filesystem\n", program);
		res = -1;
		goto err_magic_close;
	}

	fuse = unpackfs_fuse_data->ufsf_fuse_handle;


	/* event loop */

	/* magic library is not multi threaded */
	multithreaded = 0;

	if (multithreaded)
		res = fuse_loop_mt(fuse);
	else
		res = fuse_loop(fuse);


	unpackfs_fuse_unmount(unpackfs_fuse_data);

 err_magic_close:
	unpackfs_magic_close(magic);

 err_free_packers:
	unpackfs_config_packer_free(packer_info);

 err_remove_cachedir:
	if (cachedir_temporary != 0) {
		r = unpackfs_rmdir_recursive(cachedir);
		if (res == 0)
			res = r;
	}

 err_free_cmdline:
	if (mountpoint != NULL)
		free(mountpoint);
	if (rootdir != NULL)
		free(rootdir);
	if (cachedir != NULL)
		free(cachedir);
	if (config != NULL)
		free(config);
	if (mount_opts != NULL)
		free(mount_opts);
	if (fuselib_opts != NULL)
		free(fuselib_opts);

 err_none:
	return (res == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}


/* --- end --- */

