;;  Copyright (C) 2013 Aljosha Papsch <misc@rpapsch.de>
;;
;;  This file is part of Upmf.
;;
;;  Upmf is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  Upmf is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with Upmf.  If not, see <http://www.gnu.org/licenses/>.

(define validate-package-string
  (lambda (package-string with-version)
    (let* ((str-and-ver (string-split package-string #\:))
	    (name-and-sec (string-split (list-ref str-and-ver 0) #\/)))
      ;; Test if package-string contains a version number, if wanted
      (if (and (eq? with-version 'with-version) (eq? (length str-and-ver) 1))
	  (errormsg (string-append (_ "Package string malformed: ")
				   (_ "no version supplied: ")
				   package-string) 'fatal))
      ;; Test if package string contains section and name
      (if (or (< (length name-and-sec) 2) (> (length name-and-sec) 2))
	  (errormsg (string-append (_ "Package string malformed: ")
				   package-string) 'fatal))
      #t)))

(define is-installed?
  (lambda (package-obj release)
    (let* ((release-t (if (eq? release 1) "latest" release))
	   (package-dir (string-append package-dest-dir "/"
				       (assq-ref package-obj 'section)
				       "/" (assq-ref package-obj 'name)
				       "/" (car (get-release-for package-obj release-t 'quiet)))))

      (if (access? package-dir F_OK)
	  #t
	  #f))))

(define load-package
  (lambda (package-str)
    (let* ((pkg-and-ver (string-split package-str #\:))
	   (section-and-name (string-split (list-ref pkg-and-ver 0) #\/))
	   (package-file-name (string-append packagefile-directory
					     "/" (list-ref pkg-and-ver 0)
					     "/" (list-ref section-and-name 1)
					     ".scm")))
      (if (access? package-file-name F_OK)
	  (load package-file-name)
	  (errormsg (string-append (_ "Could not load package file: ")
				   package-file-name) 'fatal)))))

;; In package object PACKAGE-OBJ find release RELEASE.
;; If RELEASE is not found, return the first available release.
;; Print status message if VERBOSE is 'verbose.
;; Returns a pair of version string and download url (as specified
;; in the package files).
(define get-release-for
  (lambda (package-obj release verbose)
    (let* ((release-list (assq-ref package-obj 'releases))
	   (retval #f))
      (do ((k 0 (1+ k)))
	  ((> k (- (length release-list) 1)))
	(if (string=? (car (list-ref release-list k)) release)
	    (set! retval (list-ref release-list k))))
      (if (eq? retval #f)
	  (begin (if (eq? verbose 'verbose)
		     (begin (if (string=? "latest" release)
				(statusmsg (_ "Using first available release")
					   (car (list-ref release-list 0)))
				(statusmsg (_ "Release not found, using first available release")
					   release))))
		 (list-ref release-list 0))
	  retval))))

(define load-group
  (lambda (group-name)
    (let ((group-file (string-append packagefile-directory "/groups/"
				     group-name ".scm")))
      (if (access? group-file F_OK)
	  (load group-file)
	  (errormsg (string-append "Could not load group file: "
				   group-file) 'fatal)))))

(define install-group
  (lambda (group-list)
    (do ((k 0 (1+ k)))
	((> k (- (length group-list) 1)))
      (let ((group-list (load-group (list-ref group-list k))))
	(install-handler group-list)))))

