/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: sniffer.c,v 1.1 2008/03/21 11:14:25 awachtler Exp $ */
/**
 * @file
 * @brief Implementation of the @ref grpAppSniffer
 *
 * @ingroup grpAppSniffer
 */
/**
 * @note
 *   In order to get higher performance in frame receiving, this
 *   application uses a own implementation of the transceiver
 *   interrrupt routine (@ref SNIFFER_TRX_IRQ_vect),
 *   instead of the callback function based
 *   one, from librf23x.a.
 */
/* === includes ========================================== */
#include <avr/pgmspace.h>
#include <string.h>
#include <stdint.h>
#include <stdbool.h>
#include <util/crc16.h>


#include <avr/interrupt.h>

#include "transceiver.h"
#include "ioutil.h"
#include "timer.h"

/* === macros ============================================ */
#define UNIT_MS (1000)
#define SCAN_PERIOD_MS (1000)
#define SCAN_LOCK_DELAY_MS  (1)

#define NL "\n\r"

/* === types ============================================= */
/**
 * @brief Appication States.
 */
typedef enum
{
    /** Application is in idle mode, receiving commands from HIF. */
    IDLE,
    /** Application is in scanning mode. */
    SCAN,
    /** Application is in sniffing mode. */
    SNIFF
} sniffer_state_t;

/**
 * @brief Data structure for scan results.
 */
typedef struct scan_result_tag
{
    /** total number of received frames */
    uint16_t frames;
    uint16_t crc_ok;
    uint16_t edcnt;
    uint16_t edsum;
    uint16_t lqisum;
    uint16_t lqicnt;
    uint16_t ftypes[8];
} scan_result_t;

/**
 * @brief Data structure for internal state variables of
 * the application.
 */
typedef struct sniffer_context_tag
{
    /** state */
    sniffer_state_t state;
    /** channel offset */
    uint8_t         choffs;
    /** timer handle */
    timer_hdl_t     thdl;
    /** scan period */
    time_t          scanper;
    scan_result_t   scanres[16];
    uint8_t         scanres_reset;
    uint16_t frames;
    uint16_t irq_ur;


} sniffer_context_t;

typedef struct
{
    uint8_t irql;
    uint8_t tmrl;
    uint8_t err;
} dbg_t;




/* === globals =========================================== */
sniffer_context_t ctx;
static uint8_t rxbuf[MAX_FRAME_SIZE+10];
volatile uint8_t scan_done;

dbg_t dbg;

/* === prototypes ======================================== */
void scan_update_status(void);

/* === functions ========================================= */

/**
 * @brief Initialisation of hardware ressources.
 *
 * This function initializes the following components
 *  - LED Port
 *  - Timer Module
 *  - Host Interface
 *  - SPI Port for Radio
 *  - Radio (reset and state RX)
 */
void sniffer_init(void)
{
uint8_t val;
    /* init peripherals */
    memset(&ctx, 0, sizeof(ctx));
    memset((void*)&dbg, 0, sizeof(dbg));
    LED_INIT();
    timer_init();
    hif_init(38400);
    trx_io_init(DEFAULT_SPI_RATE);
    sei();

    PRINTF(NL"Sniffer 0.01 [%s]"NL, BOARD_NAME);


    /* init radio */
    TRX_RESET_LOW();
    TRX_SLPTR_LOW();
    DELAY_US(TRX_RESET_TIME_US);
    TRX_RESET_HIGH();
    trx_reg_write(RG_TRX_STATE, CMD_TRX_OFF);
    DELAY_US(2000);
    /* set TRX_OFF (in cas we come from P_ON) */
    val = trx_reg_read(RG_TRX_STATUS);
    if (val != TRX_OFF)
    {
        PRINTF("FATAL ERROR: TRX not in state TRX_OFF [0x%02x]"NL, val);
        while(1);
    }
    /* enable IRQ's that are needed */
    trx_reg_write(RG_IRQ_MASK, (TRX_IRQ_RX_START|TRX_IRQ_TRX_END|TRX_IRQ_UR));

    /* switch to RX */
    trx_reg_write(RG_TRX_STATE, CMD_RX_ON);
    DELAY_US(180);
    val = trx_reg_read(RG_TRX_STATUS);
    if (val != RX_ON)
    {
        PRINTF("FATAL ERROR: TRX not in state RX_ON [0x%02x]"NL, val);
        while(1);
    }
    PRINT("Listening ... "NL);
    LED_SET_VALUE(0);
}


inline void scan_update_frame(uint8_t flen, bool crc_ok, uint8_t lqi, uint8_t ed, uint8_t *rxbuf)
{
scan_result_t *scres;

     scres = &ctx.scanres[ctx.choffs&0xf];
     scres->frames ++;

     scres->edcnt ++;
     scres->edsum +=ed;

     scres->lqicnt ++;
     scres->lqisum += lqi;

     if (flen < 0x80)
     {
         /* process valid frame length */
         if (crc_ok == true)
         {
             scres->crc_ok++;
             scres->ftypes[rxbuf[0]&7] ++;
         }
         /* parse beacon */
     }

}

/**
 * @brief Interrupt service routine.
 */
#if defined(DOXYGEN)
void SNIFFER_TRX_IRQ_vect();
#else
ISR(TRX_IRQ_vect)

{
uint8_t cause;
    DI_TRX_IRQ();
    sei();
    cause = trx_reg_read(RG_IRQ_STATUS);

{
uint8_t ed, flen, lqi = 0;
bool crc_ok = 0;
uint8_t widx;

    if (dbg.irql) dbg.err = 1;
    dbg.irql ++;

    if ((ctx.state != SCAN) || (scan_done != 0))
    {
        /* abort IRQ, cause timer expired */
//        LED_TOGGLE(1);
        goto end;
    }

    if (cause & TRX_IRQ_UR)
    {
        LED_TOGGLE(1);
        ctx.irq_ur ++;
        goto end;
    }


    if (cause & TRX_IRQ_RX_START)
    {
        // get RSSI here
        LED_TOGGLE(0);

        // we hang around in the IRQ until
        // we can upload the frame in one Rutsch.
        DELAY_US(32);
        flen = trx_frame_get_length();
        widx = ((flen / 16) * 15) + 1;
        do
        {
            DELAY_US(32);
        }
        while(widx--);

        //flen = trx_frame_read_crc(rxbuf, MAX_FRAME_SIZE, &lqi, &crc_ok);
        ed = trx_reg_read(RG_PHY_ED_LEVEL);
        if (ctx.state == SCAN)
        {
            scan_update_frame(flen, crc_ok, lqi, ed, rxbuf);
        }

    }

    if (cause & TRX_IRQ_TRX_END)
    {
        ctx.frames++;
#if 0
        LED_TOGGLE(0);
        ctx.frames++;
        ed = trx_reg_read(RG_PHY_ED_LEVEL);
        flen = trx_frame_read_crc(rxbuf, MAX_FRAME_SIZE, &crc_ok);
        trx_sram_read(flen, 1, &lqi);
        if (ctx.state == SCAN)
        {
            scan_update_frame(flen, crc_ok, lqi, ed, rxbuf);
        }
#endif
    }

end:
    dbg.irql --;

}



    cli();
    EI_TRX_IRQ();
}
#endif

/**
 * @brief Timer routine for scan period.
 */
time_t timer_scan(timer_arg_t t)
{
    if (dbg.tmrl) dbg.err = 1;
    dbg.tmrl ++;
//    LED_TOGGLE(1);
    scan_done = 0x80  | ctx.choffs;
    dbg.tmrl --;
    return 0;
}

/**
 * @brief Start of the scan mode.
 */
void scan_start(void)
{
    ctx.state = SCAN;
    ctx.choffs = 0;
    PRINTF("Scanning channels, scan period %ums"NL,
           SCAN_PERIOD_MS);
    PRINT("             bad"
          "    Avg. "
          "        802.15.4 frames   "
          "    "NL);
    PRINT("chan   frm   crc "
          "  ed lqi "
          "     B     D     A     C "
          " PER"NL);

    ctx.thdl = timer_start(timer_scan,SCAN_PERIOD_MS,0);
}

/**
 * @brief Update Status in scan mode.
 */
void scan_update_status(void)
{
scan_result_t *scres;

    scres = &ctx.scanres[ctx.choffs];
    if(scres->edcnt == 0)
    {
        scres->edcnt += 1;
    }
    if(scres->lqicnt == 0)
    {
        scres->lqicnt +=1;
    }
    ctx.choffs = (ctx.choffs + 1) & 0x0f;
    if (ctx.scanres_reset)
    {
        scres = &ctx.scanres[ctx.choffs];
        memset(scres,0,sizeof(scan_result_t));
        if (ctx.scanres_reset < 32)
        {
            ctx.scanres_reset --;
        }
    }
}

/**
 * @brief Display scan result.
 */
void scan_show_status(uint8_t choffs)
{
static uint16_t updates = 0;
scan_result_t *scres;
uint16_t per = 0;

    updates++;
    scres = &ctx.scanres[choffs&0x0f];
    if (scres->frames > 0)
    {
        per = (scres->frames - scres->crc_ok) * 100 / scres->frames;
    }
    PRINTF(" %2d  % 5u % 5u "
           " % 3u %3u ",
            choffs+11, scres->frames, scres->frames - scres->crc_ok,
            scres->edsum / scres->edcnt,
            scres->lqisum / scres->lqicnt
            );

    PRINTF(" % 5u % 5u % 5u % 5u"
           "  % 3u"NL,
            scres->ftypes[0], scres->ftypes[1], scres->ftypes[2], scres->ftypes[3],
            per
            );
    if (choffs == 15)
    {
        PRINTF("== ur %d err %d frames: %ld\n", ctx.irq_ur, dbg.err, ctx.frames);
        /*         0   1   2   3   4   5   6   7   8   9   10  11  12  13  14  15  16 */
        hif_puts("\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\n");
    }
    hif_puts("\r*\r\b");


    if (ctx.scanres_reset)
    {
        hif_puts("\rR------------------------------------------------------\r\b");
    }
    else
    {
        hif_puts("\r*\r\b");
    }

}

/**
 * @brief Stop of the scan mode.
 */
void scan_stop(void)
{
    ctx.state = IDLE;
    ctx.choffs = 0;
    ctx.thdl = timer_stop(ctx.thdl);
}


/**
 * @brief Key processing.
 */
bool ctrl_process_hotkey(char cmdkey)
{
    bool ret = true;
    switch (cmdkey)
    {

        case 'S':
            scan_start();
            break;

        case 's':
            PRINT("SNIFF\n");
            scan_stop();
            break;

        case '+':
            if (ctx.state != SCAN)
            {
                ctx.choffs = (ctx.choffs + 1) & 0xf;
                trx_bit_write(SR_CHANNEL, ctx.choffs + 11);
                PRINTF("Channel %u"NL, ctx.choffs + 11);
            }
            break;

        case '-':
            if (ctx.state != SCAN)
            {
                ctx.choffs = (ctx.choffs - 1) & 0xf;
                trx_bit_write(SR_CHANNEL, ctx.choffs + 11);
                PRINTF("Channel %u"NL, ctx.choffs + 11);
            }
            break;

        case 'r':
            if (ctx.scanres_reset != 0)
            {
                ctx.scanres_reset = 0;
            }
            else
            {
                ctx.scanres_reset = 16;
            }
            break;

        case 'R':
            ctx.scanres_reset ^= 32;
            ctx.scanres_reset &= 32;
            break;

        default:
            ret = false;
            break;
    }
    return ret;
}

/**
 * @brief Command processing.
 */

bool ctrl_process_command(char * cmd)
{
    return false;
}


/**
 * @brief Process data received from HIF.
 */
void ctrl_process_input(void)
{
bool success;
uint16_t inchar;
static char  cmdline[16];
static uint8_t  cmdidx = 0;

    /* command processing */
    inchar = hif_getc();
    if(inchar<0x100)
    {
        cmdline[cmdidx++] = (char) inchar;
        if (inchar == '\n' || inchar == '\r')
        {
            cmdline[cmdidx-1] = 0;
            ctrl_process_command(cmdline);
            cmdidx = 0;
        }
        else if (cmdidx == 1)
        {
           hif_putc('\r');
           success = ctrl_process_hotkey(cmdline[0]);
           if (success == true)
           {
                cmdidx = 0;
           }
        }
    }

}

/**
 * @brief Main function of sniffer application.
 */
#if defined(DOXYGEN)
int sniffer_main(void)
#else
int main(void)
#endif
{

    sniffer_init();

    while(1)
    {
        ctrl_process_input();

        if (scan_done)
        {
            scan_update_status();
            scan_show_status(scan_done & 0x0f);
            trx_bit_write(SR_CHANNEL, ctx.choffs + 11);
            cli();
            scan_done = 0;
            sei();
            ctx.thdl = timer_start(timer_scan,SCAN_PERIOD_MS,0);
        }
    }
}


/* EOF */
