/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: transceiver.h,v 1.1 2008/03/21 11:14:26 awachtler Exp $ */
/**
 * @file
 * @brief Transceiver function interface.
 *
 */

#ifndef TRANSCEIVER_H
#define TRANSCEIVER_H

/* === Includes ============================================================== */
#include "board.h"

#if RADIO_TYPE == RADIO_AT86RF230 || defined(DOXYGEN)
# include "rf230/trx_rf230.h"
# include "rf230/at86rf230.h"
# include <stdbool.h>
#else
# error "RADIO_TYPE is not defined or wrong"
#endif

/* === Externals ============================================================= */

/* === Types ================================================================= */

#if defined(DOXYGEN)

    /** @todo
       this types are defined in rf230/trx_rf230.h in order
       to provide a radio abstraction ... */

    /** Data Type for Transceiver SRAM address
     * @ingroup grpTrx
     */
    typedef uint8_t trx_ramaddr_t;

    /** Data Type for Transceiver register value
     * @ingroup grpTrx
     */
    typedef uint8_t trx_regval_t;

    /** Data Type for Transceiver register address
     * @ingroup grpTrx
     */
    typedef uint8_t trx_regaddr_t;

#endif
/** Data Type for Transceiver IRQ callback function
 * @ingroup grpTrx
 */
typedef void (*trx_irq_handler_t)(uint8_t cause);

/**
 * Transceiver parameter structure
 * @ingroup grpTrx
 */
typedef struct
{
    /** current channel see sub register @ref SR_CHANNEL*/
    uint8_t chan  : 5;
    /**  TX power index see sub register @ref SR_TX_PWR*/
    uint8_t txp   : 4;
    /**  CCA mode see sub register @ref SR_CCA_MODE */
    uint8_t cca   : 2;
    /**  ED threshold see sub register @ref SR_CCA_ED_THRES */
    uint8_t edt   : 4;

    /**  clkm control  see sub register @ref SR_CLKM_CTRL */
    uint8_t clkm  : 3;
    /**  protection CRC */
    uint16_t crc;
} trx_param_t;



typedef enum
{
   CFG_FLASH,
   CFG_EEPROM,
   CFG_NONE
} trx_cfg_t;

/* === Macros ================================================================ */
#define MIN_CHANNEL (11)
#define MAX_CHANNEL (26)
#define NB_CHANNELS (15)


/* === Prototypes ============================================================ */
#ifdef __cplusplus
extern "C" {
#endif

/**
 * @brief
 *    SPI Initialization (RADIO_TYPE == AT86RF230)
 *
 *    Init the Serial Peripherial Interface of the AVR
 *
 * @param spirate Configuration Byte of the SPI Control Register (SPCR)
 *
 * @ingroup grpTrx
 */
void trx_io_init (uint8_t spirate);


/** set function pointer for IRQ handler
 *
 * @ingroup grpTrx
 */
void trx_set_irq_handler(trx_irq_handler_t irqhandler);

/**
 * @brief Write Register
 *
 * This function write to a transceiver register.
 * @param addr Address of the Register in the Transceiver (Offset) that should be written
 * @param val Byte that will be written into the Register
 *
 * @ingroup grpTrx
 */
void trx_reg_write(trx_regaddr_t addr, trx_regval_t val);

/**
 * @brief Read Register
 *
 * This function reads a transceiver register.
 * @param addr Address of the Register in the Transceiver (Offset) that should be read
 * @return Contents of the Register
 *
 * @ingroup grpTrx
 */
uint8_t trx_reg_read(trx_regaddr_t addr);



/**
 * @brief subregister read
 *
 * @param   addr  offset of the register
 * @param   mask  bit mask of the subregister
 * @param   pos   bit position of the subregister
 * @retval  data  pointer where the read and demuxed value is stored
 *
 * @code
 *   pos = 4, mask = 0xf0
 *   register value = 0xA5
 *   *data = 0x0A
 * @endcode
 *
 * @ingroup grpTrx
 */
trx_regval_t trx_bit_read(trx_regaddr_t addr, trx_regval_t mask, uint8_t pos);


/**
 * @brief subregister write
 *
 * @param   addr  offset of the register
 * @param   mask  bit mask of the subregister
 * @param   pos   bit position of the subregister
 * @retval  value  data, which is muxed into the register
 *
 * @code
 *   pos = 4, mask = 0xf0
 *   register value = 0xA5 (before operation)
 *   value = 0x05
 *   register value = 0x55 (after operation)
 * @endcode
 *
 * @ingroup grpTrx
 */
void trx_bit_write(trx_regaddr_t addr, trx_regval_t mask, uint8_t pos, trx_regval_t value);

/**
 * @brief Frame Write
 *
 * This function writes a frame to the transceiver.
 *
 * @param length Length of the frame that should be written into the frame buffer
 * @param data Pointer to an array of (Payload-) bytes that should be sent
 * @note SLP_TR! (RADIO_TYPE == AT86RF230)
 *
 * @ingroup grpTrx
 */
void trx_frame_write(uint8_t length, uint8_t *data);

/**
 * @brief Frame Read
 *
 * This function reads a frame from the transceiver.
 *
 * @retval data Pointer to an array of (Payload-) bytes that should be sent
 * @param datasz maximum number of bytes, which fit in the data buffer.
 * @retval lqi Pointer where the LQI value is stored
 * @return length of the downloaded frame (including the LQI byte [RADIO_TYPE == AT86RF230])
 *
 * @ingroup grpTrx
 */
uint8_t trx_frame_read(uint8_t *data, uint8_t datasz, uint8_t *lqi);


/**
 * @brief Frame Read with CRC check (and crc value kept)
 *
 * This function reads a frame from the transceiver.
 * While the upload is in progress, the CRC16 value is caluculated
 * and compared against the last two bytes.
 *
 * @retval data Pointer to an array of (Payload-) bytes that should be sent
 * @param  datasz maximum number of bytes, which fit in the data buffer.
 * @retval crc_ok Result of the CRC16 check.
 *
 * @return length of the downloaded frame (including the LQI byte [RADIO_TYPE == AT86RF230])
 *
 * @todo implement this function
 * @ingroup grpTrx
 */
uint8_t trx_frame_read_crc(uint8_t *data, uint8_t datasz, bool *crc_ok);


/**
 * @brief Frame Read with CRC check (and crc value suppressed)
 *
 * This function reads a frame from the transceiver.
 * While the upload is in progress, the CRC16 value is caluculated
 * and compared against the last two bytes.
 * The crc bytes are not copied in the data buffer.
 *
 * @retval data Pointer to an array of (Payload-) bytes that should be sent
 * @param datasz maximum number of bytes, which fit in the data buffer.
 * @retval lqi Pointer where the LQI value is stored
 * @retval crc_ok Result of the CRC16 check.
 *
 * @return length of the downloaded frame (including the LQI byte [RADIO_TYPE == AT86RF230])
 *
 * @todo implement this function
 * @ingroup grpTrx
 */
uint8_t trx_frame_read_data_crc(uint8_t *data, uint8_t datasz, uint8_t *lqi, bool *crc_ok);

/**
 * @brief Get length of a received frame
 *
 *  @return length of the received frame
 *
 * @ingroup grpTrx
 */

uint8_t trx_frame_get_length(void);

/**
 * @brief Write SRAM
 *
 * This function writes into the SRAM of the transceiver.
 *
 * @param addr Address in the TRX's SRAM where the write burst should start
 * @param length Length of the write burst
 * @param data Pointer to an array of bytes that should be written
 *
 * @ingroup grpTrx
 */
void trx_sram_write(trx_ramaddr_t addr, uint8_t length, uint8_t *data);

/**
 * @brief Read SRAM
 *
 * @param addr Address in the TRX's SRAM where the read burst should start
 * @param length Length of the write burst
 * @retval data Pointer to an array of bytes that should be read
 *
 * @ingroup grpTrx
 */
void trx_sram_read(trx_ramaddr_t addr, uint8_t length, uint8_t *data);

/**
 * @brief Get static transceiver parameters
 *
 * This function reads the static transceiver parameters,
 * defined in a structure and protects it with a CRC16.
 *
 *  @param p pointer to the data structure.
 * @ingroup grpTrx
 */
void trx_parms_get(trx_param_t *p);

/**
 * @brief Set static transceiver parameters
 *
 * This function writes the static transceiver parameters,
 * defined in a structure  protects it with a CRC16.
 *
 *  @param p pointer to the data structure.
 *  @param no_crc_check if this parameter is true, the
 *                       the CRC given in the sructure is not checked.
 *  @return 0 if OK, 1 if error.
 * @ingroup grpTrx
 */
uint8_t trx_parms_set(trx_param_t *p, bool no_crc_check);

/**
 * @brief Read a parameter structure from eeprom or flash location.
 *
 * The function first reads the EEPROM location, when it is not 0,
 * and the CRC of the structure is OK, the data are copied to p
 *
 *  @param eeparm address offset in EEPROM (must be not 0)
 *  @param flashparm adress in  the flash/program memory.
 *  @param p  pointer to transceiver parameter structure in RAM.
 *  @return a trx_cfg_t value, which represents which configuration is used.
 *
 * The following example shows, how
 * a structure with default parameters is created in the flash memory.
 * Note that the crc field must contain a valid CRC16.
 * @code
 #include <avr/pgmspace.h>

 trx_param_t PROGMEM trxp_flash = {chan: 13,
                                   txp: 0,
                                   cca: 1,
                                   edt: 11,
                                   clkm: 0,
                                   crc: 0x0355};
 @endcode
 * @ingroup grpTrx
 */
trx_cfg_t trx_parms_read(const void * eeparm, const void * flashparm, trx_param_t *p);


/**
 * @brief Write a parameter structure to eeprom.
 *
 * The function first computes the crc field in the RAM structure and
 * writes it to the EEPROM. This is done only, if eeparm is not 0.
 *
 *  @param eeparm address offset in EEPROM (must be not 0)
 *  @param p  pointer to transceiver parameter structure in RAM.
 *  @return CFG_NONE, if not successful, CFG_EEPROM if OK.
 * @ingroup grpTrx
 */
trx_cfg_t trx_parms_write(const void * eeparm, trx_param_t *p);


#ifdef __cplusplus
} /* extern "C" */
#endif


#endif /* TRANSCEIVER_H */
