/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: wgpio.c,v 1.2 2008/04/15 05:21:36 awachtler Exp $ */
/**
 * @file
 * @brief Implementation of the @ref grpAppWgpio.
 *
 * This is a Wireless General Purpose IO Application
 * a.k.a. a light switch application.
 *
 * @ingroup grpAppWgpio
 */


/* === includes ========================================== */
#include <avr/pgmspace.h>
#include <string.h>
#include <stdint.h>
#include <stdbool.h>
#include <util/crc16.h>
#include <avr/interrupt.h>

#include "transceiver.h"
#include "ioutil.h"
#include "timer.h"

/* === macros ============================================ */

/* === types ============================================= */

/**
 * Application state type
 */
typedef enum
{
    IDLE, TX, RX
} state_t;


typedef struct
{
    uint16_t frmctl;
    uint8_t  seq;
    uint8_t  keystat;
    uint16_t crc;
} key_frame_t;

/* === globals =========================================== */
state_t state;
uint8_t rxbuf[10];
volatile uint8_t keys;

trx_param_t trxparm = {
            chan  : 12,
            txp   : 0,
            cca   : 3,
            clkm  : 0,
            crc   : 0
            };

/* === prototypes ======================================== */

/* === functions ========================================= */
/**
 * Application specific transceiver IRQ routine.
 *
 */
#if defined(DOXYGEN)
void WGPIO_TRX_IRQ_vect();
#else
ISR(TRX_IRQ_vect)
{
uint8_t cause, flen, i;
uint16_t crc;

    //DI_TRX_IRQ();
    //sei();
    cause = trx_reg_read(RG_IRQ_STATUS);
    if (cause & TRX_IRQ_TRX_END)
    {
        if (state == TX)
        {
            trx_reg_write(RG_TRX_STATE, CMD_RX_ON);
            state = RX;
        }
        else if (state == RX)
        {
            /* protect frame */
            trx_reg_write(RG_TRX_STATE, CMD_PLL_ON);
            flen = trx_frame_read (rxbuf, 10, NULL);
            trx_reg_write(RG_TRX_STATE, CMD_RX_ON);
            crc = 0;
            for (i = 0; i< flen-1; i++)
            {
                crc = _crc_ccitt_update(crc, rxbuf[i]);
            }
            if (crc == 0)
            {
                LED_TOGGLE(1);
            }
        }
    }
    //cli();
    //EI_TRX_IRQ();
}
#endif

/**
 * Application specific timer IRQ routine.
 * used to read key status (debounced).
 */

#if defined(DOXYGEN)
void WGPIO_TIMER_IRQ_vect();
#else
ISR(TIMER_IRQ_vect)
{
    keys = keys_debounced();
}
#endif


/**
 * Initialization function.
 */
void wgpio_init()
{
 uint8_t val;
    LED_INIT();
    KEY_INIT();
    TIMER_INIT();
    trx_io_init(DEFAULT_SPI_RATE);
    sei();

    /* init radio */
    TRX_RESET_LOW();
    TRX_SLPTR_LOW();
    DELAY_US(TRX_RESET_TIME_US);
    TRX_RESET_HIGH();
    trx_reg_write(RG_TRX_STATE, CMD_TRX_OFF);
    DELAY_US(2000);
    /* set TRX_OFF (in cas we come from P_ON) */
    val = trx_reg_read(RG_TRX_STATUS);
    if (val != TRX_OFF)
    {
        while(1);
    }
    /* enable IRQ's that are needed */
    trx_reg_write(RG_IRQ_MASK, TRX_IRQ_TRX_END);

    /* switch to RX */
    trx_reg_write(RG_TRX_STATE, CMD_RX_ON);
    state = RX;
    DELAY_US(180);
    val = trx_reg_read(RG_TRX_STATUS);
    if (val != RX_ON)
    {
        while(1);
    }

    /* enable auto_crc,
       use trx_reg_read/write in order to save memory */
    val = trx_reg_read(RG_PHY_TX_PWR) | 0x80;
    trx_reg_write(RG_PHY_TX_PWR, val);

    LED_SET_VALUE(0);
}

void wgpio_send_status(uint8_t keystat)
{
static uint8_t txseq = 0;
key_frame_t frm;

    /* init data */
    frm.frmctl = 1;
    frm.seq = txseq ++;
    frm.keystat = keystat;

    /* send frame */
    trx_reg_write(RG_TRX_STATE, CMD_PLL_ON);
    trx_frame_write (sizeof(frm), (uint8_t*)&frm);
    TRX_SLPTR_HIGH();
    TRX_SLPTR_LOW();
    state = TX;
}
/**
 * Main function of WGPIO application.
 */
#if defined(DOXYGEN)
int wgpio_main(void);
#else
int main(void)
#endif
{
   wgpio_init();
   while(1)
   {
        if (keys)
        {
        # if 0
            trx_reg_write(RG_TRX_STATE, CMD_PLL_ON);
            trx_frame_write (5, &keys);
            TRX_SLPTR_HIGH();
            TRX_SLPTR_LOW();
            state = TX;
        #else
            wgpio_send_status(keys);
        #endif
            LED_PORT ^= (keys<<LED_SHIFT) & LED_MASK;
            keys = 0;
        }
   }
}
