# $Id: usertools.ph,v 1.22 2004/06/01 21:55:54 forun Exp $
#
# usertools.ph - header for functions used by UserTools
#
# Copyright (C) 2002 Steven Barrus
# Copyright (C) 2002 Dana Dahlstrom
# Copyright (C) 2002 Robert Ricci
# Copyright (C) 2002 Spencer Visick
#
# See the AUTHORS file for contact info
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use Net::LDAP qw(:all);
use Net::LDAP::Extension::SetPassword;
use Term::ReadKey;
use Crypt::Cracklib;
use Crypt::PasswdMD5;
use Text::ParseWords;

read_config();

sub read_config {
  my $user = (getpwuid($>))[0];
  %config = (port => 389, 
             scope => "subtree",
             idle_timelimit => 0,
             default_shell => "ask",
             password_hash => "crypt",
             homedir_command => "mkdir",
	     pw_dict => "/usr/local/share/cracklib/pw_dict",
             version => 3);
  
  my $rc = parse_config("/etc/usertools.conf");
  $rc &= parse_config("$ENV{HOME}/.usertoolsrc");
  if ($rc == 1) {
    warn "No config file found\n";
    exit 1;
  }

  $config{userbase} = "ou=People,$config{base}" if !$config{userbase};
  $config{groupbase} = "ou=Group,$config{base}" if !$config{groupbase};
  $config{binddn} = "uid=$user,ou=People,$config{base}" if !$config{binddn};
}

sub parse_config { 
  my $conf = shift;
  my @opts = ("host", "port", "base", "userbase", "groupbase", "sslversion",  
              "password_hash", "binddn", "idle_timelimit", "default_shell", 
              "group_attribute", "user_attribute", "group_objectclass", 
              "user_objectclass", "password_attribute", "required_attribute", 
              "default_gid", "homedir_command", "version", "cafile", "pw_dict");

  if (-r $conf){
    open (CONF, $conf) || return 1;
    while (<CONF>){
      if (/^#/){
        next;
      }
      $supported = undef;
      foreach $opt (@opts) {
        if (/^$opt\b/i){
          @option = split();
          shift(@option);
          # I have thought about lc() these, but I think it will
          # lead to problems. Maybe we should just lc hash, etc...
          $config{$opt} = shift(@option);
          for(@option){
            $config{$opt} = $config{$opt} . " " . $_;
          }
          $supported = 1;
          last; #save a little time
        }
      }
      if (!$supported){
        warn "Ignoring unsupported option: " . (split())[0] . "\n";
      }
    }
    return 0;
  }

  return 1;
}

sub ldap_connect {

  if (@_) {
    my ($binddn, $base) = @_;
    $config{binddn} = $binddn;
    $config{base} = $base;
  }

  die "No server hostname specified.\n" if !$config{host};
  
  my $ldap;
  my $pw;
  my $mesg;
  
  $ldap = Net::LDAP->new($config{host}, version => $config{version}) 
          or die "$@";

  if (defined($config{sslversion}) && lc($config{sslversion}) ne "none"){
    if (-e $config{cafile}){
      $mesg = $ldap->start_tls(verify => 'require',
                               sslversion => "$config{sslversion}",
                               cafile => "$config{cafile}");
      $mesg->code && die "Failed to start TLS: ", $mesg->error;
    }else{
      die "CA certificate $config{cafile} does not exist.\n"
    }
  }

  do {
    $pw = read_pass();
    $mesg = $ldap->bind($config{binddn}, password => $pw);

    if ($mesg->code) {
      print STDERR "Could't connect to $config{host}. ", $mesg->error, "\n"; 
      if ($mesg->code != LDAP_INVALID_CREDENTIALS) {
        exit(1);
      }
    }
  } while ($mesg->code == LDAP_INVALID_CREDENTIALS);
    
  return $ldap;
}

sub ldap_connect_anon {
  
  my $ldap = Net::LDAP->new($config{host}, version => $config{version}) or die "$@";
  
  if (@_) {
    my ($base) = @_;
    $config{base} = $base;
  }
  my $mesg = $ldap->bind();
    die "Could't connect to $config{host}.\n", $mesg->error if $mesg->code;

  return $ldap;
}

sub read_pass {
  my $pw;
  print STDERR "Password: ";
  ReadMode 2;
  chomp($pw = <STDIN>);
  ReadMode 0;
  print STDERR "\n";
  return $pw;
}

sub get_new_pass_plain {
  my $gotpasswd = 0;

  while (1) {
    print STDERR "New ";
    $password = read_pass();

    $reason = fascist_check($password, $config{pw_dict});
    unless ($reason eq "ok") {
      print STDERR "BAD PASSWORD: $reason\n";
      next;
    }
    
    print STDERR "Retype New ";
    $password2 = read_pass();

    if ($password eq $password2) {
      return($password);
    } else {
      print STDERR "Passwords don't match.\n";
    }
  }
}

sub get_new_pass {
  return(enc_passwd(get_new_pass_plain()));
}

#generates a 'random' password
sub gen_pass {
  my @chars = ('a' .. 'k','m' .. 'z',2..9);
  my $pass = "";
  for (1 .. 8) {
    $pass .= $chars[rand @chars];
  }
  return $pass;
}

sub set_pass_update {
  my ($ldap, $entry, $pass) = @_;
  my $mesg;

  if ($config{password_hash} eq "exop") {
    $mesg = $entry->update($ldap);
    return $mesg if $mesg->code;
    $mesg = $ldap->set_password( user=> $entry->dn, newpasswd => $pass );
    return $mesg;
  } else {
    $entry->replace( userPassword => $pass );
  }

  $mesg = $entry->update($ldap);

  return $mesg;
}

sub enc_passwd {
  my $password = shift;
  my @chars = ( 'A' .. 'Z', 'a' .. 'z', '0' .. '9', '/', '.' );
  my $salt = "";
  if ($config{password_hash} eq "crypt") {
    $salt = $chars[rand @chars] . $chars[rand @chars];
    return ("{crypt}" . crypt($password,$salt));
  }
  if ($config{password_hash} eq "md5") {
    for (1..8) {
      $salt .= $chars[rand @chars];
    }
    return ("{md5}" . unix_md5_crypt($password,$salt));
  }
  # If not crypt of md5 we want plain
  return ($password);
}

# Parse a string of comma seperated values.
# This function allows commas in values if they
# are escaped with a \ or in quotes.
sub parse_csv {
  return quotewords('\s*,\s*',0, $_[0]);
}

1;
