/* Vgrid - Virtual grid program for radiology
   Copyright (C) 2020, 2021 Sonia Diaz Pacheco.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// defined in main.cc
extern int verbosity;

class Pretty_print		// requires global var 'int verbosity'
  {
  std::string name_;
  std::string padded_name;
  const char * const stdin_name;
  unsigned longest_name;
  mutable bool first_post;

public:
  Pretty_print( const std::vector< std::string > & filenames )
    : stdin_name( "(stdin)" ), longest_name( 0 ), first_post( false )
    {
    if( verbosity < 0 ) return;
    const unsigned stdin_name_len = std::strlen( stdin_name );
    for( unsigned i = 0; i < filenames.size(); ++i )
      {
      const std::string & s = filenames[i];
      const unsigned len = ( s == "-" ) ? stdin_name_len : s.size();
      if( longest_name < len ) longest_name = len;
      }
    if( longest_name == 0 ) longest_name = stdin_name_len;
    }

  void set_name( const std::string & filename )
    {
    if( filename.size() && filename != "-" ) name_ = filename;
    else name_ = stdin_name;
    padded_name = "  "; padded_name += name_; padded_name += ": ";
    if( longest_name > name_.size() )
      padded_name.append( longest_name - name_.size(), ' ' );
    first_post = true;
    }

  void reset() const { if( name_.size() ) first_post = true; }
  const char * name() const { return name_.c_str(); }
  void operator()( const char * const msg = 0, FILE * const f = stderr ) const;
  };


inline int real_bits( unsigned value )
  {
  int bits = 0;
  while( value > 0 ) { value >>= 1; ++bits; }
  return bits;
  }


struct Error
  {
  const char * const msg;
  explicit Error( const char * const s ) : msg( s ) {}
  };


struct Color_info				// 3 x rows x cols
  {
  std::vector< std::vector< std::vector< unsigned short > > > data;

  void resize( const long rows, const long cols )
    {
    data.clear(); data.resize( 3 );		// 3 color planes (RGB)
    for( unsigned long plane = 0; plane < data.size(); ++plane )
      {
      data[plane].resize( rows );
      for( unsigned long row = 0; row < data[plane].size(); ++row )
        data[plane][row].reserve( cols );
      }
    }
  };

struct Raw_params { unsigned width, height, bit_depth, channels; };


class Matrix;			// forward declaration

// defined in main.cc
void show_error( const char * const msg, const int errcode = 0,
                 const bool help = false );
void show_file_error( const char * const filename, const char * const msg,
                      const int errcode = 0 );

// defined in png_io.cc
bool read_check_png_sig8( FILE * const f );
int raw_to_png( const char * const input_filename, FILE * const infile,
                FILE * const f, const Raw_params & raw_params );
int show_png_info( const char * const input_filename, FILE * const f,
                   const int sig_read );

// defined in recontrast.cc
int recontrast( const Matrix & image, FILE * outfile, const int maxval,
                const unsigned bit_depth, const int median_radius,
                const Color_info & color_info, const bool expand );

// defined in self_test.cc
int self_test1();
int self_test2( const Matrix & image, const int maxval,
                const char * const program_name );
