/**
 * @file  wcngdmn.c
 * @copyright Copyright (C) 2019-2020 Michał Bąbik
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @brief Wallpaper changer
 *
 * Automatic wallpaper changer
 *
 * @date May 24, 2020
 *
 * @version 1.4.7
 *
 * @author Michał Bąbik <michalb1981@o2.pl>
 */
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <err.h>
#include "cmdfn.h"
#include "dmfn.h"
#include "chkwch.h"
#include "randomm.h"
/*----------------------------------------------------------------------------*/
/**
 * @brief Main function.
 *
 * @param[in] argc Arguments passed to the program from the environment in
 *                 which the program is run
 * @param[in] argv Pointer to the first element of an array of pointers that
 *                 represent the arguments passed to the program
 * @return    Return value
 */
int
main (int    argc,
      char **argv)
{
    uint32_t  ui_cnt    = 0;    /* Minute counter */
    uint32_t  ui_ch_int = 0;    /* Change interval */
    int       i_opt     = 0;    /* Command line options */
    char     *s_cfgfile = NULL; /* Config file path */
    RandMem   rm_rand;          /* Ramdom memory structure */

    /* Parse command line options */
    cmdfn_parse (argc, argv, &i_opt, &s_cfgfile);
    /* Init random number structure */
    randomm_init (&rm_rand);

    /* Printing status */
    if (i_opt & CMD_OPT_STATUS) {
        dmfn_print_status ();
        exit (EXIT_SUCCESS);
    }
    /* Stopping daemon */
    if (i_opt & CMD_OPT_STOP) {
        dmfn_kill ();
        exit (EXIT_SUCCESS);
    }
    /* Restarting daemon - stopping existing one */
    if (i_opt & CMD_OPT_RESTART && dmfn_kill ()) {
        sleep500 ();
    }
    /* Check wchangerd process presence, exit if it is running */
    dmfn_check_exit ();
    /* Checking if display is available */
    check_display_exit ();
    /* Check config file correctness */
    check_config_file (&s_cfgfile);
    /* Load settings and set wallpaper */
    ui_ch_int = check_settings_change_wallpaper (s_cfgfile, &rm_rand);

    /* Starting daemon */
    if ((i_opt & CMD_OPT_START) || (i_opt & CMD_OPT_RESTART)) {
        puts ("Starting wchangerd daemon");
        dmfn_daemonize ();
    }
    while (1) {

        if (++ui_cnt > ui_ch_int) {
            ui_ch_int = check_settings_change_wallpaper (s_cfgfile, &rm_rand);
            ui_cnt = 1;
        }
        sleep (60);
    }
    free (s_cfgfile);
    return 0;
}
/*----------------------------------------------------------------------------*/

