#!/usr/bin/env python
# -*- coding: utf-8 -*- #
'''WatchVideo Preferences'''
###
#
# WatchVideo is the legal property of Leonardo Gastón De Luca
# leo[at]kde.org.ar Copyright (c) 2009 Leonardo Gastón De Luca
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.

# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
###

import os

from PyQt4 import QtCore, QtGui
from PyQt4.QtGui import QComboBox, QRadioButton
from watchvideo.ui_preferences import Ui_PreferencesWindow
from watchvideo import notification
from watchvideo import gui_utils
from watchvideo import main
import watchvideo.constants as c
from getmediumurl import Matcher

from watchvideo.utils import is_command

class Preferences(QtGui.QDialog):
    """The class responsible for the Preferences window."""
    
    def __init__(self, parent, settings):
        """Initialize the GUI."""
        
        QtGui.QDialog.__init__(self, parent)
        self.ui = Ui_PreferencesWindow()
        self.ui.setupUi(self)
        
        self.settings = settings
        self.parent = parent
        self.ordered_options = []
        self.update_global_quality = True

        #Populate the QTreeWidget with the plugins' names and options
        self.ui.tree_quality.header().resizeSection(0, 350)
        plugins = Matcher().plugins.values()
        for plugin in plugins:
            plug_name = plugin.get_plugin_name()

            if plugin.has_high_quality():
                cb_qualityOptions = QComboBox(self)
                cb_qualityOptions.addItem("Normal")
                cb_qualityOptions.addItem("High")
                
                item = QtGui.QTreeWidgetItem([plug_name, ""])
                self.ui.tree_quality.addTopLevelItem(item)
                self.ui.tree_quality.setItemWidget(item, 1, cb_qualityOptions)
                
                cb_qualityOptions.setCurrentIndex(self.settings.value('VideoQuality/%s' % plug_name).toInt()[0])
        
        #update the bottom radiobuttons when a combobox changes its state
        for i in range(self.ui.tree_quality.topLevelItemCount()):
            item = self.ui.tree_quality.topLevelItem(i)
            cb = self.ui.tree_quality.itemWidget(item, 1)
            QtCore.QObject.connect(cb, QtCore.SIGNAL("currentIndexChanged(int)"), self.updateGlobalQuality)
            
        self.updateGlobalQuality(0)
            
        # Update the window to display the current configuration
        self.ui.ledit_player.setText(self.settings.value('General/mediaPlayer').toString())
        self.last_valid_player=self.settings.value('General/mediaPlayer').toString()
        
        self.ui.edit_folder.setText(self.settings.value('DownloadOpt/folder').toString())
        
        self.ui.combo_options.setCurrentIndex(self.isChecked(self.settings.value('DownloadOpt/afterDl').toInt()))
        self.ui.combo_dc.setCurrentIndex(self.isChecked(self.settings.value('DownloadOpt/dc_download').toInt()))
        
        self.ui.s_notifyDone.setChecked(self.isChecked(self.settings.value('Notification/notifyDone').toInt()))
        self.ui.s_notifyError.setChecked(self.isChecked(self.settings.value('Notification/notifyError').toInt()))
        
        if notification.HAS_PYNOTIFY:
            self.ui.f_note.hide()

        
        # Connect signals with slots
        QtCore.QObject.connect(self.ui.btn_player, QtCore.SIGNAL("clicked()"), self.browseForFile)
        QtCore.QObject.connect(self.ui.button_folder, QtCore.SIGNAL("clicked()"), self.browseFolder)
        QtCore.QObject.connect(self.ui.buttonBox, QtCore.SIGNAL("accepted()"), self.saveSettings)  
        QtCore.QObject.connect(self.ui.buttonBox, QtCore.SIGNAL("rejected()"), self.close)
        QtCore.QObject.connect(self.ui.rb_all_high_quality, QtCore.SIGNAL("clicked()"), self.setHighQualityToAll)
        QtCore.QObject.connect(self.ui.rb_all_low_quality, QtCore.SIGNAL("clicked()"), self.setLowQualityToAll)
    
    def updateGlobalQuality(self, index):
        if self.update_global_quality:
            global_quality = self.getQualityFromAll()
            if global_quality is None:
                self.ui.rb_mixed_quality.setCheckable(True)
                self.ui.rb_mixed_quality.setChecked(True)
            elif global_quality == "low":
                self.ui.rb_all_low_quality.setChecked(True)
            else:
                self.ui.rb_all_high_quality.setChecked(True)
        
    def setLowQualityToAll(self):
        #Since we're going to change the index of many comboboxes,
        #to prevent them to send a signal everytime, we use this 
        #variable to control the updateGLobalQuality
        self.update_global_quality = False 
        self.setQualityToAll(0)
        self.ui.rb_mixed_quality.setCheckable(False)
        self.update_global_quality = True
        
    def setHighQualityToAll(self):
        #Since we're going to change the index of many comboboxes,
        #to prevent them to send a signal everytime, we use this 
        #variable to control the updateGLobalQuality
        self.update_global_quality = False 
        self.setQualityToAll(1)
        self.ui.rb_mixed_quality.setCheckable(False)
        self.update_global_quality = True
        

    def setQualityToAll(self, value=0):
        for i in range(self.ui.tree_quality.topLevelItemCount()):
            item = self.ui.tree_quality.topLevelItem(i)
            cb = self.ui.tree_quality.itemWidget(item, 1)
            cb.setCurrentIndex(value)
            
    def getQualityFromAll(self):
        high_quality_count = 0
        low_quality_count = 0
        for i in range(self.ui.tree_quality.topLevelItemCount()):
            item = self.ui.tree_quality.topLevelItem(i)
            cb = self.ui.tree_quality.itemWidget(item, 1)
            if cb.currentIndex() == 0:
                low_quality_count += 1
            else:
                high_quality_count += 1
        
        i += 1
        
        if i == low_quality_count:
            return "low"
        elif i == high_quality_count:
            return "high"
        else:
            return None
            

    
    
    def saveSettings(self):
        """Saves the settings."""
        
        #if self.ui.s_rememberUrl.checkState() == 2:
            #self.urlStorage.sync()
        
        if self.isValidPlayer():        
            self.settings.setValue('General/mediaPlayer', QtCore.QVariant(str(self.ui.ledit_player.text())))
        else:
            gui_utils.warning(self, self.tr("Media Player"), self.tr("Given media player is not valid"))
            return 0
        
        self.settings.setValue('DownloadOpt/folder',QtCore.QVariant(self.ui.edit_folder.text()))
        self.settings.setValue('DownloadOpt/afterDl', QtCore.QVariant(self.ui.combo_options.currentIndex()))
        self.settings.setValue('DownloadOpt/dc_downloads', QtCore.QVariant(self.ui.combo_dc.currentIndex()))

        for i in range(self.ui.tree_quality.topLevelItemCount()):
            item = self.ui.tree_quality.topLevelItem(i)
            cb = self.ui.tree_quality.itemWidget(item, 1)
            self.settings.setValue('VideoQuality/%s' % item.text(0), QtCore.QVariant(cb.currentIndex()))

        self.settings.setValue('Notification/notifyDone', QtCore.QVariant(int(self.ui.s_notifyDone.checkState())))
        self.settings.setValue('Notification/notifyError', QtCore.QVariant(int(self.ui.s_notifyError.checkState())))
        self.settings.sync()
        
        #self.parent.settings.sync()
        self.parent.settings = self.settings
        #self.parent.adjustInputs(updatedSettings=True)
        #self.parent.loadConfig()
        self.close()

    def check_save_url(self):
        if self.ui.save_url.isChecked():
            self.ui.save_url_disk.setEnabled(True)
        else:
            if self.ui.save_url_disk.isChecked():
                self.ui.save_url_disk.setChecked(False)
            self.ui.save_url_disk.setDisabled(True)
            
    def browseForFile(self):
        filepath = gui_utils.selectFile(self)
        if filepath: 
            filepath = str(filepath).decode("UTF-8")
            self.ui.ledit_player.setText(os.path.split(filepath)[1])

    def browseFolder(self):
        """Shows a folder selection dialog and (if the user doesn't cancel it) 
        sets the new destination in the input."""
        folder = gui_utils.selectFolder(self, self.ui.edit_folder.text())
        if folder: self.ui.edit_folder.setText(folder)
    
    def isValidPlayer(self):
        return is_command(str(self.ui.ledit_player.text()))
    

    def isChecked(self, value):
        """Converts a integer (accepted values are: 0, 1, 2) to a CheckState."""
        try:
            value = value[0]
        except TypeError:
            pass
            
        return QtCore.Qt.CheckState(value)
        
    @classmethod
    def loadSettings(cls):
        """Returns a QSettings instance and, if needed, populates it with the default settings."""
        
        settings = QtCore.QSettings(QtCore.QSettings.IniFormat, QtCore.QSettings.UserScope, 'WatchVideo', 'watchvideo')
        
        if not settings.contains('General/version') or settings.value('General/version') != c.VERSION:
            cls.defaultSettings(settings)
        
        return settings
    
    @classmethod
    def setDefaultSettings(cls):
        settings = QtCore.QSettings(QtCore.QSettings.IniFormat, QtCore.QSettings.UserScope, 'WatchVideo', 'watchvideo')
        
        cls.defaultSettings(settings)

    @classmethod
    def defaultSettings(cls, settings):
        """Overwrittes a QSettings instance with all default settings."""
        
        from os.path import expanduser
        
        settings.clear()    # Delete everything from the file
        
        settings.beginGroup('General')
        settings.setValue('version', QtCore.QVariant(c.VERSION))
        settings.setValue('mediaPlayer', QtCore.QVariant(main.get_default_player()))
        settings.endGroup()

        settings.beginGroup('DownloadOpt')
        settings.setValue('folder', QtCore.QVariant(expanduser('~')))
        settings.setValue('afterDl', QtCore.QVariant(0))
        settings.setValue('dc_downloads', QtCore.QVariant(0))
        settings.endGroup()
        
        settings.beginGroup('VideoQuality')
        plugins = Matcher().plugins.values()
        for plugin in plugins:
            if plugin.website_name is not None and type(plugin.website_name) == 'str':
                name = plugin.website_name
            else:
                name = plugin.__name__
                
            if plugin.has_high_quality():
                settings.setValue('%s' % name, QtCore.QVariant(0))
        settings.endGroup()

        settings.beginGroup('Notification')
        settings.setValue('notifyDone', QtCore.QVariant(2))
        settings.setValue('notifyError', QtCore.QVariant(0))
        settings.endGroup()
        
