# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Finding plugins and matching URLs to them.
"""


from getmediumurl.compat import OrderedDict
from getmediumurl.plugin import Plugin
import getmediumurl.plugins
from getmediumurl.urlreaders import DEFAULT_URLREADER


class Matcher(object):

    """Store plugins and match URLs to them."""

    def __init__(self, plugins=None, urlreader=DEFAULT_URLREADER):
        """Make a new matcher.

        Plugins from `plugins` are added using the `add_plugins`
        method.

        The `urlreader` callable is used to make
        `getmediumurl.reader.URLReader` instances used in plugins.
        """
        self.plugins = OrderedDict()
        """An `OrderedDict` of usable plugin classes."""
        self.disabled = OrderedDict()
        """An `OrderedDict` of non-usable plugins.

        Keys are plugin class names, values are lists of strings
        describing why the plugins cannot be used.
        """
        self.add_plugins(plugins)
        self.urlreader = urlreader
        """`getmediumurl.reader.URLReader` subclass used by plugins."""

    def add_plugins(self, plugins):
        """Add more plugins to the matcher.

        :Parameters:
          plugins
            An iterable of classes to be used as plugins, modules
            containing them or strings naming them, or a single such
            object.

            If it is `None`, then all modules from the
            `getmediumurl.plugins` package will be used instead.

            Only classes deriving from `getmediumurl.plugin.Plugin`
            whose `getmediumurl.plugin.Plugin.disabled` method yields
            no element will be used.
        """
        if plugins is None:
            plugins = ("getmediumurl.plugins.%s" % module
                       for module in getmediumurl.plugins.__all__)
        # Replace single plugin by a tuple of it.
        elif isinstance(plugins, basestring):
            plugins = (plugins,)
        elif not hasattr(plugins, "__iter__"):
            plugins = (plugins,)
        for plugin in plugins:
            if self._add_plugin(plugin):
                continue
            if isinstance(plugin, basestring):
                submodules = plugin.split(".")
                try:
                    module = __import__(plugin)
                except ImportError:
                    continue
                for attr in submodules[1:]:
                    module = getattr(module, attr)
            else:
                module = plugin
            if hasattr(module, "__all__"):
                attributes = module.__all__
            else:
                attributes = dir(module)
            for attr in attributes:
                klass = getattr(module, attr)
                self._add_plugin(klass)

    def _add_plugin(self, plugin):
        """Add a class `plugin` to `self.plugins` if valid.

        Returns `False` if `plugin` is not a class.
        """
        try:
            if not issubclass(plugin, Plugin):
                return True
        except TypeError:
            return False
        reasons = list(plugin.disabled())
        if reasons:
            self.disabled[plugin.get_plugin_name()] = reasons
        else:
            self.plugins[plugin.get_plugin_name()] = plugin

    def __repr__(self):
        """Return string representation of the plugin matcher."""
        plugins = self.plugins.values()
        return "<%s %s>" % (self.__class__.__name__, repr(plugins))

    def match(self, url):
        """Match an URL to a known plugin.

        Returns a `getmediumurl.plugin.Plugin` object or `None` if no
        plugin matched this URL.

        """
        iterable = self.plugins.itervalues()
        for plugin in iterable:
            obj = plugin.match(url, self.urlreader)
            if obj is not None:
                return obj

    def get_plugin(self, name, mediumid):
        """Get plugin of specific class `name` and `mediumid`.

        Returns a `getmediumurl.plugin.Plugin` object or `None` if no
        known plugin has this name.

        """
        plugin = self.plugins.get(name, None)
        if plugin is not None:
            return plugin(mediumid, self.urlreader)
