/*

    Copyright (C) 2002,2003  John Darrington 

    This program is free software; you can redistibute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
const static char RCSID[]="$Id: wday.c,v 1.31 2003/06/03 05:23:59 john Exp $";

#include "config.h"

#include <stdio.h>
#include <locale.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include "event_list.h"
#include <string.h>


#if HAVE_GETOPT_H
# include <getopt.h>
#endif
#include <unistd.h>

#include "zone.h"
#include "common.h"
#include "event.h"
#include "holiday.h"
#include "symbol.h"


void descend_directory(char *datadir, char * prefix);


extern const char VERSION_STRING[];
extern const char DATADIR_STRING[];
extern const char LOCALEDIR_STRING[];

extern FILE *yyin;
int yyparse();

eventlist elist;


#if HAVE_GETOPT_LONG
#  define getopt_func(argc,argv,optstring,lopts,lindex)  \
        getopt_long(argc, argv,optstring,lopts,lindex)

#else
#  define getopt_func(argc,argv,optstring,lopts,lindex)  \
        getopt(argc, argv,optstring)

#endif

#if HAVE_GETOPT_LONG
const static struct option opts[] = {
  {"help",   no_argument,NULL,'h'},
  {"version",no_argument,NULL,'v'},
  {"quiet",  no_argument,NULL,'q'},
  {"list",   no_argument,NULL,'l'},
  {"week-end",   no_argument,NULL,'w'},
  {"no-week-end",   no_argument,NULL,'W'},
  {"zone",required_argument,NULL,'z'},
  {"list-zones",optional_argument,NULL,'L'},
  {"format",required_argument,NULL,'f'},
  {"data-dir",required_argument,NULL,'d'},
  {0,0,0,0}
};
#endif
const static char *desc[] = { 
  "Displays this help message",
  "Show the version number",
  "Don't print anything",
  "List all the holidays",
  "Show week ends",
  "Don't show weekends",
  "Use <zone> as the current zone",
  "List all configured zones",
  "Display dates in this format",
  "Use <data-dir> as the data directory",
  0
};


  char *zonefile=0;

void nofile(char *z);


int
main(int argc, char **argv)
{

  struct dt the_date;
  char *zone=0;
  int list=0;
  int listZones=0;
  char *zoneListStart =0;
  int quiet=0;
  int status = 0;
  int the_year=-1;
  int weekend = -1;
  const char default_date_format[]="%a %B %d %Y";
  char *date_format = (char *) default_date_format;
  char *datadir = 0;
  int zone_query = 0;

  int c;

  setlocale (LC_MESSAGES, "");
  setlocale (LC_TIME, "");
  bindtextdomain (PACKAGE, LOCALEDIR_STRING);
  textdomain (PACKAGE);

  /* Parse command line options */

  while (-1 != ( c = getopt_func(argc, argv,"hvqz:lf:wWd:L::",opts,0) ) ) {
    switch (c) { 
    case 'd':
      datadir = (char *) calloc(strlen(optarg)+2,sizeof(char));
      if ( !datadir) { 
	perror("Cannot calloc");
	exit(1);
      }
      strcpy(datadir,optarg);
      strcat(datadir,"/");
      break;
    case 'w':
      weekend = 1;
      break;
    case 'W':
      weekend = 0;
      break;
    case 'f':
      date_format = optarg;
      break;
    case 'l':
      list=1;
      break;
    case 'L':
      if ( optarg ) { 
      zoneListStart = (char *) calloc(strlen(optarg)+1,sizeof(char));
      if ( !zoneListStart) { 
	perror("Cannot calloc");
	exit(1);
      }
      strcpy(zoneListStart,optarg);
      }
      listZones=1;
      break;
    case 'h':
      printf("wday [OPTIONS] [[YYYY]MMDD]\n");
      printf("Valid options are:\n");
      {
	char **s = (char **) desc;
	struct option *o= (struct option *) opts;
	for( o = (struct option *) opts ; o->name ; ++o ) { 
	  printf("-%c",o->val);
	  if (o->has_arg)
	    printf(" <%s>", o->name);
#if HAVE_GETOPT_LONG 
	  printf("  \t| ");
	  printf("--%s",o->name);
	  if ( o->has_arg)
	    printf("=<%s>",o->name);
#endif
	  printf("\t%s\n",*s++);
	}
      }
      exit(0);
      break;
    case 'v':
      printf(_("%s Version %s\n"),PACKAGE,VERSION_STRING);
      exit(0);
      break;
    case 'z':
      zone=optarg;
      break;
    case 'q':
      quiet = 1;
      break;
    default: {
#if HAVE_GETOPT_LONG	
      const char hopt[]="--help";
#else
      const char hopt[]="-h";
#endif
      fprintf(stderr,_("Try wday %s for more information\n"),hopt);
      exit(1);
      break; }
    }
  }


  if ( 0 == datadir ) {
   datadir = (char *) calloc(strlen(DATADIR_STRING)+strlen("/wdinfo/")+1,sizeof(char));
   if ( !datadir) { 
	perror("Cannot calloc");
	exit(1);
   }
   strcpy(datadir,DATADIR_STRING);
   strcat(datadir,"/wdinfo/");
  }


  if ( listZones ) {
    char *startDir = datadir;
    if ( zoneListStart)  {
      startDir = malloc(strlen(datadir)+strlen(zoneListStart)+2);
      if ( !startDir ) { 
	perror("Cannot malloc");
      }
      strcpy(startDir,datadir);
      strcat(startDir,zoneListStart);
      strcat(startDir,"/");
    }
    descend_directory(startDir,zoneListStart);
    exit(0);
  }


  /* Get the date to be examined */
  if ( argv[optind] ) { 
    if ( !list ) {
      the_date = date_from_string(argv[optind]);

      if ( ! date_is_valid(the_date) ) 
      {
	fprintf(stderr,_("Invalid date format: %s\n"), argv[optind]);
	exit(1);
      }
    }
    else { 
      the_year = year_from_string(argv[optind]);
    }
  }
  else {
    /* If not specified, then use today's date */
    the_date = date_now();
    the_year = year_now();
  }


  /* Do we want to worry about weekends ? */
  /* Only if -list is not given ( and not explicitly set ) */
  if ( -1 == weekend ) { 
    if ( list) 
	weekend = 0 ;
    else 
      weekend = 1 ;
  }
  

  if(zone && (strcmp(zone,"?")==0) )
      zone_query = 1;

  if ( !zone || (strcmp(zone,"?")==0) ) { 
    /* If zone was not specified on the command line, try to find out it out */

    zone = getzone();

    if ( !zone) { 
      fprintf(stderr,_("Cannot determine zone\n"));
      exit (1);
    }

  }


  if ( zone_query ) 
    printf("Using zone: %s\n",zone);


  zonefile = (char*) calloc(
			    strlen(datadir)+strlen(zone)+1,
			    sizeof (char)
			    );
  if ( !zonefile) { 
    perror("Cannot calloc");
    exit(1);
  }
  strcpy(zonefile,datadir);
  if ( datadir ) {
    free(datadir);
    datadir = 0;
  }
  strcat(zonefile,zone);
  
  {
    struct stat sb;
    if ( -1 == stat(zonefile,&sb) ) { 
      int the_err = errno;
      if ( the_err == ENOENT ) {
	nofile(zone);
      }
      else { 
	char *err = strerror(the_err);
	fprintf(stderr,_("Cannot stat %s: %s\n"),zonefile,err);
	exit(1);
      }
    }
    if ( S_ISDIR(sb.st_mode)) { 
      fprintf(stderr,_("%s is a directory\n"),zonefile);
      exit(1);
    }
  }


  yyin = fopen(zonefile,"r");
  if ( !yyin ) { 
    char *err = strerror(errno);
    fprintf(stderr,_("Cannot open %s: %s\n"),zonefile,err);
    exit(1);

  }

  eventl_init(&elist);

  sym_init();

  if ( 0 != yyparse()) 
	return 1;

  free(zonefile);
  zonefile=0;
  fclose(yyin);

  /* check that the list is not empty */
  if ( ! *elist.table) { 
    nofile(zone);
  }

  if ( ! list ) {
    const char *holiday_title=isholiday(&the_date, elist.table,weekend) ;
    if ( ! quiet ) print_date(stdout,&the_date,date_format);
    if ( holiday_title ) {
      if ( ! quiet ) 
       printf(_(" is %s\n"),holiday_title);
      else 
       status = 0;
    }
    else {
      if ( ! quiet )
       printf(_(" is not a public holiday\n"));
      else 
	status = 1;
    }
  }
  else { 
    struct dt a_date;

    a_date.dt_mday = 1;
    a_date.dt_mon = 0;
    a_date.dt_year = the_year;
    
    while( a_date.dt_year == the_year) {
      const char *holiday_title = isholiday(&a_date, elist.table,weekend) ;
      if ( holiday_title ) { 
	print_date(stdout,&a_date,date_format);
	printf(" \t%s\n",holiday_title);
      }
      a_date = next(&a_date);
    }
    
  }

  eventl_delete(&elist); 

  return status;

};


void 
nofile(char *zone)
{
    fprintf(stderr,_("Sorry, holidays for zone %s have not yet been entered. If you know the rules for this zone, please create a definition file and submit it to the maintainer. See wdinfo(5) for the format.\n"),zone);
    exit(1);
}



/* Recursively descend the directory dirname, 
   printing out the names of all the files
   which are not zero length.
*/
void
descend_directory(char *dirname,char *prefix)
{
  DIR *dir; 
  struct dirent *entry ;

  static char *topdirname = 0 ;

  if ( ! topdirname ) 
    topdirname = dirname;

  dir = opendir(dirname);
  if ( !dir ) { 
    /* This dir is probably non existant 
       So no zones in this region are configured
    */
    return;
  }

  while ( (entry = readdir(dir)) ) { 
    struct stat buf;
    char *filename;

    if ( 0 == strcmp(".",entry->d_name)) continue;
    if ( 0 == strcmp("..",entry->d_name)) continue;

    if ( strstr(entry->d_name,",D") ) continue;

    filename = (char *) malloc(strlen(entry->d_name)+strlen(dirname)+1+1);
    if ( ! filename ) { 
      perror("Cannot malloc:");
      exit(1);
    }

    strcpy(filename,dirname);
    strcat(filename,entry->d_name);
	

    if ( -1 == stat(filename,&buf) ) { 
      int err = errno;
      fprintf(stderr,"Cannot stat %s: %s\n",filename, strerror(err));
      exit(1);
    }
    if ( S_ISREG(buf.st_mode) ) {
      if ( buf.st_size > 0 ) {
	char *zonefile ;
	/* strip off the directory name */
	zonefile = filename + strlen(topdirname) ; 
	if ( prefix)
	  printf("%s/",prefix);
	printf("%s\n",zonefile);
      }
    }
    else if ( S_ISDIR(buf.st_mode) ) {
	strcat(filename,"/");
	descend_directory(filename,prefix);
    }
    else {
	fprintf(stderr,"Error: %s is not a regular file\n",filename);
    }

    free(filename);
  }
    

  closedir(dir);

}




