<?php

namespace JsonConfig;

use FormatJson;
use MWException;
use MWHttpRequest;
use stdClass;

/**
 * Various useful utility functions (all static)
 */
class JCUtils {

	/**
	 * Uses wfLogWarning() to report an error. All complex arguments are escaped with FormatJson::encode()
	 * @param string $msg
	 * @param mixed|array $vals
	 * @param bool|array $query
	 */
	public static function warn( $msg, $vals, $query = false ) {
		if ( !is_array( $vals ) ) {
			$vals = array( $vals );
		}
		if ( $query ) {
			foreach ( $query as $k => &$v ) {
				if ( stripos( $k, 'password' ) !== false ) {
					$v = '***';
				}
			}
			$vals['query'] = $query;
		}
		$isFirst = true;
		foreach ( $vals as $k => &$v ) {
			if ( $isFirst ) {
				$isFirst = false;
				$msg .= ': ';
			} else {
				$msg .= ', ';
			}
			if ( is_string( $k ) ) {
				$msg .= $k . '=';
			}
			if ( is_string( $v ) || is_int( $v ) ) {
				$msg .= $v;
			} else {
				$msg .= FormatJson::encode( $v );
			}
		}
		wfLogWarning( $msg );
	}

	/** Init HTTP request object to make requests to the API, and login
	 * @param string $url
	 * @param string $username
	 * @param string $password
	 * @throws \MWException
	 * @return \CurlHttpRequest|\PhpHttpRequest|false
	 */
	public static function initApiRequestObj( $url, $username, $password ) {
		$apiUri = wfAppendQuery( $url, array( 'format' => 'json' ) );
		$options = array(
			'timeout' => 3,
			'connectTimeout' => 'default',
			'method' => 'POST',
		);
		$req = MWHttpRequest::factory( $apiUri, $options );

		if ( $username && $password ) {
			$query = array(
				'action' => 'login',
				'lgname' => $username,
				'lgpassword' => $password,
			);
			$res = self::callApi( $req, $query, 'login' );
			if ( $res !== false ) {
				if ( isset( $res['login']['token'] ) ) {
					$query['lgtoken'] = $res['login']['token'];
					$res = self::callApi( $req, $query, 'login with token' );
				}
			}
			if ( $res === false ) {
				$req = false;
			} elseif ( !isset( $res['login']['result'] ) ||
			     $res['login']['result'] !== 'Success'
			) {
				self::warn( 'Failed to login', array(
						'url' => $url,
						'user' => $username,
						'result' => isset( $res['login']['result'] ) ? $res['login']['result'] : '???'
					) );
				$req = false;
			}
		}
		return $req;
	}

	/**
	 * Make an API call on a given request object and warn in case of failures
	 * @param \CurlHttpRequest|\PhpHttpRequest $req logged-in session
	 * @param array $query api call parameters
	 * @param string $debugMsg extra message for debug logs in case of failure
	 * @return array|false api result or false on error
	 */
	public static function callApi( $req, $query, $debugMsg ) {
		$req->setData( $query );
		$status = $req->execute();
		if ( !$status->isGood() ) {
			self::warn( 'API call failed to ' . $debugMsg, array( 'status' => $status->getWikiText() ),
				$query );
			return false;
		}
		$res = FormatJson::decode( $req->getContent(), true );
		if ( isset( $res['warnings'] ) ) {
			self::warn( 'API call had warnings trying to ' . $debugMsg,
				array( 'warnings' => $res['warnings'] ), $query );
		}
		if ( isset( $res['error'] ) ) {
			self::warn( 'API call failed trying to ' . $debugMsg, array( 'error' => $res['error'] ), $query );
			return false;
		}
		return $res;
	}

	/**
	 * Helper function to check if the given value is an array,
	 * and all keys are integers (non-associative array)
	 * @param array $value array to check
	 * @return bool
	 */
	public static function isList( $value ) {
		return is_array( $value ) &&
		       count( array_filter( array_keys( $value ), 'is_int' ) ) === count( $value );
	}

	/**
	 * Helper function to check if the given value is an array,
	 * and all keys are strings (associative array)
	 * @param array $value array to check
	 * @return bool
	 */
	public static function isDictionary( $value ) {
		return is_array( $value ) &&
		       count( array_filter( array_keys( $value ), 'is_string' ) ) === count( $value );
	}

	/**
	 * Helper function to check if the given value is an array and if each value in it is a string
	 * @param array $array array to check
	 * @return bool
	 */
	public static function allValuesAreStrings( $array ) {
		return is_array( $array ) && count( array_filter( $array, 'is_string' ) ) === count( $array );
	}

	/**
	 * Converts an array representing path to a field into a string in 'a/b/c[0]/d' format
	 * @param array $fieldPath
	 * @throws \MWException
	 * @return string
	 */
	public static function fieldPathToString( array $fieldPath ) {
		$res = '';
		foreach ( $fieldPath as $fld ) {
			if ( is_int( $fld ) ) {
				$res .= '[' . $fld . ']';
			} elseif ( is_string( $fld ) ) {
				$res .= $res !== '' ? ( '/' . $fld ) : $fld;
			} else {
				throw new MWException( 'Unexpected field type, only strings and integers are allowed' );
			}
		}
		return $res === '' ? '/' : $res;
	}

	/**
	 * Recursively copies values from the data, converting JCValues into the actual values
	 * @param mixed|JCValue $data
	 * @param bool $skipDefaults if true, will clone all items except those marked as default
	 * @return mixed
	 */
	public static function sanitize( $data, $skipDefaults = false ) {
		if ( is_a( $data, '\JsonConfig\JCValue' ) ) {
			$value = $data->getValue();
			if ( $skipDefaults && $data->defaultUsed() ) {
				return is_array( $value ) ? array() : ( is_object( $value ) ? new stdClass() : null );
			}
		} else {
			$value = $data;
		}
		return self::sanitizeRecursive( $value, $skipDefaults );
	}

	private static function sanitizeRecursive( $data, $skipDefaults ) {
		if ( !is_array( $data ) && !is_object( $data ) ) {
			return $data;
		}
		if ( is_array( $data ) ) {
			// do not filter lists - only subelements if they were checked
			foreach ( $data as &$valRef ) {
				if ( is_a( $valRef, '\JsonConfig\JCValue' ) ) {
					/** @var JCValue $valRef */
					$valRef = self::sanitizeRecursive( $valRef->getValue(), $skipDefaults );
				}
			}
			return $data;
		}
		$result = new stdClass();
		foreach ( $data as $fld => $val ) {
			if ( is_a( $val, '\JsonConfig\JCValue' ) ) {
				/** @var JCValue $val */
				if ( $skipDefaults === true && $val->defaultUsed() ) {
					continue;
				}
				$result->$fld = self::sanitizeRecursive( $val->getValue(), $skipDefaults );
			} else {
				$result->$fld = $val;
			}
		}
		return $result;
	}
}
