<?php

namespace Wikibase\Repo\View;

/**
 * Helper for injecting text by substituting placeholders.
 * This class is designed to aid with the technique of putting placeholders into
 * cacheable HTML (in ParserOutput), and later replacing it with non-cacheable HTML
 * snippets (for use by OutputPage).
 *
 * @since 0.5
 * @licence GNU GPL v2+
 *
 * @author Daniel Kinzler
 */
class TextInjector {

	/**
	 * @var string
	 */
	protected $uniqPrefix;

	/**
	 * @var int
	 */
	protected $markerIndex;

	/**
	 * @var array string -> array
	 */
	protected $markers;

	/**
	 * @param array $markers markers generated by another instance of TextInjector,
	 *        for use by inject(); a map of string markers associated with
	 *        parameter arrays.
	 */
	public function __construct( $markers = array() ) {
		$this->markers = $markers;

		// idea stolen from Parser class in core
		$this->uniqPrefix = "\x7fUNIQ" . wfRandom( 16 );
		$this->markerIndex = 0;
	}

	/**
	 * Associates a new marker with the given parameters, and returns it.
	 * All parameters passed to this function will be associated with the marker.
	 *
	 * @param $name
	 * @param ...
	 *
	 * @return string
	 */
	public function newMarker( $name /* ... */ ) {
		$marker = '$' . $this->uniqPrefix . '#' . ++$this->markerIndex . '$';
		$args = func_get_args();

		$this->markers[$marker] = $args;
		return $marker;
	}

	/**
	 * Returns a map of markers associated with lists of arguments.
	 *
	 * @return array[] marker -> array
	 */
	public function getMarkers() {
		return $this->markers;
	}

	/**
	 * Replaces the markers in $html by calling $callback for each marker in $markers,
	 * passing the arguments associated with each marker to $callback.
	 *
	 * @param string $html
	 * @param callable $callback
	 *
	 * @return string
	 */
	public function inject( $html, $callback ) {
		$search = array();
		$replace = array();

		foreach ( $this->markers as $marker => $args ) {
			$subst = call_user_func_array( $callback, $args );

			if ( is_string( $subst ) ) {
				$search[] = $marker;
				$replace[] = $subst;
			}
		}

		$html = str_replace( $search, $replace, $html );
		return $html;
	}
}
