<?php

namespace ZeroBanner;
use BaseTemplate;
use DOMElement;
use FormatJson;
use Html;
use IContextSource;
use JsonConfig\JCSingleton;
use Language;
use MinervaTemplate;
use MobileContext;
use OutputPage;
use RawMessage;
use SpecialPage;
use Title;
use TitleValue;
use WebRequest;
use Xml;

/**
 * Class PageRendering handles all the hook callbacks in a stateful manner
 * @package ZeroBanner
 */
class PageRendering extends \ContextSource {

	private $zerodotParent;
	private $mdotParent;

	/** @var bool */
	private $isZeroSpecial;
	/** @var bool */
	private $isZeroSubdomain;

	/**
	 * @param IContextSource $context
	 */
	public function __construct( IContextSource $context = null ) {
		global $wgZeroBannerClusterDomain;
		$this->zerodotParent = ".zero.wikipedia.$wgZeroBannerClusterDomain/";
		$this->mdotParent = ".m.wikipedia.$wgZeroBannerClusterDomain/";

		if ( $context === null ) {
			$context = \RequestContext::getMain();
		}
		$this->setContext( $context );
		$this->isZeroSpecial =
			$this->getTitle() ? $this->getTitle()->isSpecial( 'ZeroRatedMobileAccess' ) : false;

		// Allow URL override for testing purposes
		$request = $this->getRequest();
		$sd = $request->getVal( 'X-SUBDOMAIN' );
		if ( $sd === null ) {
			$sd = $request->getHeader( 'X-SUBDOMAIN' );
		}
		if ( $sd ) {
			$this->isZeroSubdomain = strcasecmp( $sd, 'ZERO' ) === 0;
		} else {
			// Either running from root redirector or from shell
			$this->isZeroSubdomain = false;
			$urlParts = wfParseUrl( $request->detectServer() );
			if ( isset( $urlParts['host'] ) ) {
				$urlParts = explode( '.', $urlParts['host'], 3 );
				if ( count( $urlParts ) >= 2 ) {
					$this->isZeroSubdomain =
						strcasecmp( $urlParts[0], 'ZERO' ) === 0 || strcasecmp( $urlParts[1], 'ZERO' ) === 0;
				};
			}
		}
	}

	/**
	 * @param string $id
	 * @param null|ZeroConfig $config
	 * @param bool $enabled
	 * @return string
	 */
	public static function getJsConfigBlock( $id, $config, $enabled ) {
		$cfg = array(
			'enabled' => $enabled,
			'id' => $id,
		);
		if ( $config ) {
			$cfg['showImages'] = $config->showImages();
			$cfg['whitelistedLangs'] = $config->whitelistedLangs();
			$cfg['enableHttps'] = $config->enableHttps();
			$cfg['sites'] = $config->sites(); // todo: remove this value once it's out of cache
			$cfg['domains'] = $config->domains();
			$cfg['proxy'] = $config->proxy();
			$cfg['ipset'] = $config->ipset();
			$cfg['bannerUrl'] = $config->bannerUrl();
			$cfg['bannerWarning'] = $config->bannerWarning();
			if ( $config->testInfoScreen() ) {
				$cfg['testInfoScreen'] = true;
			}
		}
		return 'window.zeroGlobalConfig=' . Xml::encodeJsVar( $cfg ) . ';';
	}

	// TODO: Clean up HTML concatenation. Review for any non-escaped user input.

	/**
	 * GetMobileUrl hook handler - replaces *.m.* with *.zero.* if the current site is zero.
	 *
	 * @param $subdomainTokenReplacement string
	 * @return bool
	 */
	public function onGetMobileUrl( &$subdomainTokenReplacement ) {
		if ( $this->isZeroSubdomain ) {
			$subdomainTokenReplacement = 'zero.';
		}
		return true;
	}

	/**
	 * Rewrite DOM to replace content's external and image links with links to the special redirect page.
	 * @param \MobileFormatter $formatter
	 * @return bool
	 */
	public function onMobileFrontendBeforeDOM( \MobileFormatter $formatter ) {

		if ( !$this->isZeroSite() ) {
			return true;
		}
		$unified = $this->isUnifiedZero();
		$config = $unified ? null : $this->getZeroConfig();
		if ( !$unified && !$config ) {
			return true;
		}
		if ( $this->isZeroSpecial ) {
			return true;
		}

		wfProfileIn( __METHOD__ );
		$isFilePage = $this->getTitle()->inNamespace( NS_FILE );
		$isZerodotNonFilePage = !$isFilePage && $this->isZeroSubdomain;
		$isLowQualityImg =
			!$isFilePage && !$this->isZeroSubdomain && ( $unified || $config->shrinkImg() );
		$doc = $formatter->getDoc();

		// ZERO site - replace all images with links to images
		if ( $isZerodotNonFilePage || $isLowQualityImg ) {

			// Watchlist thumbnails are not lazy loaded,
			// yet we don't set a disableImages cookie on zerodot,
			// nor would we, so we just deal with them if needed
			if ( $this->getTitle()->isSpecial( 'Watchlist' ) ) {
				/* @var DOMElement $div */
				foreach ( $doc->getElementsByTagName( 'div' ) as $div ) {
					$divClass = $div->getAttribute( 'class' );
					$divStyle = $div->getAttribute( 'style' );
					$overwriteAttrs = $divClass && preg_match('/^listThumb /', $divClass ) &&
						( !$divStyle || preg_match( '/^background-image: url/', $divStyle ));
					if ( $overwriteAttrs ) {
						$div->setAttribute( 'class', 'listThumb icon icon-max-x' );
						$div->setAttribute( 'style', '' );
					}
				}
			}

			$replacements = array();
			foreach ( $doc->getElementsByTagName( 'img' ) as $tag ) {
				/* @var DOMElement $tag */
				if ( !$tag ) {
					continue;
				}
				if ( $isZerodotNonFilePage ) {
					$alt = $tag->getAttribute( 'alt' );
					$spanNodeText =
						wfMessage( 'zero-click-to-view-image', lcfirst( substr( $alt, 0, 40 ) ) )->text();
					$replTag = $doc->createElement( "span", str_replace( "&", "&amp;", $spanNodeText ) );
					if ( $alt ) {
						$replTag->setAttribute( 'title', $alt );
					}
					$replacements[] = array( $tag, $replTag );
				} else {
					self::reduceImgQuality( $tag, 'src' );
					self::reduceImgQuality( $tag, 'srcset' );
				}
			}
			if ( $replacements ) {
				foreach ( $replacements as $element ) {
					$element[0]->parentNode->replaceChild( $element[1], $element[0] );
				}
			}
		}

		if ( $isZerodotNonFilePage ) {
			// MW normalizes all links to the main namespace, ignoring all alternative ones
			$ns = $this->getLanguage()->convertNamespace( NS_FILE );
			$filePrefix = '#^/wiki/' . $ns . ':.#';
		} else {
			$filePrefix = '';
		}
		/* @var $link DOMElement */
		foreach ( $doc->getElementsByTagName( 'a' ) as $link ) {
			$href = $link->getAttribute( 'href' );
			if ( !$href ) {
				continue; // Missing HREF
			}
			$ch0 = substr( $href, 0, 1 );
			if ( $ch0 === '#' ) {
				continue; // Anchor is never checked
			}
			$ch1 = substr( $href, 1, 1 );
			$class = $link->getAttribute( 'class' );
			// Any link to image (class=image), or any non-local link, or, in a zerodot
			// article, any direct File: link will be redirected.
			// Local links start with either '?...' or '/...', but not '//...'
			// Local links to File: are indicative of a media type.
			if ( $class !== 'image' && ( $ch0 === '?' || ( $ch0 === '/' && $ch1 !== '/' ) ) &&
			     !( $isZerodotNonFilePage && preg_match( $filePrefix, $href ) )
			) {
				continue;
			}
			$link->setAttribute( 'href', $this->makeRedirect( $href ) );
		}

		// This cookie instructs JavaScript in interstitial.js
		$cVal = !$unified && $config->enableHttps() ? 'tls' : '';
		$cExp = $cVal === '' ? '1' : null;
		$this->getRequest()
			->response()
			->setcookie( 'ZeroOpts', $cVal, $cExp,
				array( 'httpOnly' => false, 'path' => '/', 'prefix' => '' ) );
		wfProfileOut( __METHOD__ );
		return true;
	}

	/**
	 * Invoked by MinervaTemplate to add banners
	 * @param MinervaTemplate $template
	 * @return bool
	 */
	public function onMinervaPreRender( MinervaTemplate $template ) {
		// If the page is Special:ZeroRatedMobileAccess just wipe out the menu "hamburger".
		if ( $this->isZeroSpecial ) {
			$template->set( 'menuButton', '' );
		}

		// TODO: alias $template->data where appropriate
		$bannersSupported = array_key_exists( 'banners', $template->data );

		$banner = $this->createBanner();
		if ( $banner ) {
			if ( $bannersSupported ) {
				$template->set( 'banners', array( $banner ) );
			}
			if ( $bannersSupported ) {
				$this->rewriteLangLinks( $template );
			}
			// Add warnings to footer links
			$this->addWarning( $template, 'mobile-switcher' );
			$this->addWarning( $template, 'mobile-license' );
			$this->addWarning( $template, 'privacy' );
			$this->addWarning( $template, 'terms-use' );
		}
		return true;
	}

	/**
	 * BeforePageDisplayMobile hook handler
	 *
	 * @param $out OutputPage
	 * @return bool
	 */
	public function onBeforePageDisplay( &$out ) {
		if ( !$this->isZeroSite() ) {
			return true;
		}

		wfProfileIn( __METHOD__ );

		$isZeroTraffic = $this->isUnifiedZero() || $this->getZeroConfig();

		$this->logWarnings( $isZeroTraffic );

		if ( $isZeroTraffic || $this->isZeroSubdomain ) {
			$out->addModuleStyles( 'zerobanner.styles' );
			$out->addModules( 'zerobanner' );
		}

		$out->addVaryHeader( 'X-CS' );
		$out->addVaryHeader( 'X-Subdomain' );

		if ( !( $this->isUnifiedZero() ) && $this->getConfigId() !== null ) {
			$out->addVaryHeader( 'X-Forwarded-By' );
			$out->addVaryHeader( 'X-Forwarded-Proto' );
		}

		// If zerodot isn't supported here and the user isn't already on
		// Special:ZeroRatedMobileAccess, send the user to Special:ZeroRatedMobileAccess
		// and include the original URL so that the user may continue to the m. server.
		if ( $this->isZeroSubdomain() && !$isZeroTraffic && !$this->isZeroSpecial ) {
			$info = $this->getWikiInfo();
			// As we're on zerodot and it failed, let's explicitly send user to http://
			$url = $this->getStartPageUrl( $info[0], self::GET_LANDING | self::FORCE_HTTP );
			$url = wfAppendQuery( $url, array( 'from' => $this->getRequest()->getFullRequestURL() ) );
			$out->redirect( $url );
			$out->output();
		}

		wfProfileOut( __METHOD__ );
		return true;
	}

	/**
	 * MakeGlobalVariablesScript hook handler
	 *
	 * We use this to instruct the RL JavaScript,
	 * largely MobileFrontend's JavaScript, that
	 * images really are not wanted on zerodot.
	 * This is useful in places like the search
	 * facility, which shows thumbnails otherwise.
	 *
	 * As more JavaScript is built making use of
	 * images, it should use mw.config.get( 'wgImagesDisabled' )
	 * before pulling down images.
	 *
	 * @param array $vars the list of RL JS variables
	 * @param OutputPage $out the current OutputPage
	 * @return bool
	 */
	public function onMakeGlobalVariablesScript(
		/** @noinspection PhpUnusedParameterInspection */ &$vars, OutputPage $out ) {
		if ( $this->isZeroSubdomain ) {
			// We cannot modify $vars directly,
			// as it will get merged and trumped.
			// So we trump it here with addJsConfigVars.
			$out->addJsConfigVars( array( 'wgImagesDisabled' => true ) );
		}
		return true;
	}

	/**
	 * @return string with banner html
	 */
	private function createBanner() {
		$title = $this->getTitle();
		if ( !$title ) {
			return '';
		}
		$isFilePage = $title->inNamespace( NS_FILE );
		if ( $this->isUnifiedZero() ) {

			$query = array( 'zcmd' => 'js-banner' );
			if ( $isFilePage ) {
				$query['file'] = '1';
			}
			$urlJsBanner = self::getSpecial()->getLocalURL( $query );

			$script = Html::linkedScript( $urlJsBanner );
			$noScript = self::renderBannerImgTag( $isFilePage );
//			$noScript =
//				Html::rawElement( 'a', array( 'href' => self::getSpecial()->getLinkURL( 'info=1' ) ),
//					$noScript );
			$noScript = Html::rawElement( 'noscript', array(), $noScript );
			return $script . $noScript;
		}
		$config = $this->getZeroConfig();
		if ( !$config ) {
			// @fixme logic: In case of no X-CS, should we ignore all redirect attempts?
			return '';
		}

		if ( $this->isZeroSpecial ) {
			$redir = $this->getRedirectInfo();
			if ( array_key_exists( 'warn', $redir ) || array_key_exists( 'redirect', $redir ) ) {
				return '';
			}
		}
		$cfg = self::getJsConfigBlock( $this->getConfigId(), $config, true );
		return self::renderBanner( $this, $config, null, null, $isFilePage, $this->isZeroSpecial ) .
			Html::rawElement( 'script', array(), $cfg );
	}

	/**
	 * Render <img> tag for the Zero image banner
	 * @param bool $isFilePage
	 * @param string|null $langCode
	 * @return string
	 */
	public static function renderBannerImgTag( $isFilePage, $langCode = null ) {
		$query = array( 'zcmd' => 'img-banner' );
		if ( $isFilePage ) {
			$query['zfile'] = '1';
		}
		if ( $langCode ) {
			$query['zlang'] = $langCode;
		}
		$url = self::getSpecial()->getLocalURL( $query );
		return Html::rawElement( 'img', array( 'src' => $url ) );
	}

	/**
	 * @param PageRendering|null $self If non-null, decorates banner link as redirect
	 * @param ZeroConfig $config
	 * @param string|null $langCode If null, use site's default.
	 * @param string $sitename
	 * @param bool $isFilePage
	 * @param bool $undismissible
	 * @return string
	 */
	public static function renderBanner(
		$self, ZeroConfig $config, $langCode, $sitename = null, $isFilePage = false, $undismissible = false
	) {
		$banner = self::getBannerText( $config, $isFilePage, $langCode, $sitename );
		if ( !$banner ) {
			return '';
		}
		$background = $config->background();
		$foreground = $config->foreground();
		$bannerUrl = $config->bannerUrl();
		$showInfo = $config->testInfoScreen();
		if ( !$showInfo && $bannerUrl ) {
			if ( !$config->isDisabled() && $config->bannerWarning() && $self !== null ) {
				$bannerUrl = $self->makeRedirect( $bannerUrl );
			}
			$attr = array( 'href' => $bannerUrl, 'style' => "color:$foreground;" );
			$banner = Html::rawElement( 'a', $attr, $banner );
		}

		$attr = array(
			'class' => 'mw-mf-message',
			'id' => 'zero-rated-banner-text',
		);
		$style = $config->fontSize() !== '' ? 'font-size:' . $config->fontSize() . ';' : '';
		if ( !$bannerUrl ) {
			$style .= "color:$foreground;";
		}
		if ( $style !== '' ) {
			$attr['style'] = $style;
		}
		$banner = Html::rawElement( 'span', $attr, $banner );
		if ( !$undismissible ) {
			$dismissTitle = wfMessage( 'zero-dismiss-notification' )->escaped();
			$dismiss = Html::rawElement(
				'button',
				array( 'class' => 'notify-close',
					'style' => "background:$background;",
					'title' => $dismissTitle ),
				Html::rawElement(
					'span',
					array(
						'class' => 'notify-close-x',
						'style' => "background:$background;border-color:$foreground;color:$foreground;" ),
					'&#120;'
				)
			);
		} else {
			$dismiss = '';
		}
		$banner = Html::rawElement(
			'div',
			array(
				'id' => 'zero-rated-banner',
				'class' => 'mw-mf-banner',
				'style' => "background:$background;color:$foreground;" ),
			$dismiss . $banner );
		return $banner;
	}

	/**
	 * If a particular language could cause a charge, send user to an interstitial.
	 *
	 * @param BaseTemplate $template
	 * @return bool
	 */
	private function rewriteLangLinks( BaseTemplate $template ) {

		if ( isset( $template->data['language_urls'] ) ) {
			$languageUrls = $template->data['language_urls'];
			if ( is_array( $languageUrls ) && count( $languageUrls ) > 0 ) {
				foreach ( $languageUrls as &$lang ) {
					$link = $lang['href'];
					if ( $this->isZeroSubdomain ) {
						$link = str_replace( $this->zerodotParent, $this->mdotParent, $link );
					}
					$lang['href'] = $this->makeRedirect( $link );
				}
				$template->set( 'language_urls', $languageUrls );
			}
		}
		return true;
	}

	/**
	 * @param ZeroConfig $config
	 * @param bool $isFilePage will this banner show on File: page?
	 * @param string $langCode
	 * @param string $sitename
	 * @throws \MWException
	 * @return bool|String
	 */
	public static function getBannerText( $config, $isFilePage = false, $langCode = null, $sitename = null ) {
		wfProfileIn( __METHOD__ );

		if ( !$config || ( $isFilePage && !$config->showImages() ) ) {
			return false;
		}
		if ( $langCode === null ) {
			global $wgLang;
			$lang = $wgLang;
		} else {
			$lang = Language::factory( $langCode );
		}
		$name = self::pickLocalizedString( $config->name(), $lang );
		if ( $name === false ) {
			wfProfileOut( __METHOD__ );
			return false;
		}
		$nameMsg = new RawMessage( $name );

		$banners = $config->banner();
		if ( array_key_exists( $lang->getCode(), $banners ) ) {
			$linkText = $banners[$lang->getCode()];
		} else {
			$linkText = wfMessage( 'zero-banner-text' )->inLanguage( $lang )->plain();
		}
		if ( $sitename !== null ) {
			// Rendering for the zero-config page, fake {{SITENAME}}
			$linkText = str_replace( '{{SITENAME}}', $sitename, $linkText );
		}
		$linkTextMsg = new RawMessage( $linkText );
		$res = $linkTextMsg->inLanguage( $lang )->rawParams( $nameMsg->inLanguage( $lang )->escaped() )->escaped();

		wfProfileOut( __METHOD__ );
		return $res;
	}

	/**
	 * Return true if the request is from a zero network, but not to a Zero-specific.
	 * Varnish sends us X-CS == "ON" instead of the code
	 * @return true
	 */
	public function isUnifiedZero() {
		return $this->getConfigId() === "ON";
	}

	/**
	 * Return Carrier ID (X-CS) value or null if no value was set
	 * @return null|string
	 */
	public function getConfigId() {
		if ( $this->configId === false ) {
			// Allow URL override of the X-CS parameter for testing purposes
			$id = $this->getRequest()->getVal( 'X-CS' );
			if ( $id === null ) {
				$id = $this->getRequest()->getHeader( 'X-CS' );
			}
			if ( $id === '(null)' || !$id ) {
				$id = null;
			}
			$this->configId = $id;
		}
		return $this->configId;
	}
	private $configId = false;

	/**
	 * Returns cached carrier configuration based on X-CS header or query parameter
	 * @param bool $allowDisabled if true, a valid but disabled config object would still be returned
	 * @return ZeroConfig|null Carrier configuration or null if it doesn't exist or is disabled
	 */
	public function getZeroConfig( $allowDisabled = false ) {
		// Cached configuration or null if missing.
		if ( $this->config === false || ( $this->config && $this->config->isAllowDisabled() !== $allowDisabled ) ) {
			wfProfileIn( __METHOD__ );

			$xcs = $this->getConfigId();
			// Unified is treated as if there is no config - we don't know the actual ID
			if ( $xcs !== null && !$this->isUnifiedZero() ) {
				@list( $id, $ipset ) = explode( '|', $xcs, 2 );
				if ( $ipset === null ) {
					$ipset = '';
				}
			} else {
				$id = null;
				$ipset = '';
			}

			if ( $id !== null && $this->config === false ) {
				$this->config = JCSingleton::getContent( new TitleValue( NS_ZERO, $id ) );
			} else {
				$this->config = null;
			}

			if ( $this->config ) {
				list( $lang, $subdomain, $project ) = $this->getWikiInfo();
				$this->config->setContext(
					$ipset,
					$this->getRequest()->getHeader( 'X-FORWARDED-BY' ),
					$this->getRequest()->getProtocol() === 'https',
					$subdomain . '.' . $project, $lang, $this->getRequest()->getCheck( 'X-CS' ) );
			}

			wfProfileOut( __METHOD__ );
		}
		return !$this->config || $this->config->isDisabled() ? null : $this->config;
	}
	/** @var bool|null|ZeroConfig */
	private $config = false;

	/**
	 * @return array with three values - language code (en), subdomain (m), site (wikipedia), cluster domain (org)
	 */
	public function getWikiInfo() {
		static $info = null;
		if ( $info === null ) {
			global $wgConf, $wgDBname, $wgZeroBannerClusterDomain, $wgZeroSiteOverride;
			if ( $wgZeroSiteOverride ) {
				list( $site, $langCode ) = $wgZeroSiteOverride;
			} else {
				list( $site, $langCode ) = $wgConf->siteFromDB( $wgDBname );
			}
			$subdomain = $this->isZeroSubdomain ? 'zero' : 'm';
			$info = array( $langCode, $subdomain, $site, $wgZeroBannerClusterDomain );
		}
		return $info;
	}

	/**
	 * Returns true if X-SUBDOMAIN header is set to 'ZERO'
	 * @return bool
	 */
	public function isZeroSubdomain() {
		return $this->isZeroSubdomain;
	}

	/**
	 * Determine if this could be a Zero-related site shown with a mobile device
	 * Must be wikipedia, with mobile view enabled
	 * @return bool
	 */
	private function isZeroSite() {
		static $isZero = null;
		if ( $isZero === null ) {
			$info = $this->getWikiInfo();
			// @FIXME: this check should only see if this is a mobileview, the rest is a check in partner's config
			$isZero = $info[2] === 'wikipedia' && MobileContext::singleton()->shouldDisplayMobileView();
		}
		return $isZero;
	}

	/**
	 * For the special zero page when used as a redirector,
	 * determine if redirection can be done automatically or a warning should be displayed.
	 * Returns empty array if no redirection,
	 * array( 'redirect' => $url, 'code' => '302' ) to auto-redirect
	 * array( 'softredirect' => $url ) in case there is no valid config (X-CS)
	 * array( 'warn' => 'external' or 'file', 'from' => $fromUrl, 'to' => $toUrl ) to show warning
	 */
	public function getRedirectInfo() {
		if ( $this->redirectInfo === null ) {
			$this->redirectInfo = $this->makeRedirectInfo();
		}
		return $this->redirectInfo;
	}
	private $redirectInfo = null;

	/**
	 * See getRedirectInfo() for description
	 * @return array
	 */
	private function makeRedirectInfo() {
		$request = $this->getRequest();
		if ( $request->getCheck( 'isroot' ) ) {
			$url = $this->getLandingRedirect();
			return array( 'redirect' => $url, 'code' => '302' );
		}
		$config = $this->getZeroConfig();
		$from = $request->getVal( 'from' );
		$toUrl = $request->getVal( 'to' );
		if ( $toUrl === null || $from === null ) {
			// This is not an external link or file redirect, see if we need to redirect anywhere
			$langCode = null;
			$flags = 0;
			if ( $config ) {
				// Known carrier, if we skip landing page, redirect to the default (first) language
				if ( !$config->showZeroPage() ) {
					$showLangs = $config->showLangs();
					$langCode = $showLangs[0];
				}
			} elseif ( !$this->isZeroSubdomain ) {
				// Unidentified carrier, M subdomain redirect to main page
				// TODO: discuss if we ever want to send 'm.wikipedia.org/wiki/Special:ZRMA' -> 'zero.'
				// TODO: in case provider exists and they have only whitelisted zero.
				$info = $this->getWikiInfo();
				$langCode = $info[0];
			} else {
				// Unidentified carrier, ZERO subdomain
				$config = $this->getZeroConfig( true );
				if ( $config ) {
					// The traffic is from a carrier, but this specific access is not zero-rated
					// Try to redirect to the free traffic if possible
					$info = $this->getWikiInfo();
					if ( $config->enabled() && in_array( 'm.' . $info[2], $config->sites() ) ) {
						$langCode = $info[0];
						$flags |= self::FORCE_MDOT;
					}
				}
				// else - do not redirect if this is "*.zero.wikipedia.org" - need to show red warning
			}

			if ( $langCode !== null ) {
				$url = $this->getStartPageUrl( $langCode, $flags );
				return array( 'redirect' => $url, 'code' => '302' );
			}
			return array();
		}
		$fromTitle = Title::newFromText( $from );
		if ( !$fromTitle ) {
			$this->logDebug( $request, 'invalidParams' );
			return array();
		}
		if ( !$config ) {
			$this->logDebug( $request, '!config' );
			return array( 'softredirect' => $toUrl );
		}
		$fromUrl = MobileContext::singleton()->getMobileUrl( $fromTitle->getFullURL() );
		$redir = false; // true if silent redirect, false for the warning
		$urlBits = wfParseUrl( $toUrl );
		$toHost = is_array( $urlBits ) && array_key_exists( 'host', $urlBits ) ? $urlBits['host'] : false;
		$toScheme = is_array( $urlBits ) && array_key_exists( 'scheme', $urlBits ) ? $urlBits['scheme'] : false;
		if ( $toHost ) {
			$toHost = strtolower( $toHost );
			// fixme: need to check $host against local wikipedia host name
			// fixme: this is a very rare case - a full URL to the local host
			// if ( strcasecmp( $toHost, getCurrentWikiHostName ) === 0 ) {
			// 	$toHost = false;
			// }
		}
		if ( $toHost ) {
			// Check http->https switch and match (optional-language.)(subdomain.site).org
			if ( $toScheme !== 'https' || $config->enableHttps() ) {
				global $wgZeroBannerClusterDomain;
				if ( $toHost === 'upload.wikimedia.' . $wgZeroBannerClusterDomain && $config->showImages() ) {
					$redir = true;
				} elseif ( preg_match(
					'/^([^.]+\.)?([^.]+\.[^.]+)\.' . preg_quote( $wgZeroBannerClusterDomain ) . '$/',
					$toHost,  $matches
				) ) {
					// Another language in wikipedia
					$lang = $matches[1];
					$site = $matches[2];
					// see if the site is whitelisted, and if it is, make sure the language is.
					if ( in_array( $site, $config->sites() ) ) {
						if ( $lang ) {
							$freeLangs = $config->whitelistedLangs();
							$redir = count( $freeLangs ) == 0 || in_array( rtrim( $lang, '.' ), $freeLangs );
						} else {
							$redir = true; // there is no language, but the site is whitelisted, so don't warn
						}
					}
				}
			}
			// else - external link, always warn
		} else {
			// there was no host - must be a local link to an image
			$redir = $config->showImages();
		}
		if ( $redir ) {
			return array( 'redirect' => $toUrl, 'code' => '302' );
		}
		return array(
			'warn' => ( $toHost ? 'external' : 'file' ),
			'from' => $fromUrl, 'to' => $toUrl );
	}

	/** always use m. even on zero. sites */
	const FORCE_MDOT = 1;
	/** URL should point to "Special:Zero" instead of "Main Page" */
	const GET_LANDING = 2;
	/** will set URL scheme to "http://" */
	const FORCE_HTTP = 4;

	/**
	 * Format URL to the zero or m main page for the specific language (or current if null)
	 * @param null $langCode
	 * @param int $flags zero or more of the above flags
	 * @return string
	 */
	public function getStartPageUrl( $langCode = null, $flags = 0 ) {
		$info = $this->getWikiInfo();
		if ( $langCode !== null ) {
			$info[0] = $langCode;
		}
		if ( $flags & self::FORCE_MDOT ) {
			$info[1] = 'm';
		}
		$page = ( $flags & self::GET_LANDING ) ? self::getSpecial( false )->getPrefixedDBkey(): 'Main_Page';
		$url = sprintf( '//%s/wiki/%s', implode( '.', $info ), $page );
		if ( $flags & self::FORCE_HTTP ) {
			$url = 'http:' . $url;
		}
		return $url;
	}

	/**
	 * Find a message in a dictionary for the given language,
	 * or use language fallbacks if message is not defined.
	 * @param array $map Dictionary of languageCode => string
	 * @param Language $lang language object
	 * @return string|bool message from the dictionary or false if nothing found
	 */
	private static function pickLocalizedString( $map, $lang ) {
		$langCode = $lang->getCode();
		if ( array_key_exists( $langCode, $map ) ) {
			return $map[$langCode];
		}
		$fallbacks = $lang->getFallbackLanguages();
		if ( count( $fallbacks ) === 0 ) {
			$fallbacks = array( 'en' );
		}
		foreach ( $fallbacks as $l ) {
			if ( array_key_exists( $l, $map ) ) {
				return $map[$l];
			}
		}
		return false;
	}

	/**
	 * Create redirect URL to be inserted by DOM processing instead of all external links
	 * and other Zero-related redirects.
	 * @param string $url to redirect to in case the user accepts (or if its free for the user)
	 * @return String
	 */
	public function makeRedirect( $url ) {
		if ( $this->query === false ) {
			$this->query = 'from=' . $this->getTitle()->getPrefixedURL() . '&to=';
		}
		return self::getSpecial()->getLinkURL( $this->query . wfUrlencode( $url ) );
	}
	private $query = false;

	/**
	 * Adds parameters to URLs. Helper for onMinervaPreRender( BaseTemplate &$template )
	 * @param BaseTemplate $template
	 * @param string $templateValName The name of the template
	 */
	private function addWarning( BaseTemplate $template, $templateValName ) {
		$link = $template->get( $templateValName );
		$link = $this->createWarningLink( $link );
		$template->set( $templateValName, $link );
	}

	/**
	 * Given a link, returns an interstitial-ized version of it as appropriate
	 * @param $link
	 * @return string The link that will route the user through the redirector
	 */
	public function createWarningLink( $link ) {
		$self = $this;
		return preg_replace_callback( '/href=([\'"])(.*?)\1/',
			function ( $matches ) use ( $self ) {
				$quoteChar = $matches[1];
				$href = $matches[2];
				$ch0 = substr( $href, 0, 1 );
				$ch1 = substr( $href, 1, 1 );
				// Local links start with either '#...', '?...' or '/...', but not '//...'
				if ( $ch0 !== '#' && $ch0 !== '?' && ( $ch0 !== '/' || $ch1 === '/' ) ) {
					return 'href=' .
					$quoteChar . $self->makeRedirect( htmlspecialchars_decode( $href ) ) . $quoteChar;
				} else {
					return $matches[0];
				}
			},
			$link
		);
	}

	/**
	 * Output debug info into Zero log
	 * @param WebRequest $request
	 * @param string|string[] $dbg Debug message
	 */
	private function logDebug( $request, $dbg ) {
		static $printedHeaders = false;

		if ( is_array( $dbg ) ) {
			array_walk( $dbg,
				function ( & $v, $k ) { $v = is_string( $k ) ? $k . '=' . FormatJson::encode( $v ) : $v; } );
		} else {
			$dbg = (array)$dbg;
			$dbg[] = $request->getIP();
		}
		$host = implode( '.', $this->getWikiInfo() );
		$dbg[] = ( $request->getHeader( 'X-FORWARDED-PROTO' ) ?: 'http' ) . '://' . $host;
		$dbg[] = urldecode( $request->getRawQueryString() );
		$dbg = implode( "\t", $dbg );

		if ( !$printedHeaders ) {
			$headers = $request->getAllHeaders();
			ksort( $headers );
			$dbg .= "\nHeaders: " . print_r( $headers, true );
			$printedHeaders = true;
		}
		wfDebugLog( 'zero', $dbg );
	}

	/**
	 * Returned cached instance of the ZRMA special page
	 * @param bool $isLocalized - should the namespace "Special:" be localized
	 * @return Title Zero rated special page
	 */
	private static function getSpecial( $isLocalized = true ) {
		static $local = null, $global = null;
		if ( $isLocalized ) {
			if ( $local === null ) {
				$local = SpecialPage::getTitleFor( 'ZeroRatedMobileAccess' );
			}
			return $local;
		} else {
			if ( $global === null ) {
				$global = Title::makeTitle( NS_SPECIAL, 'ZeroRatedMobileAccess' );
			}
			return $global;
		}
	}

	/**
	 * Create redirect URL for m.* and zero.* root pages
	 * ATTENTION: This method is used from mediawiki-config/mobilelanding.php
	 * ATTENTION: This method may remove forceHTTPS cookie and logoff users
	 * @return string
	 */
	public function getLandingRedirect() {
		$protocol = PROTO_CURRENT;
		$config = $this->getZeroConfig();
		$request = $this->getRequest();
		$flags = 0;
		if ( !$config ) {
			// Staying consistent with the current behaviour - redirect to en.*.*
			$langCode = 'en';
			// zero.* -> Landing, m.* -> Main Page
			if ( $this->isZeroSubdomain ) {
				$flags |= self::GET_LANDING;
			}
		} else {
			$showLangs = $config->showLangs();
			$langCode = $showLangs[0];
			if ( $config->showZeroPage() ) {
				$flags |= self::GET_LANDING;
			}
			// For zero.wikipedia.org, remove forceHTTPS cookie and user state
			if ( $this->isZeroSubdomain && !$config->enableHttps() ) {
				$protocol = PROTO_HTTP;
				// This code should be kept in sync with \MediaWiki::main()
				$deleteCookies =
					$request->getCookie( 'forceHTTPS', '' ) ||
					// check for prefixed version for currently logged in users
					$request->getCookie( 'forceHTTPS' );
				if ( !$deleteCookies ) {
					$user = $this->getUser();
					// Avoid checking the user and groups unless it's enabled.
					$deleteCookies = $user->isLoggedIn() && $user->requiresHTTPS();
				}
				if ( $deleteCookies ) {
					global $wgZeroBannerClusterDomain;
					$time = time() - 86400;
					$resp = $request->response();
					$resp->setcookie( 'forceHTTPS', '', $time,
						array( 'domain' => '.wikipedia.' . $wgZeroBannerClusterDomain ) );
					$resp->setcookie( 'forceHTTPS', '', $time,
						array( 'prefix' => '', 'domain' => '.wikipedia.' . $wgZeroBannerClusterDomain ) );
				}
			}
		}
		$url = $this->getStartPageUrl( $langCode, $flags );
		return wfExpandUrl( $url, $protocol );
	}

	/**
	 * Log any suspicious requests
	 * @param $isZeroTraffic
	 */
	private function logWarnings( $isZeroTraffic ) {
		$req = $this->getRequest();
		$ua = $req->getHeader( 'USER-AGENT' );
		if ( $req->getHeader( 'FROM' ) === 'googlebot(at)googlebot.com' ||
		     strpos( $ua, 'Googlebot' ) !== false || strpos( $ua, 'bingbot' ) !== false ||
		     strpos( $ua, 'YandexBot' ) !== false || strpos( $ua, 'msnbot' ) !== false
		) {
			return;
		}
		$xcs = $req->getHeader( 'X-CS2' ) ?: $req->getHeader( 'X-CS' );
		$warn = array();
		foreach ( array(
			          'acceptbilling',
			          'renderwarning',
			          'renderZeroRatedBanner',
			          'renderZeroRatedRedirect'
		          ) as $check ) {
			if ( $req->getCheck( $check ) ) {
				$warn[] = $check;
			}
		}
		if ( $warn ) {
			$warn = array( 'obsoleteFlags' => implode( '|', $warn ) );
		}

//			if ( $this->isZeroSubdomain ) {
//				$referer = $request->getHeader( 'REFERER' );
//				if ( strpos( $referer, 'zero.wikipedia.org' ) !== false ) {
//					$warn .= ' && zero referer';
//				}
//			}

		if ( $this->isZeroSpecial && $req->getVal( 'zcmd' ) && !$xcs ) {
			$warn['special'] = 'unknown-xcs';
			$warn['xff'] = $req->getHeader( 'X-FORWARDED-FOR' );
		}

		// For Zero traffic, either it should come from Opera AND have slot, or should be neither.
		// For non-Zero traffic, it should not have opera slot
		$forwardedBy = $req->getHeader( 'X-FORWARDED-BY' ) ?: $req->getHeader( 'X-FORWARDED-BY2' );
		// route could be 0, which is the same as not having it.
		$slot = $req->getHeader( 'X-OPERAMINI-ROUTE' );
		if ( $forwardedBy === 'Opera' ) {
			if ( $slot ) {
				if ( !$xcs ) {
					// Opera has a slot for it, but we don't know who this is
					$warn['opera'] = 'slot-no-partner';
					$warn['slot'] = $slot;
					$warn['xff'] = $req->getHeader( 'X-FORWARDED-FOR' );
				} elseif ( !$isZeroTraffic ) {
					// Opera has a slot for it, and the partner is known, but we didn't zero-rate this request
					$warn['opera'] = 'slot-incomplete';
					$warn['slot'] = $slot;
					$warn['xcs'] = $xcs;
				}
			} elseif ( $isZeroTraffic ) {
				// We think it has been zero-rated, but opera has not marked it with a slot
				$warn['opera'] = 'no-slot';
				$warn['xcs'] = $xcs;
			}
		} elseif ( $slot ) {
			// Opera has marked it with a slot, but we didn't recognize it as opera traffic
			$warn['opera'] = 'slot-no-opera';
			$warn['slot'] = $slot;
			$warn['xff'] = $req->getHeader( 'X-FORWARDED-FOR' );
		}

		if ( $warn ) {
			$this->logDebug( $req, $warn );
		}
	}

	/**
	 * Add 'qlow-' to all image urls in an attribute
	 * @param DOMElement $element
	 * @param string $attrName
	 */
	private static function reduceImgQuality( $element, $attrName ) {
		$attr = $element->getAttribute( $attrName );
		if ( $attr ) {
			//upload.wikimedia.org/wikipedia/commons/thumb/e/e3/filename.jpg/220px-filename.jpg
			// @fixme todo: URLs might be direct image links, not a thumbs
			// @fixme todo: support png, gif, and svg (svg gets png'd, btw)
			$attr2 =
				preg_replace( '%(//upload\.[^ ]*/thumb/[^ ]*\.jpe?g)/(\d+px-[^/ ]+)%i', '$1/qlow-$2', $attr );
			if ( $attr2 && $attr !== $attr2 ) {
				$element->setAttribute( $attrName, $attr2 );
			}
		}
	}
}
