<?php

namespace ZeroBanner;
use JsonConfig\JCObjContent;
use JsonConfig\JCUtils;
use JsonConfig\JCValidators;
use JsonConfig\JCValue;
use stdClass;

/**
 * JSON Zero Config
 *
 * @file
 * @ingroup Extensions
 * @ingroup ZeroBanner
 *
 * @author Yuri Astrakhan <yurik@wikimedia.org>
 */
class ZeroConfig extends JCObjContent {

	/** @var stdClass current configuration section as set by setContext */
	private $config = null;
	/** @var null|bool was the context search performed with $allowDisabled */
	private $isAllowDisabled = null;
	private $ipset = null;
	private $proxy = null;

	/**
	 * Search data for the matching configuration section that matches given request
	 * @param string $ipset
	 * @param false|string $proxy
	 * @param bool $https
	 * @param string $site
	 * @param $lang
	 * @param bool $allowDisabled
	 * @throws \MWException
	 * @return bool
	 */
	public function setContext( $ipset, $proxy, $https, $site, $lang, $allowDisabled = false ) {
		if ( !$this->isValid() ) {
			return false;
		}
		$this->config = null;
		$this->isAllowDisabled = null;
		$all = $this->getDataWithDefaults();
		$this->ipset = $ipset;
		$this->proxy = $proxy; // Keep original casing
		$ipset = $ipset !== '' ? $ipset : 'default';
		$proxy = $proxy !== false ? strtoupper( $proxy ) : 'DIRECT';
		if ( !$all->enabled && !$allowDisabled ) {
			return false;
		}
		foreach ( $all->configs as $cfg ) {
			if ( !in_array( $ipset, $cfg->ipsets ) || ( !$cfg->enabled && !$allowDisabled ) ||
			     !in_array( $proxy, $cfg->proxies ) || ( $https && !$cfg->enableHttps ) ||
			     !in_array( $site, $cfg->sites ) ||
			     ( 0 !== count( $cfg->whitelistedLangs ) && !in_array( $lang, $cfg->whitelistedLangs ) )
			) {
				continue;
			}
			$this->config = $cfg;
			$this->isAllowDisabled = $allowDisabled;
		}
		return false; // nothing found
	}

	public function getConfigIds() {
		return array_keys( $this->getDataWithDefaults()->configs );
	}

	public function selectConfigById( $configId ) {
		$this->config = $this->getDataWithDefaults()->configs[$configId];
		$this->isAllowDisabled = null;
	}

	/**
	 * @return null|bool null - not initialized, true - setContext() was called with $allowDisabled == true
	 */
	public function isAllowDisabled() {
		return $this->isAllowDisabled;
	}

	public function isDisabled() {
		return $this->config === null;
	}

	public function ipset() {
		return $this->ipset;
	}

	public function proxy() {
		return $this->proxy;
	}

	public function enabled() {
		return $this->config->enabled && $this->getDataWithDefaults()->enabled;
	}

	public function name() {
		return $this->getDataWithDefaults()->name;
	}

	public function banner() {
		return $this->getDataWithDefaults()->banner;
	}

	public function bannerUrl() {
		return $this->getDataWithDefaults()->bannerUrl;
	}

	public function showLangs() {
		return $this->getDataWithDefaults()->showLangs;
	}

	public function langNameOverrides() {
		return $this->getDataWithDefaults()->langNameOverrides;
	}

	public function background() {
		return $this->getDataWithDefaults()->background;
	}

	public function foreground() {
		return $this->getDataWithDefaults()->foreground;
	}

	public function fontSize() {
		return $this->getDataWithDefaults()->fontSize;
	}

	public function showImages() {
		return $this->getDataWithDefaults()->showImages;
	}

	public function shrinkImg() {
		return $this->getDataWithDefaults()->shrinkImg;
	}

	public function showZeroPage() {
		return $this->getDataWithDefaults()->showZeroPage;
	}

	public function getIpsets() {
		return $this->getDataWithDefaults()->ipsets;
	}

	public function testInfoScreen() {
		return $this->getDataWithDefaults()->testInfoScreen;
	}

	public function admins() {
		return $this->getDataWithDefaults()->admins;
	}

	public function sites() {
		return $this->config->sites;
	}

	public function domains() {
		$sites = $this->sites();
		if ( $sites ) { // explicit list of domains
			return array_map( function( $v ) {
				global $wgZeroBannerClusterDomain;
				return "$v.$wgZeroBannerClusterDomain";
			}, $sites );
		}
		return array();
	}

	public function whitelistedLangs() {
		return $this->config->whitelistedLangs;
	}

	public function proxies() {
		return $this->config->proxies;
	}

	public function enableHttps() {
		return $this->config->enableHttps;
	}

	public function ipsetNames() {
		return $this->config->ipsets;
	}

	public function bannerWarning() {
		return $this->config->bannerWarning;
	}

	public function disableApps() {
		return $this->config->disableApps;
	}

	public function validateContent() {
		$isBool = JCValidators::isBool();
		$isString = JCValidators::isString();

		// Optional comment
		$this->testOptional( 'comment', '', $isString );
		// Config is enabled
		$this->testOptional( 'enabled', true, $isBool );
		// Map of localized partner names
		$this->test( 'name', self::getLangToStrValidator() );
		// Map of localized banner texts with {{PARTNER}} placeholder
		$this->test( 'banner', self::getLangToStrValidator() );
		// Partner URL, do not link by default
		$this->test( 'bannerUrl', JCValidators::useDefault( '', false ), JCValidators::isUrl() );
		// List of language codes to show on Zero page
		$hasShowLangs = $this->test( 'showLangs', self::getShowLangsValidator() );
		// Orange Congo wanted to be able to override the 'kg' language name to 'Kikongo'
		$this->test( 'langNameOverrides', self::getLangToStrValidator() );
		// Background banner color
		$this->testOptional( 'background', '#E31230', $isString );
		// Foreground banner color
		$this->testOptional( 'foreground', '#551011', $isString );
		// Banner font size override
		$this->testOptional( 'fontSize', '', $isString );
		// Zero rate images.
		// @BUG? does this have the same meaning as legacy "IMAGES_ON"?
		$this->testOptional( 'showImages', true, $isBool );
		// Reduce image quality
		$this->testOptional( 'shrinkImg', false, $isBool );
		// Show the special zero page
		if ( $hasShowLangs ) {
			$this->testOptional( 'showZeroPage', count( $this->getField( 'showLangs' )->getValue() ) > 1,
				$isBool );
		}
		// List of additional partner admins for this entry
		$this->testOptional( 'admins', array(), JCValidators::stringToList(), JCValidators::isList(),
			JCValidators::uniqueSortStrList() );
		$this->testEach( 'admins', self::getAdminsValidator() );

		if ( ! (bool) $this->getField( 'configs' ) ) {
			$newObj = new stdClass();
			$migrate = self::migrate( $newObj );
			// Which sites are whitelisted. Default - both m & zero wiki
			$this->test( 'sites', $migrate );
			// List of language codes to show banner on, or empty list to allow on all languages
			$this->test( 'whitelistedLangs', $migrate );
			// List of proxies supported by the carrier, defaults to none (empty list)
			$this->test( 'proxies', $migrate );
			// If carrier supports zero-rating HTTPS traffic
			$this->test( 'enableHttps', $migrate );
			// If carrier wants to suppress zero messaging in apps
			$this->test( 'disableApps', $migrate );
			// Show "non-zero navigation" warning when clicking the banner
			$this->testOptional( 'bannerWarning', true, $migrate );
		} else {
			$newObj = null;
		}

		// List of IP CIDR blocks for this provider
		$this->test( 'ips',
			function ( JCValue $v ) { return !$v->isMissing(); }, // if value is missing, don't create it
			self::getIpValidator() );

		if ( $this->testOptional( 'ipsets', new stdClass(), JCValidators::isDictionary() ) ) {
			$this->testEach( 'ipsets', self::getIpValidator() );

			$legacy = $this->getField( 'ips' );
			if ( $legacy ) {
				// find first value that does not exist in a form 'defaultNNN' starting from 'default'
				$count = 0;
				$dflt = 'default';
				$ipsets = $this->getField( 'ipsets' );
				while ( $ipsets->fieldExists( $dflt ) ) {
					$count ++;
					$dflt = 'default' . $count;
				}
				$ipsets->setField( $dflt, $legacy );
				$ipsets->defaultUsed( false );
			}

			$this->test( 'ips', JCValidators::deleteField() );
		}

		if ( $this->test( 'configs',
			function ( JCValue $v ) use ( $newObj ) {
				if ( $v->isMissing() ) {
					if ( !$newObj ) {
						throw new \MWException();
					}
					// This is not a default, setting UNCHECKED
					$v->setValue( array( $newObj ), JCValue::UNCHECKED );
				}
				return true;
			},
			JCValidators::isList() )
		) {
			$isValidIpSet = self::isValidIpSet( $this->getField( 'ipsets' ) );

			foreach ( $this->getField( 'configs' )->getValue() as $k => $v ) {
				$ipsetsPath = array( 'configs', $k, 'ipsets' );
				$this->testOptional( $ipsetsPath, array( 'default' ), JCValidators::isList() );
				if ( $this->testEach( $ipsetsPath, $isValidIpSet ) && $this->thorough() ) {
					$val = $this->getField( $ipsetsPath );
					// remove validation status from all names and remove duplicates
					$val->setValue( array_unique( JCUtils::sanitize( $val ) ) );
				}
				$this->test( array( 'configs', $k ), JCValidators::isDictionary() );
				$this->testOptional( array( 'configs', $k, 'comment' ), '', $isString );
				$this->testOptional( array( 'configs', $k, 'enabled' ), true, $isBool );
				$this->test( array( 'configs', $k, 'sites' ), self::getSitesValidator() );
				$this->test( array( 'configs', $k, 'whitelistedLangs' ), self::getWhitelistedLangsValidator() );
				$this->test( array( 'configs', $k, 'proxies' ), self::getProxiesValidator() );
				$this->testOptional( array( 'configs', $k, 'enableHttps' ), false, $isBool );
				$this->testOptional( array( 'configs', $k, 'disableApps' ), false, $isBool );
				$this->testOptional( array( 'configs', $k, 'bannerWarning' ), false, $isBool );
			}
		}

		$this->test( 'ipsets', null ); // Ensure that 'ipsets' is at the end

		// Temporary until we decide on using it
		$this->testOptional( 'testInfoScreen', false, $isBool );
	}

	private static function getLangToStrValidator() {
		return function ( JCValue $jcv, array $path ) {
			if ( $jcv->isMissing() ) {
				$v = array();
			} else {
				$v = $jcv->getValue();
				if ( is_object( $v ) ) {
					$v = (array)$v;
				}
			}
			if ( is_array( $v ) ) {
				if ( ZeroConfig::isListOfLangs( array_keys( $v ) ) && JCUtils::allValuesAreStrings( $v ) ) {
					// Sort array so that the values are sorted alphabetically except 'en' which will go as first
					uksort( $v,
						function ( $a, $b ) {
							if ( $a === $b ) {
								return 0;
							} elseif ( $a === 'en' ) {
								return -1;
							} elseif ( $b === 'en' ) {
								return 1;
							} else {
								return strcasecmp( $a, $b );
							}
						} );
					$jcv->setValue( $v );
					return true;
				}
			}
			$jcv->error( "zero-config-" . end( $path ), $path );
			return false;
		};
	}

	private static function getSitesValidator() {
		return function ( JCValue $jcv, array $path ) {
			$validValues = array(
				'm.wikipedia',
				'zero.wikipedia',
			);
			if ( $jcv->isMissing() ) {
				$v = array( 'm.wikipedia', 'zero.wikipedia' );
			} else{
				$v = $jcv->getValue();
			}
			if ( is_array( $v ) ) {
				$v = array_map( 'strtolower', $v );
				if ( count( array_intersect( $v, $validValues ) ) !== count( $v ) ) {
					$v = false;
				} else {
					$v = array_unique( $v );
					sort( $v );
				}
			}
			if ( JCUtils::isList( $v )
			     && count( $v ) > 0
			     && JCUtils::allValuesAreStrings( $v )
			) {
				$jcv->setValue( $v );
				return true;
			}

			$jcv->error( 'zero-config-sites', $path, "'" . implode( "', '", $validValues ) . "'" )
				->numParams( count( $validValues ) );
			return false;
		};
	}

	private static function getProxiesValidator() {
		return function ( JCValue $jcv, array $path, ZeroConfig $self ) {
			if ( $jcv->isMissing() ) {
				$jcv->setValue( array( 'DIRECT' ) );
			} else {
				$v = & $jcv->getValue();
				if ( JCUtils::isList( $v ) ) {
					if ( !$v ) {
						$v = array( 'DIRECT ' );
					} else {
						$v = array_map( function ( $vv ) { return !$vv ? 'DIRECT' : strtoupper( $vv ); }, $v );
						if ( $self->thorough() ) {
							// Remove duplicates while preserving original order
							$v = array_unique( $v );
							sort( $v );
						}
					}
				} else {
					$jcv->error( 'zero-config-proxies', $path );
					return false;
				}
			}
			return true;
		};
	}

	private static function getShowLangsValidator() {
		return function ( JCValue $jcv, array $path ) {
			$v = & ZeroConfig::toList( $jcv );
			if ( JCUtils::isList( $v ) && ZeroConfig::isListOfLangs( $v ) && count( $v ) > 0 ) {
				// Remove duplicates while preserving original order
				$v = array_unique( $v );
				ksort( $v );
				return true;
			}

			$jcv->error( 'zero-config-show_langs', $path );
			return false;
		};
	}

	/**
	 * @return callable
	 */
	private static function getWhitelistedLangsValidator() {
		return function ( JCValue $jcv, array $path, ZeroConfig $self ) {
			if ( $jcv->isMissing() ) {
				$jcv->setValue( array() );
				return true;
			}
			$v = & ZeroConfig::toList( $jcv );
			$lastInd = count( $path ) - 1;
			$oldLast = $path[$lastInd];
			$path[$lastInd] = 'showLangs';
			$showLangs = $self->getField( $path );
			$showLangs = $showLangs ? $showLangs->getValue() : array();
			$path[$lastInd] = $oldLast;
			if ( JCUtils::isList( $v ) && ZeroConfig::isListOfLangs( $v ) &&
			     ( count( $v ) === 0 || count( array_diff( $showLangs, $v ) ) === 0 )
			) {
				if ( count( $v ) > 0 ) {
					// Make $v in the same order as $showLangs, followed by alphabetical leftovers
					$v = array_unique( $v );
					$leftovers = array_diff( $v, $showLangs );
					sort( $leftovers );
					$v = array_merge( $showLangs, $leftovers );
				} // else empty list is the same as whitelist all languages
				$jcv->setValue( $v );
				return true;
			}

			$jcv->error( 'zero-config-whitelisted_langs', $path );
			return false;
		};
	}

	private static function getAdminsValidator() {
		return function ( JCValue $jcv, array $path, ZeroConfig $self ) {
			if ( !is_string( $jcv->getValue() ) ) {
				$jcv->error( 'zero-config-admins', $path );
				return false;
			}
			if ( $self->thorough() ) {
				$usr = \User::newFromName( $jcv->getValue() );
				if ( $usr === false || $usr->getId() === 0 ) {
					$jcv->error( 'zero-config-admins', $path );
					return false;
				}
				$jcv->setValue( $usr->getName() ); // canonical
			}
			return true;
		};
	}

	private static function getIpValidator() {
		return function ( JCValue $jcv, array $path, ZeroConfig $self ) {
			$val = $jcv->getValue();
			if ( is_string( $val ) ) {
				$val = array( $val );
			}
			if ( is_array( $val ) ) {
				$v2 = new stdClass();
				$v2->default = $val;
				$val = $v2;
			}
			$errKey = false;
			if ( $self->thorough() ) {
				foreach ( $val as $key => &$v ) {
					if ( is_string( $v ) ) {
						$v = array( $v );
					} elseif ( !JCUtils::isList( $v ) || !JCUtils::allValuesAreStrings( $v ) ) {
						$errKey = $key;
						break;
					}
					$ipFlags = FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE;
					$optIPv4 = array( 'options' => array( 'min_range' => 1, 'max_range' => 32 ) );
					$optIPv6 = array( 'options' => array( 'min_range' => 1, 'max_range' => 128 ) );

					foreach ( $v as $cidr ) {
						$parts = explode( '/', $cidr, 3 );
						if ( count( $parts ) > 2 ||
						     false === filter_var( $parts[0], FILTER_VALIDATE_IP, $ipFlags )
						) {
							$errKey = $key;
							break;
						}
						if ( count( $parts ) === 2 ) {
							// CIDR block, second portion must be an integer within range
							// If the first part has ':', treat it as IPv6
							// Make sure there are no spaces in the block size
							$blockSize =
								filter_var( $parts[1], FILTER_VALIDATE_INT,
									strpos( $parts[0], ':' ) !== false ? $optIPv6 : $optIPv4 );
							if ( false === $blockSize || $parts[1] !== strval( $blockSize ) ) {
								$errKey = $key;
								break;
							}
						}
					}
					// Sort in natural order, but force sequential keys to prevent json to treat it as dictionary.
					natsort( $v );
				}
				if ( $errKey ) {
					$path[] = $errKey;
					$jcv->error( 'zero-config-ips', $path );
					return false;
				}
			}
			// @bug @fixme return true/false
			return $val;
		};
	}

	/**
	 * Returns true if each of the array's values is a valid language code
	 * @param array $arr
	 * @return bool
	 */
	static function isListOfLangs( $arr ) {
		return count( $arr ) === count( array_filter( $arr, function ( $v ) {
			return \Language::isValidCode( $v );
		} ) );
	}

	/**
	 * @param JCValue|bool $fld
	 * @return callable
	 */
	private static function isValidIpSet( $fld ) {
		if ( $fld && !$fld->error() ) {
			$ipsets = get_object_vars( $fld->getValue() );
		} else {
			$ipsets = array();
		}
		return function ( JCValue $jcv, array $path ) use ( $ipsets ) {
			$val = $jcv->getValue();
			if ( !is_string( $val ) || !array_key_exists( $val, $ipsets ) ) {
				$jcv->error( 'zero-config-ipsets', $path );
				return false;
			}
			return true;
		};
	}

	static function & toList( JCValue $jcv ) {
		// @fixme: replace this by JCValidators::stringToList
		$v = & $jcv->getValue();
		if ( is_string( $v ) ) {
			$v = array( $v );
		}
		return $v;
	}

	/** Appends an extra validator that will copy value into the new object and delete it in original
	 * @param stdClass $newObj
	 * @return callable
	 */
	private function migrate( $newObj ) {
		return function ( JCValue $jcv, array $path ) use ( $newObj ) {
			if ( !$jcv->isMissing() ) {
				$fld = end( $path );
				$newObj->$fld = $jcv->getValue();
				$jcv->status( JCValue::MISSING );
			}
			return true;
		};
	}
}
