<?php

namespace ZeroPortal;

use Html;
use JsonConfig\JCContent;
use JsonConfig\JCDefaultObjContentView;
use JsonConfig\JCValue;
use Language;
use Title;
use Xml;
use ZeroBanner\PageRendering;
use ZeroBanner\ZeroConfig;

/**
 * Represents the content of a JSON Zero Config article.
 * @file
 * @ingroup Extensions
 * @ingroup ZeroRatedMobileAccess
 *
 * @author Yuri Astrakhan <yurik@wikimedia.org>, based on Ori Livneh <ori@wikimedia.org> schema extension
 */
class ZeroConfigView extends JCDefaultObjContentView {

	/**
	 * Render JCContent object as HTML, or null if the default rendering should be used
	 * @param ZeroConfig|JCContent $content
	 * @return string|null
	 */
	public function valueToHtml( JCContent $content ) {
		$html = parent::valueToHtml( $content );
		if ( $content->isValid() ) {
			$html .= '<br />' . $this->renderBanners( $content );
		}
		return $html;
	}

	/**
	 * Returns default content for this object, or null if JCContent's default should be used
	 * @param string $modelId
	 * @return string|null
	 */
	public function getDefault( $modelId ) {
		return <<<JSON
{
    "comment": "Full Company Name",
    "enabled": false,
    "name": {
        "en": "Company"
    },
    "banner": {
        "en": "Free {{SITENAME}} access by $1"
    },
    "showLangs": [
        "en"
    ],
//
// ********** MOST COMMENTS ARE SET TO DEFAULTS **********
// If you need the same value as shown, just delete the comment.
//
//    "bannerUrl": "http://wikipedia.org",
//    "background": "#E31230",
//    "foreground": "#551011",

	"configs": [{
//    "ipsets": ["default"],
//    "sites": [
//        "zero.wikipedia", "m.wikipedia"
//    ],
//    "whitelistedLangs": [],   <-- ALL by default
//    "proxies": [ "DIRECT" ]
//    "enableHttps": false
//    "bannerWarning": false,
    }],

    "ipsets": {
        "default": [
//           "127.0.0.1/32"
    ]}
}
JSON;
	}

	/**
	 * @param ZeroConfig|JCContent $content
	 * @param JCValue|mixed $data
	 * @param array $path
	 * @return string
	 */
	public function renderValue( JCContent $content, $data, array $path ) {
		if ( $path && is_a( $data, '\JsonConfig\JCValue' ) && !$data->error() ) {
			$val = $data->getValue();
			switch ( $path[0] ) {
				case 'admins':
					if ( count( $path ) === 2 ) {
						$url = Title::newFromText( $val, NS_USER )->getLinkURL();
						return Xml::element( 'a', array( 'href' => $url ), $val );
					}
					break;
				case 'configs':
					if ( count( $path ) === 3 && $path[2] === 'whitelistedLangs' && count( $val ) === 0 ) {
						return wfMessage( 'zeroportal-all-langs' )->escaped();
					}
			}
		}
		return parent::renderValue( $content, $data, $path );
	}

	/**
	 * @param ZeroConfig|JCContent $content
	 * @param JCValue|mixed $data
	 * @param array $path
	 * @return string
	 */
	public function renderRowContent( JCContent $content, $data, array $path ) {
		$html = parent::renderRowContent( $content, $data, $path );
		if ( count( $path ) !== 2 || is_a( $data, '\JsonConfig\JCValue' ) ) {
			return $html;
		}
		switch ( $path[0] ) {
			case 'name':
			case 'langNameOverrides':
				$key = array_pop( $path );
				$jcv = $content->getField( $path );
				if ( !$jcv || $jcv->error() ) {
					break;
				}
				if ( $path[0] === 'langNameOverrides' ) {
					$langName = Language::fetchLanguageName( $key );
					$original =
						$langName === $data ? Xml::span( $langName, 'mw-jsonconfig-same' )
							: htmlspecialchars( $langName );
					$html .= Xml::tags( 'td', array( 'class' => 'mw-zeroportal-info' ), $original );
					break;
				}

				// 'name'
				$container = $jcv->getValue();
				$lang = Language::factory( $key );
				$fallbacks = $lang->getFallbackLanguages();
				if ( count( $fallbacks ) === 0 && $key !== 'en' ) {
					$fallbacks = array( 'en' );
				}
				if ( count( $fallbacks ) > 0 ) {
					$fbElems = array();
					foreach ( $fallbacks as $fb ) {
						$attr = null;
						if ( array_key_exists( $fb, $container ) ) {
							if ( $container[$fb] === $data ) {
								$attr = array( 'class' => 'mw-jsonconfig-same' );
							}
						} else {
							$attr = array( 'class' => 'mw-jsonconfig-unknown' );
						}
						$fbElems[] = Xml::element( 'span', $attr, $fb );
					}
					$html .= Xml::tags( 'td', array( 'class' => 'mw-zeroportal-info' ), '&rarr; ' . implode( ' &rarr; ', $fbElems ) );
				}
				break;
		}
		return $html;
	}

	private function renderBanners( ZeroConfig $config ) {
		global $wgConf, $wgOut;

		wfProfileIn( __METHOD__ );
		$wgConf->loadFullData();

		// Merging keys preserves the order - first all the banners, then - name, etc
		$langs = array_merge( $config->banner(), $config->name(), array_flip( $config->showLangs() ) );
		$configsData = $config->getDataWithDefaults()->configs;
		foreach ( $configsData as $cfg ) {
			$langs =
				array_merge( $langs, array_flip( $cfg->whitelistedLangs ) );
		}

		$configXcs = $wgOut->getTitle()->getRootText();

		$sites = $configsData[0]->sites;
		$bannerSubdomain = !$sites || in_array( 'm.wikipedia', $sites ) ? 'm' : 'zero';

		$rows = array();

		foreach ( $langs as $langCode => $_ ) {
			$th = Xml::elementClean( 'th', null, $langCode );
			// render the banner without hiding it
			// Get localized "wikipedia" sitename. If null, will use default {{SITENAME}}
			$sitename = $wgConf->get( 'wgSitename', str_replace( '-', '_', $langCode ) . 'wiki' );

			$bannerInTheory = PageRendering::renderBanner( null, $config, $langCode, $sitename );
			if ( $bannerInTheory ) {
				$bannerInTheory .= Html::rawElement( 'div', array(), PageRendering::renderBannerImgTag( false, $langCode, $sitename ) );
			} else {
				$bannerInTheory = '???';
			}
			$bannerInTheory = Xml::tags( 'td', null, $bannerInTheory );

			$iframe = Xml::tags( 'iframe', array(
					'src' => "//{$langCode}.{$bannerSubdomain}.wikipedia.org/wiki/MediaWiki:terms-use?X-CS={$configXcs}",
					'frameborder' => '0',
					'sandbox' => '', // Good idea to sandbox for new browsers!
					'scrolling' => 'no',
					'seamless' => 'seamless',
					'class' => 'mw-mf-iframe',
				),
				''
			) . Html::rawElement( 'div', array(), Html::rawElement(
				'img',
				array( 'src' =>
					"//{$langCode}.{$bannerSubdomain}.wikipedia.org/wiki/Special:ZeroRatedMobileAccess?zcmd=img-banner&X-CS={$configXcs}"
			) ) );
			$bannerInPractice = Xml::tags( 'td', null, $iframe );
			$rows[] = Xml::tags( 'tr', null, $th . $bannerInTheory . $bannerInPractice );
		}
		$res = Xml::tags( 'table', array( 'class' => 'mw-zeroportal-banners' ),
			Xml::tags( 'tbody', array(), join( "\n", $rows ) )
		);

		wfProfileOut( __METHOD__ );
		return $res;
	}
}
