{-
Board.hs

Xadrez - A p2p chess written in Haskell
Copyright (C) 2004-2005 Marco Tulio Gontijo e Silva

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write al the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Bosaln, MA  02111-1307  USA
-}
module Board (Board, Place, module Piece, move, standard, pieceIn, inverse)
    where

import Char

import Piece

type Line = [Piece]
type Place = (Int, Int)

newtype Board = MakeBoard [Line]

cross :: (a -> b, c -> d) -> (a, c) -> (b, d)
cross (f, g) (a, b) = (f a, g b)

vertical, whiteSquare, blackSquare :: Char
vertical = '|'
whiteSquare = '#'
blackSquare = ' '

instance Show Board where
    show t
	= "\n" ++ places ++ line ++ show' t
	

places, line :: String
places = "    A   B   C   D   E   F   G   H"
line = "\n  ---------------------------------\n"

show' :: Board -> String
show' (MakeBoard []) = []
show' (MakeBoard ts)
    = [chr (l + ord '0')] ++ " " ++
      (showLine (l `mod` 2 == 0) $ last ts) ++ line ++
      (show' $ MakeBoard $ init ts) 
      where l = length ts

showLine :: Bool -> Line -> String
showLine _ [] = [vertical]
showLine c ls = [vertical] ++ [color] ++ (show $ head ls) ++
		(showLine (not c) $ tail ls)
    where color = if c then whiteSquare else blackSquare

move :: (Place, Place) -> Board -> Board
move (p1@(i1, j1), p2@(i2, j2)) board
    | (isCastling p1 p2) && (j2 > j1)
	= (flip put (i1, 5) . cross (moved, id) . get' (i1, 7) .
	   flip put (i1, 6) . cross (moved, id) . get' (i1, 4)) board
    | isCastling p1 p2
	= (flip put (i1, 3) . cross (moved, id) . get' (i1, i1) .
	   flip put (i1, 2) . cross (moved, id) . get' (i1, 4)) board
    | (isPassant board p1 p2) && (i2 == 5)
	= (flip put p2 . get' p1 . flip put p2 . get' (4, j2)) board
    | (isPassant board p1 p2) && (i2 == 2)
	= (flip put p2 . get' p1 . flip put p2 . get' (3, j2)) board
    | isTwo board p1 p2 = (flip put p2 . cross (passant ,id) . get' p1) board
    | otherwise = (flip put p2 . cross (moved ,id) . get' p1) board

isTwo :: Board -> Place -> Place -> Bool
isTwo board from@(i1, _) (i2, _)
    = (isPawn $ pieceIn board from) && (abs(i2 - i1) == 2)

isPassant :: Board -> Place -> Place -> Bool
isPassant board from@(i1, j1) to@(i2, j2)
    = (isPawn $ pieceIn board from) && (i2 - i1 == 1) && (abs(j2 - j1) == 1) &&
      (isEmpty $ pieceIn board to)

isCastling :: Place -> Place -> Bool
isCastling (i1, j1) (i2, j2) = ((i1 == 0) || (i1 == 7)) && (j1 == 4) &&
			       ((j2 == 2) || (j2 == 6)) && (i2 == i1)

put :: (Piece, Board) -> Place -> Board
put (c, (MakeBoard ts)) (i, j)
    = MakeBoard (before ++ new : after) 
    where before = take i ts
	  (this : after) = drop i ts
	  new = putLine j (c, this)

putLine :: Int -> (Piece, Line) -> Line
putLine j (c, ls) = before ++ c : after
    where before = take j ls
	  (_ : after) = drop j ls

get' :: Place -> Board -> (Piece, Board)
get' (i, j) (MakeBoard ts)
    = (c, MakeBoard (before ++ new : after))
    where before = take i ts
	  (this : after) = drop i ts
	  (c, new) = getLine' j this

getLine' :: Int -> Line -> (Piece, Line)
getLine' i ls = (this, before ++ empty : after)
    where before = take i ls
	  (this : after) = drop i ls

standard :: Bool -> Board
standard c = MakeBoard 
	   [[rook c, knight c, bishop c, queen c, king c, bishop c, knight c,
	     rook c],
	    [pawn c, pawn c, pawn c, pawn c, pawn c, pawn c, pawn c, pawn c],
	    [empty, empty, empty, empty, empty, empty, empty, empty],
	    [empty, empty, empty, empty, empty, empty, empty, empty],
	    [empty, empty, empty, empty, empty, empty, empty, empty],
	    [empty, empty, empty, empty, empty, empty, empty, empty],
	    [pawn d, pawn d, pawn d, pawn d, pawn d, pawn d, pawn d, pawn d],
	    [rook d, knight d, bishop d, queen d, king d, bishop d, knight d,
	     rook d]]
    where d = not c

pieceIn :: Board -> Place -> Piece
pieceIn (MakeBoard b) (i, j)= (b !! i) !! j

inverse :: Board -> Board
inverse (MakeBoard b) = MakeBoard $ reverse b