/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Gameplay_State.cc @see Gameplay_State.hh */

#include <stdexcept>
#include "Gameplay_State.hh"
#include "Game.hh"
#include "User_Input_Manager.hh"

namespace XBobble
{

void
Gameplay_State::set_substate(int arg)
{
	if(arg == substate)
		return;

	switch(arg) {
	case WAITING:
	case LAUNCHING:
	case NONE:
		this->subexit();
		substate = arg;
		this->subentry();
		break;

	default:
		throw Bad_State_Error("Gameplay_State::set_substate()");
		break;

	} // switch(arg)

} // Gameplay_State::set_substate()

void
Gameplay_State::subentry()
{
	switch(substate) {
	case WAITING:
		// Enable these as they are needed
		get_timeout_meter().enable(true);
		get_grid_shaker().enable(true);
		break;

	// We have just launched a ball...
	case LAUNCHING:
		// Gets current ball from chooser with
		// get_current_ball() and gives it to
		// launched_ball_manager with launch_ball(ball), and
		// tells chooser to make the 'next ball' the 'current
		// ball' with chooser.next_to_current(), which then
		// randomly creates a new next ball from the available
		// ball types in grid.ball_traits and starts the
		// animation to move the old next ball into the launch
		// postion with launcher.start_animation().
		get_launcher().launch_ball();
		break;

	case NONE:
		break;

	default:
		// We should never be here!
		throw Bad_State_Error("Gameplay_State::entry() - "
				      "invalid current substate");
	} // switch(substate)

} // Gameplay_State::subentry()

void
Gameplay_State::subexit()
{
	switch(substate) {
	case WAITING:
		// Disable these as they aren't needed
		get_timeout_meter().enable(false);
		get_grid_shaker().enable(false);
		break;

	case LAUNCHING:
		get_launched_ball_manager().enable(false);
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::exit() - "
				      "invalid current substate");
	} // switch(substate)

} // Gameplay_State::subexit()

void
Gameplay_State::entry()
{
	set_substate(WAITING);
	// i.e. entering this compound state
	/// @todo enable all game elements
}

void
Gameplay_State::exit()
{
	set_substate(NONE);
	// i.e. Leaving compound state.
	/// @todo disable all game elements
}

void
Gameplay_State::tock(uint32_t tocks_passed)
{
	User_Input_Manager& ui = game.game_manager.system.user_input_manager;

	get_launcher().tock();
	get_ambient_effects_manager().tock();

	switch(substate) {
	case WAITING:
		get_timeout_meter().tock();
		get_grid_shaker().tock();
		if(get_timeout_meter().timeout() || ui.get_button()) {
			this->set_substate(LAUNCHING);
		}
		break;

	case LAUNCHING:
		get_launched_ball_manager().tock();

		// Here we find which state to switch to.
		if(get_launched_ball_manager().ball_stopped()) {
			get_launched_ball_manager().ball_to_grid();
			/*
			get_launched_ball_manager().ball_to_grid();
			// find_popped_balls() below also updates
			// player_score to reflect number of balls
			// popped
			get_popped_balls_manager().find_popped_balls();
			get_grid().move_down(); // only does it if necessary
			*/

			switch(game.grid.condition()) {
			case Grid::OVERHANG:
				game.set_state(GAME_OVER);
				break;
			case Grid::EMPTY:
				game.set_state(FINISHED_LEVEL);
				break;
			} // switch(grid.condition())

			this->set_substate(WAITING);

		} // if(game.launched_ball_manager.ball_stopped() ... )
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::tock() - "
				      "invalid current substate");
	} // switch(substate)

} // Gameplay_State::tock()

void
Gameplay_State::tick(uint32_t tick_delta)
{
	game.ambient_effects_manager.tick();

	// As we don't need the position of popped balls for
	// processing in the logic stage, we leave it to be updated in
	// the tick func (it will be called before the screen is
	// updated and the positions are used to draw the balls onto
	// the screen.)
	game.popped_balls_manager.tick();

	switch(substate) {
	case WAITING:
		break;

	case LAUNCHING:
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::tick() - "
				      "invalid current substate");
	} // switch(substate)

} // Gameplay_State::tick()

void
Gameplay_State::sync_tick(uint32_t tick_delta)
{
	game.ambient_effects_manager.sync_tick();

	switch(substate) {
	case WAITING:
		break;

	case LAUNCHING:
		// We need to use ball position and state for logic
		// processing (ie tock processing) so update its
		// position here.
		game.launched_ball_manager.sync_tick();
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::sync_tick() - "
				      "invalid current substate");
	} // switch(substate)

} // Gameplay_State::sync_tick()


