/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Grid.cc @see Grid.hh */

#include <cstdlib>
#include "Grid.hh"
#include "Ball.hh"
#include "Ball_Trait.hh"
#include "Game_Manager.hh"
#include "Popped_Balls_Manager.hh"
#include <SDL_opengl.h>
#include "glstuff.hh"
#include <vector>
#include "gameplay_variables.hh"
#include <stack>
#include "util.hh"
#include <fstream>
#include <iterator>
#include <memory>
#include "Sound_Output_Manager.hh"

namespace XBobble
{

void
Grid::make_dlist(uint32_t lod)
{
	dlist = glGenLists(1);
	glNewList(dlist, GL_COMPILE);
	 glMaterialf(GL_FRONT, GL_SHININESS, 100);
	 if(lod != 3) {
		 glBindTexture(GL_TEXTURE_2D, texname);
		 glMaterialf(GL_FRONT, GL_SHININESS, 60);
		 call_glMaterialfv(GL_FRONT, GL_AMBIENT, .0f, .0f, .0f, 1);
		 call_glMaterialfv(GL_FRONT, GL_DIFFUSE, .8f, .8f, .8f, 1);
		 call_glMaterialfv(GL_FRONT, GL_SPECULAR, 1.0f, 1.0f, 1.0f, 1);
	 } else {
		 glMaterialf(GL_FRONT, GL_SHININESS, 20);
		 call_glMaterialfv(GL_FRONT, GL_AMBIENT,
				   0.2f, 0.2f, 0.2f, 0.1f);
		 call_glMaterialfv(GL_FRONT, GL_DIFFUSE,
				   0.1f, 0.2f, 0.6f, 1);
		 call_glMaterialfv(GL_FRONT, GL_SPECULAR,
				   1.6f, 1.6f, 1.6f, 1);
	 }

	 // leave room for the balls!
	 glTranslatef(0, 0, get_r());

	 glBegin(GL_QUADS);
	  glNormal3f(0, 0, -1);
	  // Top left corner
	  if(lod != 3) glTexCoord2f(0.0f, 1.5f);
	  glVertex3f(-get_w()/2, -get_h()/2, 0);
	  // Bottom left corner
	  if(lod != 3) glTexCoord2f(0.0f, 0.0f);
	  glVertex3f(-get_w()/2, get_h()/2, 0);
	  // Bottom right corner
	  if(lod != 3) glTexCoord2f(1.5f, 0.0f);
	  glVertex3f(get_w()/2, get_h()/2, 0);
	  // Top right corner
	  if(lod != 3) glTexCoord2f(1.5f, 1.5f);
	  glVertex3f(get_w()/2, -get_h()/2, 0);
	 glEnd();

	glEndList();
}

namespace
{

void
check_args(std::size_t rows, std::size_t cols, float arg_r)
{
	// When setting touching pointers, row counter
	// increments two at a time.  Allow for overrun of
	// counter (plus a little bit extra.)
	const uint32_t int_max = std::numeric_limits<int32_t>::max()-5;

	if(rows < 2)
		throw std::out_of_range("Grid(rows, cols) - "
					"rows < 2");

	if(cols < 2)
		throw std::out_of_range("Grid(rows, cols) - "
					"cols < 2");

	if(rows > int_max) {
		std::ostringstream os;
		os << "Grid(rows, cols) - rows > " << int_max;
		throw std::out_of_range(os.str());
	}

	if(cols > int_max) {
		std::ostringstream os;
		os << "Grid::Grid() - cols > " << int_max;
		throw std::out_of_range(os.str());
	}

	if(arg_r <= 0) {
		std::ostringstream os;
		os << "Grid::Grid() - radius < " << arg_r;
		throw std::out_of_range(os.str());
	}

} // check_args()

} // namespace

namespace Vars
{
extern const uint32_t num_traits;
extern const float traits[][13];
extern const uint32_t min_same_trait_balls;
} // namespace Vars

Grid::Grid(Game::Impl& arg_game, size_type rows, size_type cols, float arg_r,
	   const std::string& filename)
 : Parent(rows, cols), Game_Elements_Accessor(arg_game), r(arg_r),
   w(num_cols()*r*2.0),
   h((std::sqrt(3.0)*r) * (num_rows()-1) + 2.0*r), ///< @see Grid.hh
   max_move_down_count(0), move_down_count(0),
   game_over_row(rows-1)
{
	check_args(rows, cols, arg_r);

	const uint32_t lodval = game.game_manager.get_lod();

	setup_spaces();

	// Construct ball traits
	for(uint32_t i = 0; i != Vars::num_traits; ++i) {
		ball_traits.push_back(Ball_Trait(lodval, get_r(),
						 Vars::traits[i]));
	}

	// do this now, so if it throws, then KABOOM
	load_from_file(filename);

	Video_Output_Manager& vo
		= game.game_manager.system.video_output_manager;
	vo.register_data(this);

	texname = vo.get_texture("images/gridtex.png");

	make_dlist(lodval);

	Sound_Output_Manager& so
		= game.game_manager.system.sound_output_manager;
	so.load_sound("sounds/ex12.voc", expl_snd);

} // Grid::Grid()

void
Grid::load_from_file(const std::string& arg)
{
	std::string root = game.game_manager.get_option("data_root");
	std::string file = root + "/levels/" + arg;
	std::ifstream fin(file.c_str());
#	undef CHK
#	define CHK(words) \
	if(!fin.good()) {\
		throw std::invalid_argument(std::string() + \
					    "Grid::load_from_file() - " \
					    "file " + file \
					    + " "words"\n"); }

	CHK("unaccessible");
	size_type rows, cols;
	uint32_t max_mv_cnt;
	fin >> rows;
	CHK("incomplete file");
	fin >> cols;
	CHK("incomplete file");
	fin >> max_mv_cnt;
	CHK("incomplete file");

	std::vector<uint32_t> tmpballs; // trait index per ball
	copy(std::istream_iterator<uint32_t>(fin),
	     std::istream_iterator<uint32_t>(),
	     std::back_inserter(tmpballs));

#	undef CHK
	fin.close();

	if(tmpballs.size() != size())
		throw std::invalid_argument(std::string() +
					    "Grid::load_from_file() - "
					    "file " + file
					    + " has wrong number of balls\n");

	// Now begin setting stuff up!!!!!!!!!!
	if(rows != num_rows()
	   || cols != num_cols())
		throw std::invalid_argument(
			std::string() +
			"Grid::load_from_file() - file " + file
			+ " has different params from grid!\n");

	// OK now clear everything out
	reset();
	max_move_down_count = max_mv_cnt;
	move_down_count = 0;
	game_over_row = num_rows() - 1;
	balls_count = 0;

	// Put balls into Grid mate
	std::vector<uint32_t>::iterator bit = tmpballs.begin();
	for(iterator it = begin(), itend = end(); it != itend; ++it, ++bit) {
		it->visited = 0;
		Ball* b = 0;
		if(*bit < ball_traits.size()) {
			b = new Ball(*bit, to_world(it->pos), V2D());
			++balls_count;
			get_trait(*bit).inc_users();
		}
		it->ball.reset(b);
	}
}

class No_Traits_Error { };

std::size_t
Grid::get_random_trait() const
{
	if(!ball_traits.size())
		throw std::out_of_range("Grid::get_random_trait() - "
					"no ball traits");

	if(balls_count == 0)
		throw No_Traits_Error();

	while(1) {
		// Get random number in open-closed range
		// [0, ball_traits.size)
		std::size_t rnd = static_cast<std::size_t>(
			static_cast<float>(std::rand())
			*ball_traits.size()/RAND_MAX);
		if(get_trait(rnd).get_users() != 0)
			return rnd;
	}
}

std::size_t
Grid::add_trait(const Ball_Trait& arg)
{
	ball_traits.push_back(arg);
	return ball_traits.size()-1;
}

void
Grid::draw_ball_commands(const Ball& b) const
{
	const V2D& v = b.get_pos();
	glDisable(GL_TEXTURE_2D);
	glTranslatef(v.x, v.y, 0.0f);
	get_trait(b.get_trait_index()).draw_commands();
}

namespace
{


} // namespace

namespace
{
const float sqrt3 = std::sqrt(3.0f);

} // namespace

void
Grid::draw_handler() const
{
	switch(game.game_manager.get_lod()) {
	case 3:
		glDisable(GL_TEXTURE_2D);
		break;
	default:
		glEnable(GL_TEXTURE_2D);
		break;
	}
	glPushMatrix();
	glCallList(dlist);
	glPopMatrix();

	glDisable(GL_TEXTURE_2D);
	for(const_iterator it = begin(), itend = end(); it != itend; ++it) {

		Ball* ball = it->ball.get();
		if(!ball)
			continue;
		glPushMatrix(); {
			draw_ball_commands(*ball);
		}; glPopMatrix();
	}
	// Draw a white line across where the game_over_row is situated.
	glTranslatef(-get_w()/2, -get_h()/2, 0);
	glLineWidth(5);
	glDisable(GL_LIGHTING);
	glColor3f(1, 1, 1);
	glBegin(GL_LINES);
	glVertex2f(0.0f, (1.0f-sqrt3*r)+sqrt3*r*(game_over_row+1));
	glVertex2f(r*2*num_cols(), (1.0f-sqrt3*r)+sqrt3*r*(game_over_row+1));
	glEnd();
	glEnable(GL_LIGHTING);
}

V2D
Grid::to_world(const V2D& arg) const
{
	return V2D(arg.x - get_w()/2, arg.y - get_h()/2);
}

V2D
Grid::to_grid(const V2D& arg) const
{
	return V2D(arg.x + get_w()/2, arg.y + get_h()/2);
}

void
Grid::reset_visited_flags()
{
	for(iterator it = begin(), itend = end(); it != itend; ++it) {
		it->visited = 0;
	}
}

void
Grid::reset_visited_flags() const
{
	for(const_iterator it = begin(), itend = end(); it != itend; ++it) {
		it->visited = 0;
	}
}

Grid::Condition
Grid::add_ball(const Ball& b)
{
	V2D local = to_grid(b.get_pos());
	Grid_Space& nearest = get_nearest(local);
	if(nearest.ball.get() != 0)
		throw std::invalid_argument("Grid::add_ball() - "
					    "a ball exists!");

	// Put into nearest grid space, make its coords the same but
	// in world space rather than in grid space.
	nearest.ball = std::auto_ptr<Ball>(new Ball(b.get_trait_index(),
						    to_world(nearest.pos),
						    b.get_vel()));

	get_trait(nearest.ball->get_trait_index()).inc_users();
	++balls_count;

	std::size_t old_count = balls_count;

	if(count_same_trait_balls(nearest) >= Vars::min_same_trait_balls) {
		drop_same_trait_balls(nearest);
	}
	reset_visited_flags();
	for(iterator it = begin(), itend = it+num_cols(); it != itend; ++it) {
		// Starting at each column head, flag all touchings
		// from them.
		if(it->ball.get())
			flag_touchings(*it);
	}

	// Drop each flagged ball.  Count the balls dropped.
	for(reverse_iterator rit = rbegin(), ritend = rend();
	    rit != ritend; ++rit) {
		if(rit->ball.get() && !rit->visited)
			drop_ball(*rit);
	}

	std::size_t dropped_balls = old_count - balls_count;

	// Update player score based on dropped balls.

	// @todo Use real 'player_score' object!
	std::cout << "Player dropped " << dropped_balls << " balls!\n";

	// If any balls dropped, play explosion sound
	if(dropped_balls) {
		if(expl_snd.is_playing())
			expl_snd.stop_sound();
		expl_snd.play_sound(0);
	}

	if(++move_down_count == max_move_down_count) {
		move_down_count = 0;
		if(!game.game_manager.get_option("NoMove").size())
			--game_over_row;
	}

	return get_condition();
}

Grid::Condition
Grid::get_condition() const
{
	if(balls_count == 0)
		return FINISHED_LEVEL;

	if(get_lowest_space().coord.first >= game_over_row)
		return GAME_OVER;

	return OK;
}

void
Grid::flag_touchings(Grid_Space& arg)
{
	arg.visited = true;
	for(std::size_t i = 0; i != 6; ++i) {
		if(Grid_Space* gs = arg.get_adj(i))
			if(!gs->visited && gs->ball.get())
				flag_touchings(*gs);
	}
}

void
Grid::drop_ball(Grid_Space& arg)
{
	if(arg.ball.get() == 0)
		throw std::invalid_argument("Grid::drop_ball() - "
					    "no ball!");

	std::size_t trait = arg.ball->get_trait_index();

	get_popped_balls_manager().add_ball(*arg.ball);
	arg.ball.reset();

	get_trait(trait).dec_users();
	if(balls_count == 0)
		throw Paranoia_Confirmed_Error("Grid::drop_ball()");
	--balls_count;
}

void
Grid::count_same_trait_balls_helper(const Grid_Space& arg,
				    uint32_t& count) const
{
	const std::size_t trait = arg.get_ball()->get_trait();
	++count;
	arg.visited = true;
	for(std::size_t i = 0; i !=6; ++i) {
		if(const Grid_Space* myadj = arg.get_adj(i)) {
			if(myadj->ball.get()
			   && myadj->ball.get()->get_trait() == trait
			   && !myadj->visited)
				count_same_trait_balls_helper(*myadj, count);
		}
	}
}

uint32_t
Grid::count_same_trait_balls(Grid_Space& arg) const
{
	uint32_t count = 0;
	reset_visited_flags();
	if(arg.ball.get()) {
		count_same_trait_balls_helper(arg, count);
	}
	return count;
}

void
Grid::drop_same_trait_balls(Grid_Space& arg)
{
	if(arg.ball.get() == 0)
		return;

	std::size_t trait = arg.ball->get_trait_index();

	drop_ball(arg);

	for(std::size_t i = 0; i != 6; ++i) {
		if(Grid_Space* myadj = arg.get_adj(i))
			if(myadj->ball.get()
			   && myadj->ball->get_trait() == trait)
				drop_same_trait_balls(*myadj);
	}
}

Grid_Space&
Grid::get_nearest(const V2D& pos)
{
	iterator it = begin();
	Grid_Space* current = &(*it);
	float curmag = (current->get_pos() - pos).magnitude();
	++it;
	for(iterator itend = end(); it != itend; ++it) {
		float nextmag = (it->get_pos() - pos).magnitude();
		if(nextmag < curmag) {
			current = &(*it);
			curmag = (current->get_pos() - pos).magnitude();
		}
	}
	return *current;
}

void
Grid::reset()
{
	for(iterator it = begin(), itend = end(); it != itend; ++it) {
		it->ball.reset();
	}
	balls_count = 0;
	for(std::vector<Ball_Trait>::iterator it = ball_traits.begin(),
		    itend = ball_traits.end(); it != itend; ++it) {
		it->set_users(0);
	}
}

Grid_Space&
Grid::get_lowest_space()
{
	if(balls_count == 0)
		throw std::invalid_argument("Grid::get_lowest_space() - "
					    "no balls found!");

	for(reverse_iterator it = this->rbegin(), itend = this->rend();
	    it != itend; ++it) {
		if(it->ball.get())
			return *it;
	}

	// We shouldn't really ever get here, but wha' the hey...
	throw std::invalid_argument("Grid::get_lowest_space() - "
				    "no balls found!");
}

const Grid_Space&
Grid::get_lowest_space() const
{
	if(balls_count == 0)
		throw std::invalid_argument("Grid::get_lowest_space() - "
					    "no balls found!");

	for(const_reverse_iterator it = this->rbegin(), itend = this->rend();
	    it != itend; ++it) {
		if(it->ball.get())
			return *it;
	}

	// We shouldn't really ever get here, but wha' the hey...
	throw std::invalid_argument("Grid::get_lowest_space() - "
				    "no balls found!");
}

void
Grid::setup_spaces()
{
	float tmpx, tmpy;
	for(size_type rc = 0; rc < num_rows(); ++rc) {
		for(size_type cc = 0; cc < elements_in_row(rc); ++cc) {
			tmpx = 0; tmpy = 0;

			/* Move it along an r's worth. */
			if(odd_row(rc))
				tmpx += r;

			/* The first x co-ordinate will be
			   this much along */
			tmpx += r;

			/* And this is depending on which
			   column. */
			tmpx += cc * 2.0*r;

			/* The y again starts at an r's worth. */
			tmpy += r;

			/* Then add on a multiple of sqrt(3)*r
			   depending on row. */
			tmpy += std::sqrt(3.0)*r*rc;

			Grid_Space& gs = at(rc, cc);
			gs.pos.x = tmpx;
			gs.pos.y = tmpy;
			gs.coord = std::make_pair(rc, cc);
		}
	}

	setup_touchings();

} // Grid::setup_spaces()

Grid_Space*
Grid::getgsptr(int32_t row, int32_t col)
{
	if(row < 0)
		return NULL;

	if(col < 0)
		return NULL;

	if(static_cast<uint32_t>(row) >= num_rows())
		return NULL;

	if(static_cast<uint32_t>(col) >= elements_in_row(row))
		return NULL;

	return &at(row, col);

} // Grid::getgsptr()

void
Grid::setup_touchings()
{
	int32_t rc, cc;

	// Process even rows
	for(rc = 0; (uint32_t)rc < num_rows(); rc += 2) {
		for(cc = 0; (uint32_t)cc < num_cols(); ++cc) {
			// Convenience
			Grid_Space& gs = at(rc, cc);

			gs.adj[Grid_Space::R] = getgsptr(rc, cc+1);
			gs.adj[Grid_Space::DR] = getgsptr(rc+1, cc);
			gs.adj[Grid_Space::DL] = getgsptr(rc+1, cc-1);
			gs.adj[Grid_Space::L] = getgsptr(rc, cc-1);
			gs.adj[Grid_Space::UL] = getgsptr(rc-1, cc-1);
			gs.adj[Grid_Space::UR] = getgsptr(rc-1, cc);
		}

	} // Process even rows

	// Process odd rows
	for(rc = 1; (uint32_t)rc < num_rows(); rc += 2) {
		for(cc = 0; (uint32_t)cc < num_cols()-1; ++cc) {
			// Convenience
			Grid_Space& gs = at(rc, cc);

			gs.adj[Grid_Space::R] = getgsptr(rc, cc+1);
			gs.adj[Grid_Space::DR] = getgsptr(rc+1, cc+1);
			gs.adj[Grid_Space::DL] = getgsptr(rc+1, cc);
			gs.adj[Grid_Space::L] = getgsptr(rc, cc-1);
			gs.adj[Grid_Space::UL] = getgsptr(rc-1, cc);
			gs.adj[Grid_Space::UR] = getgsptr(rc-1, cc+1);
		}

	} // Process odd rows

} // Grid::setup_touchings()



} // namespace XBobble

