/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Launched_Ball_Manager.cc @see Launched_Ball_Manager.hh */

#include "Launched_Ball_Manager.hh"
#include "Game_Manager.hh"
#include "Video_Data.hh"
#include "Ball.hh"
#include <SDL_opengl.h>
#include "Grid.hh"
#include <algorithm>
#include <utility>
#include <cstdlib>
#include <cmath>
#include <numeric>

namespace XBobble
{

Launched_Ball_Manager::Launched_Ball_Manager(Game::Impl& arg)
 : Game_Element(arg), ball(255), ball_stopped(true)
{
	Game_Manager& gm = game.game_manager;
	gm.system.video_output_manager.register_data(this);
}

void
Launched_Ball_Manager::add_ball(const Ball& arg)
{
	ball = arg;
	ball_stopped = false;
}

void
Launched_Ball_Manager::draw_handler() const
{
	if(ball_stopped) return;
	glDisable(GL_TEXTURE_2D);
	get_grid().draw_ball_commands(ball);
}

namespace
{

/// Return time of a collision between the given balls (ball1 is
/// travelling, ball2 is stationary - ie ignore its velocity vector) -
/// note that if a collision has already occured, then a negative
/// number is returned indicating the time passed since collision.
float
ball_coll(const Ball& ball1, const V2D& ball2pos, float r)
{
	using std::pow;
	using std::sqrt;
	float a = ball2pos.x - ball1.get_pos().x;
	float b = ball2pos.y - ball1.get_pos().y;
	const V2D& vel = ball1.get_vel();
	float xv = vel.x, yv = vel.y;
	float xvapyvb = xv*a + yv*b;
	float a2pb2m4rr = a*a + b*b - 4*r*r;
	float b2m4ac = xvapyvb*xvapyvb - 4.0f*(xv*xv+yv*yv)*(a2pb2m4rr);
	if(b2m4ac < 0)
		return std::numeric_limits<float>::max()-1;
	float res1 = (2*xvapyvb - sqrt(b2m4ac)) / (2*a2pb2m4rr);
		return res1;
}

} // namespace

void
Launched_Ball_Manager::sync_tick_handler(uint32_t delta)
{
	if(ball_stopped)
		return;

	// Get ready for recursion
	if(delta == 0)
		return;

	V2D local = get_grid().to_grid(ball.get_pos());

	float ct = (get_grid().get_r() - local.y)/ball.vel.y;
	if(ct < 0)
		throw std::out_of_range(
			"Launched_Ball_Manager::sync_tick_handler() - "
			" ct < 0!  Ball should never be going away "
			"from the ceiling!");
	float wt;

	if(ball.get_vel().x < 0) {
		// Ball going left
		float r = get_grid().get_r(), xp = local.x, xv = ball.vel.x;
		wt = (r - xp) / xv;
	} else if(ball.get_vel().x > 0) {
		// Ball going right
		float r = get_grid().get_r(), xp = local.x, xv = ball.vel.x;
		float a = get_grid().get_w();
		wt = (a - xp - r) / xv;
	} else {
		// Ball going straight up!
		wt = std::numeric_limits<float>::max()-1;
	}

	float stopt = ct, bt = std::numeric_limits<float>::max()-1;
	if(get_grid().get_balls_count()) {
		Grid::const_iterator it = get_grid().begin(),
			itend = get_grid().end();
		if(const Ball* gridball = it->get_ball()) {
			bt = ball_coll(ball, gridball->pos,
				       get_grid().get_r());
		}
		++it;
		for(; it != itend; ++it) {
			if(const Ball* gridball = it->get_ball()) {
				float tmp;
				tmp = ball_coll(ball, gridball->pos,
						get_grid().get_r());
				bt = std::min(bt, tmp);
			}
		}

// 		if(bt < 0)
// 			std::cerr << "bt < 0: bt == " << bt
// 				  << '\n'; std::cerr.flush();
		stopt = std::min(bt, ct);
	}

	// ie stopt is the earliest collision that happened.  So, if
	// it is negative, it is the collision that happened in the
	// past, which is where the ball will (should) end up.
	// Allahuakbar.
	if(stopt < wt) {
		// Ball going to stop without hitting a wall.
		if(stopt <= static_cast<float>(delta)) {
			ball.pos += ball.vel*stopt;
			ball_stopped = true;
			get_grid().add_ball(ball);
			return;
		} else {
			/// Just keep going!
			ball.pos += ball.vel*delta;
			return;
		}
	}

	// We're going to hit a wall... but maybe not yet.
	if(static_cast<uint32_t>(wt) >= delta) {
		ball.pos += ball.vel*delta;
		if(static_cast<uint32_t>(wt) == delta)
			ball.vel.x = -ball.vel.x;
		// Get outta here, all ticks accounted for!
		return;
	}

	// We'll be hitting a wall and rebounding off
	uint32_t wallhitdelta = delta - static_cast<uint32_t>(wt);
	ball.pos += ball.vel*wallhitdelta;
	ball.vel.x = -ball.vel.x;
	sync_tick_handler(wallhitdelta);
}

} // namespace XBobble


