/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file OliorGrid.hh An OliorGrid.  Olior means 'One less in odd
   row.'  This refers to the fact that this is a normal 2D grid, but
   has one less element in its odd rows! */

#ifndef XBOBBLE_OLIORGRID_HH
#define XBOBBLE_OLIORGRID_HH

#include <stdexcept>

#include <utility>
#include <vector>

namespace XBobble
{

//I: OliorGrid
/** Pronounced 'oleeor grid'.  Stands for 'One Less In Odd Row Grid'.

    Special grid holding elements of a template type. Odd rows have
    one less element (hereby known as the odd-row-rule.)  Arranged
    like this:

    * * * * * * * ....
    * * * * * *   .... (and so on)
    * *	* * * *	* ....
    * * * * * *   ....
       	  .
       	  . (and so on)
       	  .

    Width specifies the number of elements in an EVEN ROW (even
    includes 'zero'.)

    It is optimised for sequential access to elements (the base
    container is a std::vector) so direct access via 'at' will be
    slightly slower than normal std::vector direct access due to
    having to map the x/y offsets of the grid into the std::vector.

    NOTE: most of the functions defined below are invalid if called on
    an OliorGrid with size (0, 0).  Remember even if one row/col size
    is requested as zero, then both are set to zero.  Valid functions
    in this state are listed below:

    TODODODODODODODODODO

    */
template<typename T,
	 typename Allocator = typename std::vector<T>::allocator_type >
class OliorGrid
{
public:
	/* typedefs */

	typedef Allocator allocator_type;

	typedef T value_type;

	typedef value_type& reference;

	typedef const value_type& const_reference;

	typedef typename std::vector<value_type, allocator_type>::iterator iterator;

	typedef typename std::vector<value_type, allocator_type>::const_iterator const_iterator;

	typedef typename std::vector<value_type, allocator_type>::reverse_iterator reverse_iterator;

	typedef typename std::vector<value_type, allocator_type>::const_reverse_iterator const_reverse_iterator;

	typedef typename std::vector<value_type, allocator_type>::size_type size_type;

	typedef std::pair<size_type, size_type> key_type;

	typedef typename std::vector<value_type, allocator_type>::difference_type difference_type;

	/** Destructor - "...because it should be there." */
	virtual
	~OliorGrid()
	{
	}

	/* Constructors */

	/** Creates a grid with 0 rows and 0 columns */
	OliorGrid()
	 : size_val_(0, 0)
	{
	}

	/** Copy constructor */
	explicit OliorGrid(const OliorGrid& other)
	 : size_val_(other.size_val_), data_(other.data_)
	{
	}

// 	/// Really nice and easy assignment operator
// 	OliorGrid&
// 	operator=(const OliorGrid& other)
// 	{
// 		size_val_ = other.size_val_;
// 		data_ = other.data_;
// 	}

	/** Request a grid with rows/columns.  If any of the arguments
	    are zero, then they'll both be zero.  All elements are
	    initialised with 'value'. */
	OliorGrid(size_type arg_rows, size_type arg_cols,
		  const value_type& value = value_type())
	 : size_val_(key_type(arg_rows, arg_cols)),
	   data_(compute_data_size(arg_rows, arg_cols), value)
	{
		/** TODO Check size is OK. */

		if(num_rows()==0 | num_cols()==0)
			size_val_.first=size_val_.second=0;
	}

	/** Same as above, except take a 'key_type' as argument
	    instead of separate row/col sizes. */
	OliorGrid(key_type arg_size,
		  const value_type& value = value_type())
	 : size_val_(arg_size),
	   data_(compute_data_size(arg_size.first, arg_size.second), value)
	{
		/** TODO Check size is OK. */

		if(num_rows()==0 | num_cols()==0)
			size_val_.first=size_val_.second=0;
	}

	/* Size operations */

	/** Number of elements in grid. */
	size_type
	size() const
	{
		return data_.size();
	}

	/** Returns grid dimensions in key_type format. */
	key_type
	dimensions() const
	{
		return size_val_;
	}

	/** Number of rows in grid. */
	size_type
	num_rows() const
	{
		return size_val_.first;
	}

	/** Number of cols in an even-numbered row. */
	size_type
	num_cols() const
	{
		return size_val_.second;
	}

	/** Is this container empty? */
	bool
	empty() const
	{
		return data_.empty();
	}

	/** Maximum size */
	size_type
	max_size() const
	{
		return 0;
	}

	/* Capacity */

	/** @todo Should I make this baby relocatable? */

	/// Do it!
	void
	relocate() {}

	/* Comparison */

	///@define OLIOR_GRID_COMPARE_ME_ Generate comparison operator
	///'op'
#	define OLIOR_GRID_COMPARE_ME_(op) \
	bool \
	operator op (const OliorGrid& c2) \
	{ \
		return data_ op c2.data_; \
	}

	OLIOR_GRID_COMPARE_ME_(==)
	OLIOR_GRID_COMPARE_ME_(!=)
	OLIOR_GRID_COMPARE_ME_(<)
	OLIOR_GRID_COMPARE_ME_(>)
	OLIOR_GRID_COMPARE_ME_(<=)
	OLIOR_GRID_COMPARE_ME_(>=)

#	undef OLIOR_GRID_COMPARE_ME_

	/* Assignment */

	/** Assign from other to this. */
	OliorGrid&
	operator=(const OliorGrid& other)
	{
		size_val_ = other.size_val_;
		data_ = other.data_;
		return (*this);
	}

	/** Swaps all elements in one container with the other. */
	void
	swap(OliorGrid& other)
	{
		std::swap(size_val_, other.size_val_);
		data_.swap(other.data_);
	}

	/* Direct element access */

	/** Get element with checks. */
	reference
	at(const key_type& arg)
		throw(std::out_of_range)
	{
		check_valid_indices_(arg.first, arg.second);
		return get_data_elem_(arg.first, arg.second);
	}

	/** Does same thing as above but with different parameters. */
	reference
	at(size_type row, size_type col)
		throw(std::out_of_range)
	{
		check_valid_indices_(row, col);
		return get_data_elem_(row, col);
	}

	/** Get const element with checks. */
	const_reference
	at(const key_type& arg) const
		throw(std::out_of_range)
	{
		check_valid_indices_(arg.first, arg.second);
		return get_data_elem_(arg.first, arg.second);
	}

	/** Does same thing as above but with different parameters. */
	const_reference
	at(size_type row, size_type col) const
		throw(std::out_of_range)
	{
		check_valid_indices_(row, col);
		return get_data_elem_(row, col);
	}

	/* Iterator generators. */

	/** Points to the first element. */
	iterator
	begin()
	{
		return data_.begin();
	}

	/** Points to the first element. */
	const_iterator
	begin() const
	{
		return data_.begin();
	}

	/** Points to the last element + 1. */
	iterator
	end()
	{
		return data_.end();
	}

	/** Points to the last element + 1. */
	const_iterator
	end() const
	{
		return data_.end();
	}

	/** Points to the first element from the back. */
	reverse_iterator
	rbegin()
	{
		return data_.rbegin();
	}

	/** Points to the first element from the back. */
	const_reverse_iterator
	rbegin() const
	{
		return data_.rbegin();
	}

	/** Points to the last element + 1 from the back. */
	reverse_iterator
	rend()
	{
		return data_.rend();
	}

	/** Points to the last element + 1 from the back. */
	const_reverse_iterator
	rend() const
	{
		return data_.rend();
	}

	/* Utility functions. */

	/** Returns true if the given row is odd, or false if not. */
	bool
	odd_row(size_type row) const
	{
		return row & 0x1;
	}

	/** Computes the number of elements in the given row.  Takes
	    into account the odd-row-rule.  If size() is 0, result is
	    undefined. */
	size_type
	elements_in_row(size_type row) const
	{
		return (odd_row(row)) ? num_cols()-1 : num_cols();
	}

	/** Check if given indices (row/col) are valid. */
	bool
	valid_indices(size_type row,
		      size_type col) const
	{
		if(row > num_rows()-1)
			return false;
		if(col > num_cols()-1)
			return false;

		return true;
	}

protected:
	/* Internal functions */

	/** Check if given indices (row/col) are valid, and throw
	    std::out_of_range if not. */
	void
	check_valid_indices_(size_type row,
			     size_type col) const
		throw(std::out_of_range)
	{
		if(!valid_indices(row, col))
			throw std::out_of_range(
				"in OliorGrid::check_valid_indices_(size_type_single, size_type_single) const - indices out of range.");
	}

	/** Compute index into data_ using row/col indices given. */
	size_type
	compute_data_index_(size_type row, size_type col) const
	{
		return
			num_cols()*row /* get normal grid's row
					  start */
			- (row>>1) /* take away number of odd rows
				      passed (not including current
				      one) which is number of
				      'missing' elements. */
			+ col /* Add on column index into that row. */
			;
	}

	/** Given row and col sizes, returns size of data needed to exactly
	    capacitate them.  Yeah, so it should be static, but I can't be
	    bothered putting it in its own file and all that. */
	std::size_t
	compute_data_size(std::size_t arg_num_rows,
			  std::size_t arg_num_cols)
	{
		/* Number of odd rows are given by (num_rows())/2
		   {rounded down}. */
		return arg_num_rows*arg_num_cols - (arg_num_rows >> 1);
	}

	/** Returns the data element at given index. */
	reference
	get_data_elem_(size_type row, size_type col)
	{
		return data_[compute_data_index_(row, col)];
	}

	/** Returns the data element at given index. */
	const_reference
	get_data_elem_(size_type row, size_type col) const
	{
		return data_[compute_data_index_(row, col)];
	}

protected:

	/** Size of grid */
	key_type size_val_;

	/** Where grid data is stored. */
	std::vector<value_type, allocator_type> data_;

}; // OliorGrid


} // namespace XBobble

#endif /* #define XBOBBLE_OLIORGRID_HH */



