/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Popped_Balls_Manager.cc @see Popped_Balls_Manager.hh */

#include <list>
#include <cstdlib>
#include <iostream>
#include <iomanip>
#include "Popped_Balls_Manager.hh"
#include "Game_Manager.hh"
#include "Grid.hh"
#include "Ball.hh"
#include "Ball_Trait.hh"
#include <SDL_opengl.h>

namespace XBobble
{

namespace Vars
{

extern const float gravity;
extern const float popped_vv_x, popped_vv_y;

} // namespace Vars

class Popped_Balls_Manager::Impl
{
	friend class Popped_Balls_Manager;

	std::list<Ball> balls; // I've got plenty

}; // Popped_Balls_Manager::Impl

const V2D Popped_Balls_Manager::gravity_vector(0.0f, Vars::gravity);

Popped_Balls_Manager::Popped_Balls_Manager(Game::Impl& arg)
 : Game_Element(arg), impl(new Impl())
{
    try
    {
	Game_Manager& gm = game.game_manager;
	gm.system.video_output_manager.register_data(this);
    }
    catch(...)
    {
	    delete impl;
	    throw;
    }
}

void
Popped_Balls_Manager::add_ball(const Ball& b)
{
	impl->balls.push_back(b);
	Ball& tmp = impl->balls.back();
	tmp.vel = V2D(Vars::popped_vv_x, Vars::popped_vv_y);
	tmp.vel.x *= 2.0f*static_cast<float>(std::rand())/RAND_MAX - 1.0f;
	tmp.vel.y *= static_cast<float>(std::rand())/RAND_MAX;
}

void
Popped_Balls_Manager::reset()
{
	impl->balls.clear();
}

void
Popped_Balls_Manager::draw_handler() const
{
	glDisable(GL_TEXTURE_2D);
	glTranslatef(0.0f, 0.0f, -get_grid().get_r());
	typedef std::list<Ball>::const_iterator It;
	// Draw each stored ball
	for(It it = impl->balls.begin(), itend = impl->balls.end();
	    it != itend; ++it) {
		glPushMatrix();
		get_grid().draw_ball_commands(*it);
		glPopMatrix();
	}
}

void
Popped_Balls_Manager::tock_handler(uint32_t tocks)
{
}

void
Popped_Balls_Manager::tick_handler(uint32_t delta)
{
	typedef std::list<Ball>::iterator It;

	for(It it = impl->balls.begin(), itend = impl->balls.end();
	    it != itend; ++it) {
		if(it->pos.y > 200) {
			it = impl->balls.erase(it);
			if(it == itend)
				break;
		}
		// s = u t + half a t squared
		it->pos += it->vel*delta + gravity_vector*0.5*delta*delta;
		// v = u + at
		it->vel += gravity_vector*delta;
	}
}


} // namespace XBobble


