/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Sound_Data.hh a sound_data file for easy creation of new files */

#ifndef XBOBBLE_SOUND_DATA_HH
#define XBOBBLE_SOUND_DATA_HH

#include <ayq/stdint.h>
#include <SDL_mixer.h>

namespace XBobble
{

class Sound_Output_Manager;

/// Simple classes whose objects allow a user to play/stop/pause a
/// sound.  Obtain an object of this class with
/// Sound_Output_Manager::load_sound().  Note that these objects
/// should NOT be used until System has been initialised.
class Sound_Data
{
public:
	/// Exception that is raised if any of the "*_sound()"
	/// functions below fail for whatever reason.
	class Sound_Op_Failed_Error : public std::logic_error
	{
	public:
		Sound_Op_Failed_Error(const std::string& arg)
		 : std::logic_error(arg)
		{
		}

	}; // class Sound_Op_Failed_Error

	/** Creates blank sound.  Use
	    'sound_output_manager.load_sound()' to put sound in it!
	*/
	Sound_Data()
	 : channel(-1), sample(0)
	{
	}

	/// Release sound data if it is valid (not 0)
	~Sound_Data()
	{
		if(sample != 0) {
			Mix_FreeChunk(sample);
			sample = 0;
			channel = -1;
		}
	}

	/// Play this sound - if you try to play a sound that's
	/// already playing, it will be ignored.  pass number of times
	/// to loop, or -1 for infinite looping.
	void
	play_sound(int loop)
	{
		if(is_playing())
			return;
		if(Mix_PlayChannel(channel, sample, loop) == -1)
			throw Sound_Op_Failed_Error(
				"Sound_Data::play_sound()");
	}

	/// Gets playing status of sound (returns true also if paused)
	bool
	is_playing()
	{
		return Mix_Playing(channel);
	}

	/// Is the channel paused?
	bool
	is_paused()
	{
		return Mix_Playing(channel) && Mix_Paused(channel);
	}

	/// Stop the sound
	void
	stop_sound()
	{
		Mix_HaltChannel(channel);
	}

	/// Pause the sound
	void
	pause_sound()
	{
		if(is_playing())
			Mix_Pause(channel);
	}

	/// Unpause the sound
	void
	unpause_sound()
	{
		if(is_paused())
			Mix_Resume(channel);
	}

private:
	friend class Sound_Output_Manager;

	/// SDL_mixer channel playing on
	int channel;

	/// SDL_mixer data to use to hold sound data
	Mix_Chunk* sample;

	//@{

	/// Not allowed - use 'sound_output_manager.load_sound()' to
	/// reload file
	Sound_Data(const Sound_Data&)
	{
	}

	void
	operator=(const Sound_Data&)
	{
	}

	//@}

}; // class Sound_Data


} // namespace XBobble


#endif // #define XBOBBLE_SOUND_DATA_HH
