/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file User_Input_Manager.cc @see User_Input_Manager.hh */

#include <ayq/stdint.h>
#include "User_Input_Manager.hh"
#include "System.hh"
#include "Game_Manager.hh"
#include <iostream>
#include <numeric>

namespace XBobble
{

// Better system would pass some state flags to end user telling them
// whether the state of the input object they're reading has been
// changed by the user since the last time it was called.  Simply
// relieves the user of this burden.


User_Input_Manager::User_Input_Manager(System& arg_sys)
 : Singleton_Class("User_Input_Manager"), system(arg_sys),
   my_joystick(0), my_axis(0), axisval(0),
   spacebar(2), escape(2), left(2), right(2), kbdaxisval(0)
{
	// Init joystick stuff

	std::string jsstr = system.game_manager.get_option("joyindex");
	std::string jsaxis = system.game_manager.get_option("joyaxis");

	int32_t dz
		= atoi(system.game_manager.get_option("joydeadzone").c_str());
	if(dz < 0) deadzone = 0;
	else if(dz > 80) deadzone = 80;
	else deadzone = dz;

	jmax = static_cast<int32_t>(32768.0 * 0.8);
	jmin = static_cast<int32_t>(32768.0 * deadzone / 100.0);

	if(SDL_NumJoysticks() == 0 || jsstr == "" || jsaxis == "")
		return;

	my_joystick = atoi(jsstr.c_str());
	my_axis = atoi(jsaxis.c_str());

	SDL_JoystickEventState(SDL_ENABLE);

	SDL_Joystick* js = 0;
	if((js = SDL_JoystickOpen(my_joystick)) == 0) {
		SDL_JoystickEventState(SDL_IGNORE);
		std::cerr << "Error initialising joystick "
			  << my_joystick << "\n";
	}

	if(my_axis > SDL_JoystickNumAxes(js)-1) {
		std::cerr << "Specified axis " << my_axis << " non-existant!"
			  << "  Defaulting to axis '0'\n";
		my_axis = 0;
	}

	SDL_Event ev;
	// Flush the event queue of any lingering joystick events
	// which will simply cause trouble if processed normally (if
	// button is pressed down already, just take out 20)
	for(uint32_t tmp = 0; tmp < 20; ++tmp) {
		if(!SDL_PollEvent(&ev)) break;
		if(ev.type != SDL_JOYBUTTONUP) break;
	}
}

namespace
{
} // namespace

void
User_Input_Manager::process_events(SDL_Event& ev)
{
// 	// If they are 'pressed', make them simply 'pushed'
// 	spacebar = (spacebar == 3) ? 2 : spacebar;
// 	escape = (escape == 3) ? 2 : escape;

// 	// Reset this so that we can set it again if necessary
// 	button = false;

//  	// Reset kbdaxisval to '0' so it can be set again
// 	kbdaxisval = 0;

	switch(ev.type) {
	case SDL_KEYDOWN:
		switch(ev.key.keysym.sym) {
		case SDLK_SPACE: spacebar = 1; break;
		case SDLK_ESCAPE: escape = 1; break;
		case SDLK_LEFT:
			left = 1;
			kbdaxisval = (right == 1) ? 0 : -1;
			break;
		case SDLK_RIGHT:
			right = 1;
			kbdaxisval = (left == 1) ? 0 : 1;
			break;
		default: break;
		}
		break;

	case SDL_KEYUP:
		switch(ev.key.keysym.sym) {
// 		case SDLK_SPACE: spacebar = 2; break;
		case SDLK_ESCAPE: escape =  2; break;
		case SDLK_LEFT:
			left = 2;
			kbdaxisval = (right == 2) ? 0 : 1;
			break;
		case SDLK_RIGHT:
			right = 2;
			kbdaxisval = (left == 2) ? 0 : -1;
			break;
		default: break;
		}

		break;

	case SDL_JOYAXISMOTION: {
		const Sint16& val = ev.jaxis.value;
		if(ev.jaxis.which != my_joystick ||
		   ev.jaxis.axis != my_axis) break;
		// A dead-zone of X% is imposed in the centre of the
		// axis, where X is obtained from the option
		// 'joydeadzone'.
		//
		// At the outer-rim, the maximum value of the axis is
		// considered to end 10% before the actual limit of
		// the joystick (ie the outer 10% is ignored)

		if((val > -jmin) && (val < jmin))
			axisval = 0;
		else {
			if(val < -jmax)
				axisval = -100;
			else if(val > jmax)
				axisval = 100;
			else {
				axisval = val * 100 / jmax;
			}
		} // else

		break; } // case SDL_JOYAXISMOTION:

	case SDL_JOYBUTTONDOWN:
		button = true;
		break;

	case SDL_JOYBUTTONUP:
		// button = false;
		break;

	default:
		break;

	} // switch(ev.type)

} // User_Input_Manager::process_events()

void
User_Input_Manager::reset_kbd_keys()
{
	// Resetting all to 'up'
	spacebar = escape = left = right = 2;
	// kbdaxisval = 0;
}

void
User_Input_Manager::reset_joy()
{
	button = false;
}

int32_t
User_Input_Manager::get_axis() const
{
	return axisval;
}

bool
User_Input_Manager::get_spacebar() const
{
	bool tmp = (spacebar == 1);
	spacebar = 2;
	return tmp;
}

bool
User_Input_Manager::get_escape() const
{
	return (escape == 1);
}

int32_t
User_Input_Manager::get_kbdaxis() const
{
	return kbdaxisval;
}

bool
User_Input_Manager::get_button() const
{
	bool tmp = button;
	button = false;
	return tmp;
}


} // namespace XBobble


