#!/usr/bin/python
import os, sys, time, string
import pygame, optparse, numarray

# Default variables

DEFAULT_FONT = "courier.ttf"
DEFAULT_WIDTH = 40
DEFAULT_HEIGHT = 12

######################
def output(fd, text):
	fd.write(text)
	fd.flush()

######################
def debug(text):
	output(sys.stderr, text)

###################################
#####################
class image2text:
	#####################################
	def __init__(self, font_file):
		self.font_file = font_file
		self.ascii_range = (32,255)
		self.window_size = (320, 200)
		self.printable = "".join(map(chr, range(*self.ascii_range)))
	
	######################
	def init_pygame(self):
		debug("Initializing pygame... ")
		pygame.init()
		debug("done\n")

	######################
	def create_font(self, font_size):
		debug("Loading font from %s... " %self.font_file)
		font = pygame.font.Font(self.font_file, font_size)
		debug("done\n")

		# Draw text list in white color and get array (column, row)
		debug("Rendering fonts surface... ")
		surface = font.render(self.printable, False, (255,255,255))
		debug("done\n")
		return surface

	######################
	def load_image(self, image_file):
		# Load image to convert, get size and array
		debug("Loading image from file (%s)... " %image_file)
		image = pygame.image.load(image_file)
		debug("done\n")
		return image

	######################
	def get_grayarray(self, image, threshold):
		image_array = pygame.surfarray.array3d(image)
		image_array2 = pygame.surfarray.array2d(image)

		# Convert 2d array image into a grey scale array
		debug("Converting image to grey scale... ")
		
		for column in range(len(image_array)):
			for row in range(len(image_array[column])):
				r, g, b =  numarray.array(image_array[column][row])
				image_array2[column][row] = (r + g + b) / 3.0 < threshold
		debug("done\n")
		return image_array2

	######################
	def adjust_surfaces(self, fonts, image):		
		image_width, image_height = image.get_size()
		ax = int(image_width / self.text_width)
		ay = int(image_height / self.text_height)

		# Scale fonts surface to match required output 
		fonts_width, fonts_height = ax * len(self.printable), ay
		debug("Scaling fonts... ")
		fonts = pygame.transform.scale(fonts, (fonts_width, fonts_height))
		debug("done\n")

		# Scale input image to snap to characters size
		image_width, image_height = ax * self.text_width, ay * self.text_height
		debug("Scaling image to %d x %d... " %(image_width, image_height))
		image = pygame.transform.scale(image, (image_width, image_height))
		debug("done\n")
		return fonts, image

	################################
	def analysis(self, fonts, image, threshold):
		# Calculate some variables, ax: character width, ay: character heigth
		image_width, image_height = image.get_size()
		ax = int(image_width / self.text_width)
		ay = int(image_height / self.text_height)
		corrx, corry = 320.0 / image_width, 200.0 / image_height
		fonts_array = pygame.surfarray.array2d(fonts)
		image_array = self.get_grayarray(image, threshold)
		image2 = pygame.transform.scale(image, self.window_size)
		self.screen = pygame.display.set_mode(self.window_size, 0)
		debug("output: %dx%d chars\n" %(self.text_width, self.text_height))
		
		# Break image into (ax, ay) pieces and calc error with each character, take the closest one
		debug("Starting analysis...\n")
		nchars = 0
		for row in range(0, image_height, ay):
			for column in range(0, image_width, ax):
				best_char = -1
				self.screen.blit(image2, (0, 0))
				pygame.draw.rect(self.screen, (255, 0, 0), (corrx*column, corry*row, corrx*ax, corry*ay), 1)
				pygame.display.flip()

				for index, char in enumerate(self.printable):
					error = 0
					for y in range(ay):
						for x in range(ax):
							fonts_value = fonts_array[ax * index + x][y]
							image_value = image_array[column+x][row+y]
							if fonts_value != image_value:
								error += 1
					if best_char < 0 or error < best_error:
						best_char, best_error = char, error
						if error == 0:
							break

				yield best_char
				nchars += 1

			yield "\n"
		debug("done\n")

	#####################################
	def generator(self, image_file, text_width, text_height, threshold):
		# Get options
		self.text_width = text_width
		self.text_height = text_height

		self.init_pygame()
		fonts = self.create_font(32)
		image = self.load_image(image_file)
		fonts, image = self.adjust_surfaces(fonts, image)
		
		return self.analysis(fonts, image, threshold)
	
###################################
###################################
def main():
	usage = """image2text [options] image
	
	Convert a bitmap image (JPG, PNG, BMP, ...) to a ASCII text"""
	
	parser = optparse.OptionParser(usage)
	parser.add_option('-w', '--width', dest='width', default = DEFAULT_WIDTH, metavar='CHARS', type='int', help = 'Output text width')
	parser.add_option('-e', '--height', dest='height', default = DEFAULT_HEIGHT, metavar='LINES', type='int', help = 'Output text height')
	parser.add_option('-f', '--font', dest='font_file', default = DEFAULT_FONT, metavar='FILE', type='string', help = 'TTF font')
	parser.add_option('-t', '--threshold', dest='threshold', default = 128, metavar='VALUE', type='int', help = 'Image threshold (0..255)')
	options, args = parser.parse_args()
	if len(args) < 1: parser.error("Give a image to convert")
	image_file = args[0]
	try: import psyco; psyco.full()
	except: output(sys.stderr, "warning: psyco not available\n")
	i2t = image2text(options.font_file) 
	generator = i2t.generator(image_file, options.width, options.height, options.threshold)
	for char in generator:
		output(sys.stdout, char)
	sys.exit(0)
	
#####################
### MAIN
#####################

if __name__ == "__main__":
	main()